<?php
App::uses('EvShopAppModel', 'EvShop.Model');
/**
 * Brand Model
 *
 * @property Product $Product
 */
class Brand extends EvShopAppModel {

	public $actsAs = array(
		'Routable.Routable' => array(
			'config' => 'EvShop',
			'alias' => 'brand/:displayField/*'
		),
		'MetaData.Meta',
		'EvTemplates.Template' => array(
			'formInject' => true,
			'restrictTo' => 'Brand'
		)
	);

	/**
	 * Display field
	 *
	 * @var string
	 */
	public $displayField = 'name';

	/**
	 * Validation rules
	 *
	 * @var array
	 */
	public $validate = array(
		'name' => array(
			'notEmpty' => array(
				'rule' => array('notBlank'),
				'message' => 'A brand name must be entered.'
			),
		),
	);

	//The Associations below have been created with all possible keys, those that are not needed can be removed

/**
 * belongsTo associations
 *
 * @var array
 */
	public $belongsTo = array(
		'ParentBrand' => array(
			'className' => 'EvShop.Brand',
			'foreignKey' => 'parent_id'
		)
	);

/**
 * hasMany associations
 *
 * @var array
 */
	public $hasMany = array(
		'Product' => array(
			'className' => 'EvShop.Product'
		),
		'ChildBrand' => array(
			'className' => 'EvShop.Brand',
			'foreignKey' => 'parent_id'
		)
	);

	public $imageSlots = array(
		'logo' => array(
			'slots' => 1,
			'fields' => false
		)
	);

/**
 * Returns list of brands (grouped by parent when set in the plugin config)
 * ready to show in the products admin select element
 *
 * @param array Contains query params
 * @return array Contains list of brands
 */
	public function getForProductAdminDropDown($query = array()) {
		$query['conditions'][$this->alias . '.is_active'] = 1;

		$query['conditions'][$this->alias . '.parent_id'] = '';

		$query['order'] = $this->alias . '.name ASC';

		$query['callbacks'] = false;

		$list = $this->find('list', $query);

		// when brands have children, fetch and assign them to the array
		if (! empty($list) && Configure::read('EvShop.brandsHaveParents') == true) {
			$childBrands = $this->find('all', [
				'callbacks' => false,
				'conditions' => [
					$this->alias . '.is_active' => true,
					$this->alias . '.parent_id IS NOT NULL'
				],
				'fields' => [
					$this->alias . '.id',
					$this->alias . '.name',
					$this->alias . '.parent_id'
				],
				'order' => [
					$this->alias . '.name' => 'ASC'
				]
			]);

			if (! empty($childBrands)) {
				$childBrands = Hash::combine(
					$childBrands,
					'{n}.Brand.id',
					'{n}.Brand.name',
					'{n}.Brand.parent_id'
				);

				// bring the parent and child brands together into one array
				foreach ($list as $id => $name) {
					if (isset($childBrands[$id])) {
						foreach ($childBrands[$id] as $childId => $childName) {
							$childBrands[$id][$childId] = '- ' . $childName;
						}

						// manually split the array and add our children
						// this is done manually to preserve array keys
						$pos = array_search($id, array_keys($list));
						$pos = intval($pos) + 1;

						$list1 = array_slice($list, 0, $pos, true);
						$list2 = $childBrands[$id];
						$list3 = array_slice($list, $pos, null, true);

						// add the array back together to preserve keys
						$list = $list1 + $list2 + $list3;
					}
				}
			}
		}

		return $list;
	}

/**
 * Adds child-brand names into the list of brand names
 * @param  array  $brands
 * @return array 
 */
	public function appendChildren($brands = []) {
		$allBrands = [];
		foreach ($brands as $brand) {
			$allBrands[] = $brand;
			$fullBrand = $this->findByName($brand);

			// Get the child brands for top-level brands
			// so that we can include products belonging to sub-brands when filtering by a parent brand
			if (empty($fullBrand['Brand']['parent_id'])) {
				$allBrands = array_merge(
					$allBrands,
					$this->findChildren($fullBrand['Brand']['id'])
				);
			}
		}

		return array_unique($allBrands);
	}

/**
 * Fetches a list of child brand names associated with a top-level brand
 *
 * @param int $id brand ID
 * @return array
 */
	public function findChildren($id) {
		return $this->find('list', [
			'conditions' => [
				'is_active' => true,
				'parent_id' => $id
			],
			'fields' => ['id', 'name'],
		]);
	}

}
