<?php

App::uses('CopyableBehavior', 'EvCore.Model/Behavior');

class ProductCopyableBehavior extends CopyableBehavior {

/**
 * Configuration method.
 *
 * @param obj   $Model  Model object.
 * @param array $config Config array.
 * @access public
 * @return bool
 */
	public function setup(Model $Model, $config = array()) {
		$this->settings[$Model->alias] = array_merge($this->_defaults, $config);

		return true;
	}

/**
 * Copy a record. Associations can be copied with a recurseLevel > 0.
 *
 * @param obj $Model        Model object.
 * @param int $id           Integer model ID.
 * @param int $recurseLevel The amount of levels to include hasMany and hasOne records.
 * @access public
 * @return bool|array False if the copy failed. The new record if copy successful.
 */
	public function copy($Model, $id, $recurseLevel = 1) {
		//We will use the setting value so it can be overridden on each project.
		$copied = parent::copy($Model, $id, $this->settings[$Model->alias]['recurse_level']);

		//The product doesn't save the variants so we need to save them now
		if ($copied && !empty($this->_variants)) {
			$Variant = $Model->Variant;
			if (empty($Variant)) {
				$Variant = EvClassRegistry::init('EvShop.Variant');
			}

			$copiedProductId = $Model->id;

			$this->_variants = Hash::insert($this->_variants, '{n}.product_id', $copiedProductId);
			$this->_variants = $this->stripAllIds($Variant, $this->_variants);

			//Find any generated variants to delete once we save the new ones.
			$generatedVariants = $Variant->field(
				'id',
				[
					'product_id' => $copiedProductId,
				]
			);

			$copied = $copied && $Variant->saveMany($this->_variants, ['deep' => true]);

			if ($copied && !empty($generatedVariants)) {
				$copied = $Variant->deleteAll(
					[
						$Variant->alias . '.id' => $generatedVariants,
					]
				);
			}
		}

		return $copied;
	}

/**
 * Wrapper method that combines the results of _recursiveChildContain() with the models' HABTM associations.
 *
 * @param obj $Model        Model object.
 * @param int $recurseLevel The amount of levels to include hasMany and hasOne records.
 * @access public
 * @return array An array of contains to copy along with the main record.
 */
	public function generateContain($Model, $recurseLevel) {
		parent::generateContain($Model, $recurseLevel);

		if (isset($this->contain['Variant'])) {
			$this->contain['Variant'][] = 'Option';
		}

		//If the inventory doesn't want to be copied then it needs to be removed from the contain.
		if (isset($this->_copyData['copyInventory'])) {
			if (!$this->_copyData['copyInventory']) {
				if (!empty($this->contain['Variant'])) {
					//Find if the inventory has been contained and if so remove it
					$inventoryIndex = array_search('Inventory', $this->contain['Variant']);
					if ($inventoryIndex !== false) {
						unset($this->contain['Variant'][$inventoryIndex]);
					}
				}
			}

			unset($this->_copyData['copyInventory']);
		}

		//If the pricing doesn't want to be copied then it needs to be removed from the contain.
		if (isset($this->_copyData['copyPricing'])) {
			if (!$this->_copyData['copyPricing']) {
				if (!empty($this->contain['Variant'])) {
					//Find if the pricing has been contained and if so remove it
					$pricingIndex = array_search('VariantPricing', $this->contain['Variant']);
					if ($pricingIndex !== false) {
						unset($this->contain['Variant'][$pricingIndex]);
					}
				}
			}

			unset($this->_copyData['copyPricing']);
		}

		if (isset($this->_copyData['copyRelatedItems'])) {
			if (!$this->_copyData['copyRelatedItems']) {
				//Find if the related items have been contained and if so remove it.
				$relatedItemIndex = array_search('VariantPricing', $this->contain);
				if ($relatedItemIndex !== false) {
					unset($this->contain[$relatedItemIndex]);
				}
			}

			unset($this->_copyData['copyRelatedItems']);
		}

		return $this->contain;
	}

/**
 * Setup a record to be copied. The variants and their options are extacted here before the data is converted
 * and before the product is saved. This is required to save the variants after the products is saved to get
 * around any issues with new variants being automatically created/removed.
 *
 * @param obj $Model  The model object being copied.
 * @param int $id     The model id of the record being copied.
 * @param array  $params Custom parameters to use when finding the record.
 * @return array The found record. Also stored in $this->record.
 */
	protected function _setupRecord($Model, $id, $params = []) {
		parent::_setupRecord($Model, $id, $params);

		//We need to store the current options from the variants to save later
		if (!empty($this->record['Variant'])) {
			$this->_variants = $this->record['Variant'];
			$this->_variantOptions = Hash::combine($this->record, 'Variant.{n}.Option.{n}.id', 'Variant.{n}.Option.{n}.id', 'Variant.{n}.Option.{n}.option_group_id');
		}

		return $this->record;
	}

/**
 * Check the copyData to see if we need to copy images/documents. Also remove the variant image
 * slots from the contains used in _setupRecord() query.
 *
 * @param obj $Model   The model object being copied.
 * @param int $modelId The model id of the record being copied.
 * @return void.
 */
	protected function _checkAttachmentCopy($Model, $modelId) {
		if (isset($this->_copyData['copyImages'])) {
			$this->settings[$Model->alias]['copyImages'] = $this->_copyData['copyImages'];

			$this->_removeVariantImageContains($Model);

			unset($this->_copyData['copyImages']);
		}

		if (isset($this->_copyData['copyDocuments'])) {
			$this->settings[$Model->alias]['copyDocuments'] = $this->_copyData['copyDocuments'];

			unset($this->_copyData['copyDocuments']);
		}

		parent::_checkAttachmentCopy($Model, $modelId);
	}

/**
 * Remove variant images from the contains used in the _setupRecord() query.
 * The images will be copied after the record has been copied so the files
 * can be duplicated.
 *
 * @param obj $Model The model object being copied.
 * @return void.
 */
	protected function _removeVariantImageContains($Model) {
		$this->_removeAttachmentTypeContains($Model, 'variantImage');
	}

/**
 * Strips primary and parent foreign keys (where applicable) from $this->record in preparation for saving.
 *
 * @param object $Model Model object.
 * @access protected
 * @return array $this->record.
 */
	protected function _convertData($Model) {
		parent::_convertData($Model);

		if (!empty($this->record['Variant'])) {
			$this->settings[$Model->Variant->alias] = $this->settings[$Model->alias];

			foreach ($this->record['Variant'] as &$variant) {
				if (!empty($variant['Option'])) {
					$variant = $this->_convertHabtm($Model->Variant, $variant);
				}
			}
		}

		return $this->record;
	}

/**
 * Modify the record that is being copied. Any modification of the record data that happens here is
 * applied to the new copy.
 *
 * @param obj $Model The model oject being copied.
 * @return void.
 */
	protected function _modifyRecord($Model) {
		parent::_modifyRecord($Model);

		if (!empty($this->record['Variant']) && count($this->record['Variant']) == 1) {
			//We need to check if the only variant is a product variant to update the name of.
			$variant = reset($this->record['Variant']);
			$variantIndex = key($this->record['Variant']);
			if (empty($variant['Option'])) {
				//The variant is a product variant
				$this->record['Variant'][$variantIndex]['name'] = $this->record[$Model->alias]['name'];
				$this->_variants[$variantIndex]['name'] = $this->record[$Model->alias]['name'];
			}
		}

		//The copied product shouldn't start active so set it to inactive now.
		$this->record[$Model->alias]['is_active'] = false;

		//We need to set the variant options for the product so that no variants are created.
		if (!empty($this->_variantOptions)) {
			$this->record['Options'] = $this->_variantOptions;
			$this->record['CurrentOptions'] = $this->_variantOptions;
		}
	}
}
