<?php

if (CakePlugin::loaded('EvInventory')) {
	App::uses('InventoryLib', 'EvInventory.Lib');
}

App::uses('RouterUtil', 'Routable.Lib');

class GoogleFeedLib {

	protected $_productsPerIteration = 100;

	protected $_feedIteration = 0;

	protected $_cachedValues = [];

/**
 * Generate the google shopping feed for all the products.
 *
 * @return xml|bool The generate feed. False if the feed fails to generate.
 */
	public function generate() {
		ini_set('memory_limit', '5000M');
		set_time_limit(0);

		$this->_loadModels();

		// cache the repeated feed values
		$this->_cacheFeedValues();

		$products = $this->_getFeedProducts();

		if (empty($products)) {
			return false;
		}

		$xml = new XMLWriter;
		$xml->openMemory();
		$xml->startDocument('1.0', 'UTF-8');

		$xml->startElement('rss');
			$xml->writeAttribute('version', '2.0');
			$xml->writeAttribute('xmlns:g', 'http://base.google.com/ns/1.0');

			$xml->startElement('channel');
				$xml->writeElement('title', $this->_cachedValues['siteTitle']);
				$xml->writeElement('link', urlencode($this->_cachedValues['siteUrl']));
				$xml->writeElement('description', $this->_cachedValues['siteTitle'] . ' Product Feed');

				while (!empty($products)) {

					foreach ($products as $product) {
						$this->_addProductToFeed($xml, $product);
					}

					$this->_feedIteration++;
					$products = $this->_getFeedProducts();
				}

			$xml->endElement();
		$xml->endElement();
		$xml->endDocument();

		return $xml->outputMemory();
	}

/**
 * Load required models that will be used for the google feed.
 *
 * @return void.
 */
	protected function _loadModels() {
		$this->Product = EvClassRegistry::init('EvShop.Product');
		$this->Variant = EvClassRegistry::init('EvShop.Variant');
		$this->VariantPricing = EvClassRegistry::init('EvShop.VariantPricing');

		if (CakePlugin::loaded('EvInventory')) {
			$this->Variant->linkInventory();
		}

		if (CakePlugin::loaded('EvCurrency')) {
			$this->Currency = EvClassRegistry::init('EvCurrency.Currency');
		}
	}

/**
 * Get products for the feed. Products are found in batches based on the current feed iteration.
 *
 * @return array Products to add to the feed.
 */
	protected function _getFeedProducts() {
		$params = [
			'conditions' => [
				'Product.is_active' => true,
			],
			'contain' => [
				'ListingImage',
				'Variant' => [
					'order' => 'Variant.sequence ASC',
					'VariantPricing',
				],
				'Brand' => [
					'ParentBrand',
				],
				'CategoriesProduct' => [
					'limit' => 1,
					'Category',
				],
			],
			'group' => 'Product.id',
			'fields' => [
				'*',
			],
		];

		if (CakePlugin::loaded('EvInventory')) {
			$params['contain']['Variant'][] = 'Inventory';
		}

		$params['limit'] = $this->_productsPerIteration;
		$params['offset'] = $this->_feedIteration * $this->_productsPerIteration;

		$this->_modifyProductQueryParams($params);

		return $this->Product->find('all', $params);
	}

/**
 * Override this to modify the product query.
 *
 * @param array &$params The current query params
 * @return void.
 */
	protected function _modifyProductQueryParams(&$params) {
		// Method stub
	}

/**
 * Cache values that are used in each iteration of the feed.
 *
 * @return void.
 */
	protected function _cacheFeedValues() {
		$this->_cachedValues['siteTitle'] = Configure::read('SiteSetting.general.site_title');

		$this->_cachedValues['siteUrl'] = Configure::read('App.fullBaseUrl');

		$this->_cachedValues['fallbacks']['brand'] = null;
		if (!empty(Configure::read('EvShop.googleFeed.brand'))) {
			$this->_cachedValues['fallbacks']['brand'] = Configure::read('EvShop.googleFeed.brand');
		}

		$this->_cachedValues['fallbacks']['category'] = null;
		if (!empty(Configure::read('EvShop.googleFeed.category'))) {
			$this->_cachedValues['fallbacks']['category'] = Configure::read('EvShop.googleFeed.category');
		}

		$this->_cachedValues['useHierarchicalCategories'] = Configure::read('EvShop.googleFeed.useHierarchicalCategories');
		$this->_cachedValues['hierarchicalCategoryLimit'] = Configure::read('EvShop.googleFeed.hierarchicalCategoryLimit');

		if (!empty($this->Currency)) {
			$this->_cachedValues['currencies'] = $this->Currency->find('list');
		} else {
			//Fallback onto having 1 => GBP
			$this->_cachedValues['currencies'] = [
				1 => 'GBP',
			];
		}

		$this->_cachedValues['minImageWidth'] = 100;
		if (!empty(Configure::read('EvShop.googleFeed.minImageWidth'))) {
			$this->_cachedValues['minImageWidth'] = Configure::read('EvShop.googleFeed.minImageWidth');
		}

		$this->_cachedValues['minImageHeight'] = 100;
		if (!empty(Configure::read('EvShop.googleFeed.minImageHeight'))) {
			$this->_cachedValues['minImageHeight'] = Configure::read('EvShop.googleFeed.minImageHeight');
		}
	}

/**
 * Add a product to the google feed.
 *
 * @param xml   &$xml    The current feed.
 * @param array $product The product to add to the feed.
 * @return void.
 */
	protected function _addProductToFeed(&$xml, $product) {
		$isMultiVariant = count($product['Variant']) > 1;

		$product['Product']['url'] = $this->_getProductUrl($product);

		$topLevelCategory = $this->_getProductTopLevelCategory($product);

		foreach ($product['Variant'] as $variant) {
			$this->_addVariantToFeed($xml, $product, $variant, $topLevelCategory, $isMultiVariant);
		}
	}

/**
 * Creates a direct url for the product.
 *
 * @param array $product The product to get a url for.
 * @return string The full url for the product.
 */
	protected function _getProductUrl($product) {
		$productUrl = RouterUtil::getItemRoute('EvShop', 'Product');
		$productUrl[] = $product['Product']['id'];
		return Router::url($productUrl, true);
	}

/**
 * Get highest level of category that the product belongs to.
 *
 * @param array $product The product to get the top level category for.
 * @return array The top level category for the product.
 */
	protected function _getProductTopLevelCategory($product) {
		if (empty($product['CategoriesProduct'])) {
			return [];
		}

		return array_values($product['CategoriesProduct'])[0];
	}

/**
 * Add a variant to the google feed.
 *
 * @param xml   &$xml             The current feed.
 * @param array $product          The product the variant belongs to.
 * @param array $variant          The variant being added to the feed.
 * @param array $topLevelCategory The data for the top level category the product belongs to.
 * @param bool  $isMultiVariant   True if the product has multiple variants.
 * @return void.
 */
	protected function _addVariantToFeed(&$xml, $product, $variant, $topLevelCategory, $isMultiVariant) {
		if (!$this->_shouldAddToList($product, $variant)) {
			return;
		}

		$variant = $this->VariantPricing->calculateTax($variant, $this->Product);

		$xml->startElement('item');

			$xml->writeElement('g:id', $this->_getAttributeVariantId($variant));

			$xml->writeElement('g:item_group_id', $variant['product_id']);

			$xml->writeElement('g:title', $this->_getAttributeVariantTitle($product, $variant, $isMultiVariant));

			$xml->startElement('g:description');
				$xml->writeCData($this->_getAttributeVariantDescription($product, $variant));
			$xml->endElement();

			$googleCategory = $this->_getAttributeVariantCategory($product, $variant, $topLevelCategory);
			if (!empty($googleCategory)) {
					$xml->writeElement('g:google_product_category', $googleCategory);
			}

			$xml->writeElement('g:link', $this->_getAttributeVariantUrl($product, $variant));

			// Images
			$addedMainImage = false;
			$mainImage = $this->_getVariantMainImage($product, $variant);
			if (!empty($mainImage) && file_exists(WWW_ROOT . $mainImage)) {
				if ($this->_isImageSizeCorrect(WWW_ROOT . $mainImage)) {
					$mainImage = ltrim($mainImage, '/');
					$xml->writeElement('g:image_link', $this->_cachedValues['siteUrl'] . '/' . $mainImage);
					$addedMainImage = true;
				}
			}

			$additionalImages = $this->_getVariantAdditionalImages($product, $variant);
			if (!empty($additionalImages) && $addedMainImage) {
				$maxAdditionalImages = 10;
				$addedImages = 0;
				foreach ($additionalImages as $image) {
					if (empty($image) || !file_exists(WWW_ROOT . $image)) {
						continue;
					}

					$image = ltrim($image, '/');

					if ($image == $mainImage) {
						//Don't add the main image again as an additional image
						continue;
					}

					$xml->writeElement('g:additional_image_link', $this->_cachedValues['siteUrl'] . '/' . $image);

					$addedImages++;

					if ($addedImages >= $maxAdditionalImages) {
						break;
					}
				}
			}

			$xml->writeElement('g:condition', 'new');

			$xml->writeElement('g:availability', $this->_getAttributeVariantAvailability($product, $variant));

			// Pricing
			$pricing = $this->_getVariantPricing($product, $variant);
			$price = $this->_getAttributeVariantPrice($pricing);
			$salePrice = $this->_getAttributeVariantSalePrice($pricing);

			if ($price !== null) {
				$xml->writeElement('g:price', $price);
			}

			if ($salePrice != null) {
				$xml->writeElement('g:sale_price', $salePrice);
			}

			if (!empty($variant['sku'])) {
				$xml->writeElement('g:mpn', $variant['sku']);
			}

			if (!empty($variant['gtin'])) {
				$xml->writeElement('g:identifier_exists', 'true');
				$xml->writeElement('g:gtin', $variant['gtin']);
			} else {
				$xml->writeElement('g:identifier_exists', 'false');
			}

			$brand = $this->_getAttributeVariantBrand($product, $variant);

			if (!empty($brand['brand'])) {
				$xml->writeElement('g:brand', $brand['brand']);

				if (!empty($brand['customLabel'])) {
					$xml->writeElement('g:custom_label_0', $brand['customLabel']);
				}
			}

			$this->_addAdditionalXmlProperties($xml, $variant, $product);

		$xml->endElement();
	}

/**
 * Override to remove items from the list that match certain criteria
 *
 * @param EvShop.Product $product The product the variant belongs to
 * @param EvShop.Variant $variant The variant to check
 * @return bool Return true if the variant can be included in the product feed. False if it is to be skipped.
 */
	protected function _shouldAddToList($product, $variant) {
		return isset($variant['VariantPricing'][0]['price']) && $variant['VariantPricing'][0]['price'] > 0 && !empty($variant['sku']);
	}

/**
 * Get the id feed attribute for a variant.
 *
 * @param array $variant The variant to get the id feed attribute for.
 * @return string The id feed attribute for the provided variant.
 */
	protected function _getAttributeVariantId($variant) {
		return $variant['product_id'] . '_' . $variant['id'];
	}

/**
 * Get the title feed attribute for a variant.
 *
 * @param array $product        The product the variant belongs to.
 * @param array $variant        The variant to get the title feed attribute for.
 * @param bool  $isMultiVariant True if the product has multiple variants.
 * @return string The title feed attribute for the provided variant.
 */
	protected function _getAttributeVariantTitle($product, $variant, $isMultiVariant) {
		if (! empty($product['Product']['google_product_feed_title'])) {
			$title = $product['Product']['google_product_feed_title'];
		} else {
			$title = $product['Product']['name'];
		}

		if ($isMultiVariant) {
			// if we have actual variants append the variant name
			$title .= ': ' . $variant['name'];
		}

		return preg_replace("/[^a-z0-9\s-_.,&+]/i", "", $title);
	}

/**
 * Get the description feed attribute for a variant.
 *
 * @param array $product The product the variant belongs to.
 * @param array $variant The variant to get the description feed attribute for.
 * @return string The description feed attribute for the provided variant.
 */
	protected function _getAttributeVariantDescription($product, $variant) {
		if (! empty($product['Product']['google_product_feed_description'])) {
			$description = strip_tags($product['Product']['google_product_feed_description']);
		} else {
			$description = strip_tags($product['Product']['description']);
		}

		return preg_replace('/[\x00-\x1F\x80-\xFF]/', '', $description);
	}

/**
 * Get the category feed attribute for a variant. If hierarchical categories
 * are enabled then a list of categories will be generated, otherwise the top
 * level category for the variant will be used. If a fallback is available and
 * no category can be found for the variant then the fallback will be used.
 *
 * @param array $product          The product the variant belongs to.
 * @param array $variant          The variant to get the category feed attribute for.
 * @param array $topLevelCategory The data for the top level category the product belongs to.
 * @return string.
 */
	protected function _getAttributeVariantCategory($product, $variant, $topLevelCategory) {
		$googleCategory = '';

		if ($this->_cachedValues['useHierarchicalCategories']) {
			$googleCategory = $this->_getAttributeVariantHierarchicalCategories($product, $variant);
		}

		if (!empty($googleCategory)) {
			return $googleCategory;
		}

		if (!empty($topLevelCategory)) {
			return $this->_getCategoryFeedValue($topLevelCategory);
		}

		if (!empty($this->_cachedValues['fallbacks']['category'])) {
			return $this->_cachedValues['fallbacks']['category'];
		}

		return $googleCategory;
	}

/**
 * Get the category feed attribute for a variant in hierarchical format.
 * Find the categories associated with the product directly in depth and sequence order.
 * Then find their parents and format them into hierarchies. Generate a path for the category
 * from their top level parent down to the category. If the hierarchy is already included or
 * is a sub hierarchy of another category then the category is skipped.
 *
 * @param array $product The product the variant belongs to.
 * @param array $variant The variant to get the category feed attribute for.
 * @return string.
 */
	protected function _getAttributeVariantHierarchicalCategories($product, $variant) {
		if (empty($product[$this->Product->alias][$this->Product->primaryKey])) {
			return '';
		}

		$productId = $product[$this->Product->alias][$this->Product->primaryKey];
		$Category = $this->Product->CategoriesProduct->Category;
		$categoryAlias = $Category->alias;
		$categoryKey = $Category->primaryKey;

		//Get the deepest categories available on the product
		$categories = $Category->findByAssociatedProduct(
			$productId,
			[
				'order' => [
					$categoryAlias . '.level' => 'DESC',
					$categoryAlias . '.lft' => 'ASC',
				],
			]
		);

		if (empty($categories)) {
			return '';
		}

		$categoryHierarchies = [];
		$categorySubHierarchies = [];
		foreach ($categories as $category) {
			if (count($categoryHierarchies) === $this->_cachedValues['hierarchicalCategoryLimit']) {
				break;
			}

			$feedValue = $this->_getCategoryFeedValue($category);

			if (empty($feedValue)) {
				continue;
			}

			if (empty($category[$categoryAlias]['parent_id'])) {
				//Can't find parent so include the category by itself.
				if (
					in_array($feedValue, $categoryHierarchies)
					|| in_array($feedValue, $categorySubHierarchies)
				) {
					continue;
				}

				$categoryPaths[] = $feedValue;
				continue;
			}

			if (empty($category[$categoryAlias]['lft']) || empty($category[$categoryAlias]['rght'])) {
				continue;
			}

			$parentCategories = $Category->findParents($category);

			if (empty($parentCategories)) {
				continue;
			}

			/*
			 * Extract the parent hierarchy paths at each depth so that paths aren't duplicated.
			 * If the current category path is included in either the category hierarchies or in
			 * the sub hierarchies then don't include this category hierarchy.
			 */
			$categoryHierarchy = [];
			$parentHierarchies = [];

			foreach ($parentCategories as $parentCategory) {
				$parentFeedValue = $this->_getCategoryFeedValue($parentCategory);

				$categoryHierarchy[] = $parentFeedValue;
				$parentHierarchies[] = implode(' > ', $categoryHierarchy);
			}

			$categoryHierarchy[] = $feedValue;
			$categoryHierarchy = implode(' > ', $categoryHierarchy);

			if (
				in_array($categoryHierarchy, $categoryHierarchies)
				|| in_array($categoryHierarchy, $categorySubHierarchies)
			) {
				continue;
			}

			$categoryHierarchies[] = $categoryHierarchy;
			$categorySubHierarchies += $parentHierarchies;
		}

		return implode(', ', $categoryHierarchies);
	}

/**
 * Get the value of a category to use in the feed.
 * If a value has been specified specifically on the category in the "google_product_category" field then use that,
 * otherwise try to use the category name.
 *
 * @param array $category Category data.
 * @return string.
 */
	protected function _getCategoryFeedValue($category) {
		if (!empty($category['Category']['google_product_category'])) {
			return $category['Category']['google_product_category'];
		}

		if (!empty($category['Category']['name'])) {
			return $category['Category']['name'];
		}

		return '';
	}

/**
 * Get the url feed attribute for a variant.
 *
 * @param array $product The product the variant belongs to.
 * @param array $variant The variant to get the url feed attribute for.
 * @return string The url feed attribute for the provided variant.
 */
	protected function _getAttributeVariantUrl($product, $variant) {
		if (empty($product['Product']['url'])) {
			return '';
		}

		return $product['Product']['url'] . '?variant=' . $variant['id'];
	}

/**
 * Get the main image for a variant in the feed.
 *
 * @param array $product The product the variant belongs to.
 * @param array $variant The variant to get an image of.
 * @return string The filepath of the image to use as the variant's main image.
 */
	protected function _getVariantMainImage($product, $variant) {
		$image = '';

		if (!empty($product['ListingImage'][0]['filepath'])) {
			$image = $product['ListingImage'][0]['filepath'];

			if (!file_exists($image)) {
				$image == '';
			}
		}

		return $image;
	}

/**
 * Get the additional images for a variant in the feed.
 *
 * @param array $product The product the variant belongs to.
 * @param array $variant The variant to get an image of.
 * @return array A list of filepaths of the images to use as the variant's additional images.
 */
	protected function _getVariantAdditionalImages($product, $variant) {
		//Method stub
		return [];
	}

/**
 * Check if an image is the correct size to be added to the feed.
 *
 * @param string $filepath The filepath to the image.
 * @return bool True if the size is correct, false otherwise.
 */
	protected function _isImageSizeCorrect($filepath) {
		if (!file_exists($filepath)) {
			return false;
		}

		$imageSize = getimagesize($filepath);

		if (empty($imageSize)) {
			return false;
		}

		//Width is return in array element 0 and height in array element 1
		if (
			!empty($imageSize[0]) && $imageSize[0] >= $this->_cachedValues['minImageWidth']
			&& !empty($imageSize[1]) && $imageSize[1] >= $this->_cachedValues['minImageHeight']
		) {
			return true;
		}

		return false;
	}

/**
 * Get the availability feed attribute for a variant.
 *
 * @param array $product The product the variant belongs to.
 * @param array $variant The variant to get the availability feed attribute for.
 * @return string The availability feed attribute for the provided variant.
 */
	protected function _getAttributeVariantAvailability($product, $variant) {
		$availability = 'in stock';

		if (CakePlugin::loaded('EvInventory')) {
			if (!empty($variant['Inventory'])) {
				if (InventoryLib::hasEnoughStock($variant, 1)) {
					$availability = 'in stock';
				} else {
					$availability = 'out of stock';
				}
			} else {
				$availability = 'out of stock';
			}
		}

		return $availability;
	}

/**
 * Get the variant pricing for a variant.
 *
 * @param array $product The product the variant belongs to.
 * @param array $variant The variant to get pricing for.
 * @return array The pricing for the provided variant.
 */
	protected function _getVariantPricing($product, $variant) {
		if (empty($variant['VariantPricing'][0])) {
			return [];
		}

		return $variant['VariantPricing'][0];
	}

/**
 * Get the price feed attribute for a variant.
 *
 * @param array $pricing The pricing of the variant.
 * @return string The price feed attribute for the provided variant.
 */
	protected function _getAttributeVariantPrice($pricing) {
		if (!isset($pricing['price_incTax']) || !is_numeric($pricing['price_incTax'])) {
			return null;
		}

		$price = number_format($pricing['price_incTax'], 2);

		if (!empty($pricing['currency_id']) && !empty($this->_cachedValues['currencies'][$pricing['currency_id']])) {
			$price .= ' ' . $this->_cachedValues['currencies'][$pricing['currency_id']];
		}

		return $price;
	}

/**
 * Get the sale price feed attribute for a variant.
 *
 * @param array $pricing The pricing of the variant.
 * @return string The sale price feed attribute for the provided variant.
 */
	protected function _getAttributeVariantSalePrice($pricing) {
		if (
			!isset($pricing['sale_price_incTax'])
			|| !is_numeric($pricing['sale_price_incTax'])
			|| $pricing['sale_price_incTax'] <= 0
			|| !isset($pricing['price_incTax'])
			|| !is_numeric($pricing['price_incTax'])
			|| $pricing['sale_price_incTax'] >= $pricing['price_incTax']
		) {
			return null;
		}

		$salePrice = number_format($pricing['sale_price_incTax'], 2);

		if (!empty($pricing['currency_id']) && !empty($this->_cachedValues['currencies'][$pricing['currency_id']])) {
			$salePrice .= ' ' . $this->_cachedValues['currencies'][$pricing['currency_id']];
		}

		return $salePrice;
	}

/**
 * Get the brand feed attribute for a variant.
 *
 * @param array $product The product the variant belongs to.
 * @param array $variant The variant to get the brand feed attribute for.
 * @return array An array containing the brand feed attribute and the brand custom label attribute.
 */
	protected function _getAttributeVariantBrand($product, $variant) {
		$brand = [
			'brand' => null,
			'customLabel' => null,
		];

		// Assign the parent brand, when set, falling back to the traditional brand name field when no parent is
		// available.
		if (!empty($product['Brand']['ParentBrand']['name'])) {
			$brand['brand'] = $product['Brand']['ParentBrand']['name'];

			if (!empty($product['Brand']['name'])) {
				// Assemble the brand custom label, containing the parent and child brand names
				$brand['customLabel'] = $product['Brand']['ParentBrand']['name'];
				$brand['customLabel'] .= ' > ';
				$brand['customLabel'] .= $product['Brand']['name'];
			}
		} elseif (! empty($product['Brand']['name'])) {
			$brand['brand'] = $product['Brand']['name'];

			// assemble the brand custom label to contain just the brand name
			$brand['customLabel'] = $product['Brand']['name'];
		} elseif (!empty($this->_cachedValues['fallbacks']['brand'])) {
			$brand['brand'] = $this->_cachedValues['fallbacks']['brand'];
		}

		return $brand;
	}

/**
 *  Override to add additional properties into the generated xml file.
 *
 * @param xml   &$xml    The xml document that is being constructed.
 * @param array $variant The current variant.
 * @param array $product The product this variant belongs to.
 * @return void.
 */
	protected function _addAdditionalXmlProperties(&$xml, $variant, $product) {
		// Method stub
	}
}
