<?php
App::uses('EvShopAppController', 'EvShop.Controller');
App::uses('InflectorExt', 'EvInflector.Lib');

class VariantsController extends EvShopAppController {

	public function beforeFilter() {
		$this->adminActions[] = 'admin_manage';
		unset($this->adminActions[array_search('admin_edit', $this->adminActions)]);
		unset($this->adminActions[array_search('admin_delete', $this->adminActions)]);
		parent::beforeFilter();
	}

	protected function _adminManageToolbar($id) {
		$Model = $this->{$this->modelClass};
		$modelAlias = $Model->alias;

		$actions = array(
			'All Products' => array(
				'url' => array(
					'controller' => 'products',
					'action' => 'index'
				),
				'class' => 'toolbar__btn--listing'
			),
			'Edit Product' => array(
				'url' => array(
					'controller' => 'products',
					'action' => 'edit',
					$id
				),
				'class' => 'toolbar__btn--edit'
			)
		);

		return $actions;
	}

	/**
	 * just incase redirect to products index
	 */
	public function admin_index() {
		$this->redirect(
			array(
				'controller' => 'products'
			)
		);
	}

/**
 * display the variants management
 * styled as just pricing rather then variants to prevent confusion
 *
 * @param int $id product id
 * @return void
 */
	public function admin_manage($id) {
		$Model = $this->{$this->modelClass};

		if ($this->_handleAdminManagePost($Model, $id)) {
			$this->request->data['Variant'] = $Model->readForManage($id);
		}

		if (CakePlugin::loaded('EvCurrency')) {
			$this->Currencies->injectAdminForm(array(), $Model, $id);

			$Currencies = EvClassRegistry::init('EvCurrency.Currency');
			$defaultCurrency = $Currencies->find('first', array(
				'conditions' => array(
					'is_default' => '1'
				)
			));

			if (! empty($defaultCurrency)) {
				$this->set('defaultCurrency', $defaultCurrency['Currency']['id']);
			}
		}

		$OptionGroupsModel = EvClassRegistry::init('EvShop.OptionGroup');
		$optionIds = Hash::extract($this->request->data, 'Variant.{n}.Option.{n}.id');

		$optionGroups = $OptionGroupsModel->find('all', array(
			'contain' => array(
				'Option' => [
					'conditions' => [
						'Option.id' => $optionIds
					]
				]
			)
		));

		foreach ($optionGroups as $groupIndex => $group) {
			if (!count($group['Option']) > 0) {
				//Remove groups that don't contain any options
				unset($optionGroups[$groupIndex]);
			}
		}

		$this->set('optionGroups', $optionGroups);

		$actionLabel = 'Manage';
		$this->set(
			array(
				'actionLabel' => $actionLabel,
				'title_for_layout' => $actionLabel . ' ' . InflectorExt::humanize($Model->displayName),
				'toolbar' => $this->_adminManageToolbar($id),
				'fields' => $this->_adminVariantManageFields()
			)
		);

		$this->view = 'EvShop./Variants/admin_manage';
	}

/**
 * Handles the post/put request for admin_manage
 *
 * @param Array &$Model Cake model
 * @param int $id ID of item being managed
 * @return bool True if processing the post request was successful, false if there was an error
 */
	protected function _handleAdminManagePost(&$Model, $id) {
		$error = false;
		$modelAlias = $Model->alias;

		if (!($this->request->is('post') || $this->request->is('put'))) {
			return !$error;
		}

		if ($Model->saveMany($this->request->data['Variant'], array('deep' => true))) {

			// save was successful
			$this->Flash->success(
				array(
					'title' => 'Save Successful',
					'description' => 'Pricing data has been saved successfully'
				)
			);

			// check whether to redirect to product listing or pricing page again
			if (isset($this->request->data['return']) && $this->request->data['return'] == 1) {

				$this->Session->write("Admin.$modelAlias.return", true);
				$return = array(
					'action' => 'manage',
					$id
				);
			} else {

				$this->Session->delete("Admin.$modelAlias.return");

				$redirect = $this->Session->read("Admin.$modelAlias.filter");
				if (strpos($redirect, 'Filter') !== false) {
					$return = $redirect;
				} else {
					if (in_array('EvShopProducts', App::objects('Controller'))) {
						$return = array(
							'plugin' => false,
							'controller' => 'ev_shop_products',
						);
					} else {
						$return = array(
							'plugin' => 'ev_shop',
							'controller' => 'products'
						);
					}
					$return['action'] = 'index';
				}
			}

			$this->redirect($return);

		} else {
			// uh oh, fail.
			$error = true;
			$this->Flash->fail(
				array(
					'title' => 'Save Failed',
					'description' => 'There was a problem saving the pricing. Please check that you have correctly entered valid prices in each currency field. (e.g 20.00 and not just 20).'
				)
			);

		}

		if ($error) {
			$this->_handleAdminManagePostingError($Model, $id);
		}

		return !$error;
	}

/**
 * Handles any errors that may occur when posting to admin_manage
 *
 * @param Array &$Model Cake model
 * @param int $id ID of model that was unsuccessfully saved
 * @return void
 */
	protected function _handleAdminManagePostingError(&$Model, $id) {
		// Re-populate fields with the entered data after form validation fails
		$originalFieldData = $Model->readForManage($id); // the previously stored field data
		$formattedArray = array();
		$indexCount = 0;

		foreach ($this->request->data['Variant'] as $variant) {
			$variantVarPrices = array(); // array for setting the newly entered prices

			if (array_key_exists('VariantPricing', $variant)) {
				foreach ($variant['VariantPricing'] as $variantPrice) {
					foreach ($variantPrice as $type => $price) {
						if ($price == 0) {
							$price = null;
						}
						$variantVarPrices[$type] = $price;
					}
				}

				$originalVariantPrices = array(); // array for setting the previously stored prices for storing the lowest price

				foreach ($originalFieldData[$indexCount]['VariantPricing'] as $originalVariantPrice) {
					foreach ($originalVariantPrice as $type => $price) {
						if ($price == 0) {
							$price = null;
						}
						$originalVariantPrices[$type] = $price;
					}
				}

				$formattedArray[] = array(
					'Variant' => array(
						'id' => $variant['id'],
						'product_id' => $id,
						'name' => $variant['name'],
						'sku' => $variant['sku'],
						'cost' => $variant['cost'],
						'sequence' => $variant['sequence'],
						'is_active' => $variant['is_active'],
						'rrp' => $variantVarPrices['rrp'],
						'price' => $variantVarPrices['price'],
						'sale_price' => $variantVarPrices['sale_price']
					),
					'Product' => $originalFieldData[$indexCount]['Product'],
					'VariantPricing' => array(
						$variantVarPrices['currency_id'] => array(
							'id' => $variant['id'],
							'variant_id' => $indexCount + 1,
							'rrp' => $variantVarPrices['rrp'],
							'price' => $variantVarPrices['price'],
							'sale_price' => $variantVarPrices['sale_price'],
							'trade_price' => (isset($variantVarPrices['trade_price']) ? $variantVarPrices['trade_price'] : null),
							'currency_id' => $variantVarPrices['currency_id']
						)
					),
					'option_desc' => !empty($originalFieldData[$indexCount]['option_desc'])
						? $originalFieldData[$indexCount]['option_desc']
						: null,
					'id' => $originalFieldData[$indexCount]['id'],
					'product_id' => $originalFieldData[$indexCount]['product_id'],
					'name' => $variant['name'],
					'sku' => $variant['sku'],
					'cost' => $variant['cost'],
					'sequence' => $variant['sequence'],
					'is_active' => $variant['is_active'],
					'rrp' => $variantVarPrices['rrp'],
					'price' => $variantVarPrices['price'],
					'sale_price' => $variantVarPrices['sale_price']
				);

				if (CakePlugin::loaded('EvInventory')) {
					$formattedArray[$indexCount]['Inventory'] = [
						'id' => $variant['Inventory']['id'],
						'model' => $originalFieldData[$indexCount]['Inventory']['model'],
						'model_id' => $originalFieldData[$indexCount]['Inventory']['model_id'],
						'stock' => $variant['Inventory']['stock'],
						'warning_level' => $variant['Inventory']['warning_level'],
						'warning_action' => $variant['Inventory']['warning_action'],
						'oos_action' => $variant['Inventory']['oos_action'],
						'created' => $originalFieldData[$indexCount]['Inventory']['created'],
						'modified' => $originalFieldData[$indexCount]['Inventory']['modified']
					];
				}

				if (array_key_exists('Option', $originalFieldData[$indexCount])) {
					$formattedArray[$indexCount]['Option'] = $originalFieldData[$indexCount]['Option'];
				}

				if (array_key_exists('lowest_price', $originalVariantPrices)) {
					$formattedArray[$indexCount]['VariantPricing'][$variantVarPrices['currency_id']]['lowest_price'] = $originalVariantPrices['lowest_price'];
				}
				$indexCount++;
			}
		}

		$this->request->data['Variant'] = $formattedArray;
	}
}
