<?php
App::uses('EvShopAppController', 'EvShop.Controller');
App::uses('Folder', 'Utility');
App::uses('File', 'Utility');
App::uses('InflectorExt', 'EvInflector.Lib');

class ProductsController extends EvShopAppController {

	/**
	 * redefine to allow the index / view page through the auth component!
	 */
	public function beforeFilter() {
		$this->adminActions[] = 'admin_attention';
		$this->adminActions[] = 'admin_stock';

		parent::beforeFilter();

		$this->Auth->allow(array('index', 'view'));
	}

	protected function _adminFilterFields() {
		$fields = parent::_adminFilterFields();
		$CategoryModel = EvClassRegistry::init('EvShop.Category');
		$fields['CategoriesProduct.category_id'] = array(
			'label' => 'Category',
			'type' => 'select',
			'options' => $CategoryModel->getForDropDown(),
			'compare' => array(
				'CategoriesProduct.category_id' => "%s"
			)
		);

		return $fields;
	}

	protected function _adminIndexPaginate() {
		$paginate = parent::_adminIndexPaginate();

		$paginate['joins'][] = array(
			'table' => 'ev_shop_categories_products',
			'alias' => 'CategoriesProduct',
			'type' => 'LEFT',
			'conditions' => array(
				'Product.id = CategoriesProduct.product_id',
			)
		);

		$paginate['group'] = 'Product.id';

		return $paginate;
	}

	/**
	 * product index page
	 * list all products out
	 *
	 */
	public function index() {
		$this->view = 'EvShop./Fallbacks/Products/index';
		$pageId = Configure::read('EvShop.pageIds.product');

		// check if we have a specific content page set
		// if so - set the template
		if (! empty($pageId)) {
			$pageData = $this->assignPage($pageId);

			if (! empty($pageData)) {
				$this->view = $this->Tpl->getTemplate($pageData, 'Page');
			}
		}

		$this->set(
			'listing',
			$this->Products->listing(array(), false)
		);

		$this->Breadcrumb->addCrumb(__('All Products'));
	}

	/**
	 * product view details page
	 *
	 * @param 	int 	Product ID
	 */
	public function view($id) {
		$this->view = 'EvShop./Fallbacks/Products/view';

		$data = $this->{$this->modelClass}->readForView($id);
		if (empty($data)) {
			throw new NotFoundException();
		}

		if (is_array($data['Variant']) && count($data['Variant']) == 1) {
			// This is a single variant product. Default to that one variant.
			$initialVariantId = current($data['Variant'])['id'];
		} elseif (!empty($this->request->query['variant'])) {
			// We have been given a variant parameter. Use it.
			$initialVariantId = $this->request->query['variant'];
		} else {
			// Nothing should be preselected
			$initialVariantId = false;
		}
		$this->set('initialVariantId', $initialVariantId);

		// extract the lowest product variant price
		if (! empty($data['Variant'])) {
			$data['Product']['price'] = $this->Products->extractLowestVariantPrice(
				$data['Variant']
			);
		}

		$this->set('data', $data);
		$this->Meta->set($data);

		if (! empty($data)) {
			$dataTemplate = $this->Tpl->getTemplate($data);
			if ($dataTemplate != false) {
				$this->view = $dataTemplate;
			}
		}

		$this->_getProductBreadcrumb($data);
	}

	/**
	 * display all the products that don't have missing variant pricing,
	 * if a product has no variant pricings associated with it then they will not be displayed
	 * in listings.
	 * @return [type] [description]
	 */
	public function admin_attention() {
		$Model = $this->{$this->modelClass};
		$modelAlias = $Model->alias;

		$this->set('data', $Model->find(
			'all',
			[
				'joins' => [
					[
						'table' => 'ev_shop_categories_products',
						'alias' => 'CategoriesProduct',
						'type' => 'LEFT',
						'conditions' => array(
							'Product.id = CategoriesProduct.product_id',
						)
					],
					[
						'table' => 'ev_shop_variants',
						'alias' => 'Variant',
						'conditions' => [
							'Variant.product_id = Product.id',
						]
					],
					[
						'table' => 'ev_shop_variant_pricings',
						'alias' => 'VariantPricing',
						'type' => 'left',
						'conditions' => [
							'VariantPricing.variant_id = Variant.id'
						]
					]
				],
				'conditions' => [
					'AND' => [
						'Variant.is_active' => true,
						'VariantPricing.id IS NULL'
					]
				],
				'group' => 'Product.id',
			]
		));

		$this->_processMultiEdit();
		$this->_adminPopulateFilterLookups();
		$this->set('filter', $this->_adminFilterFields());
		$this->set('columns', $this->_adminIndexColumns());
		$this->set('toolbar', $this->_adminAttentionToolbar());
		$this->set('actions', $this->_adminIndexActions());
		$this->set('multiEditActions', $this->_adminMultiEditActions());

		$this->set('title_for_layout', InflectorExt::camelToPluralize($Model->displayName));

		$this->view = 'EvShop./Products/admin_attention';
	}

	/**
	 * Defines the toolbar buttons displayed in admin_index
	 *
	 * @return array
	 */
	protected function _adminIndexToolbar() {
		$toolbar = parent::_adminIndexToolbar();

		$toolbar['Products That Need Attention'] = [
			'url' => [
				'action' => 'admin_attention'
			],
			'class' => 'toolbar__btn--listing'
		];

		return $toolbar;
	}

	protected function _adminAttentionToolbar() {
		return $toolbar = [
			'Add New' => [
				'url' => [
					'action' => 'add'
				]
			],
			'All Products' => [
				'url' => [
					'action' => 'index'
				],
				'class' => 'toolbar__btn--listing'
			]
		];
	}

	/**
	 * add the variant management link
	 * prices / stock control (if active)
	 */
	protected function _adminIndexActions() {
		if (in_array('EvShopVariants', App::objects('Controller'))) {
			$variantManageUrl = array(
				'plugin' => false,
				'controller' => 'ev_shop_variants',
				'action' => 'manage'
			);
		} else {
			$variantManageUrl = array(
				'plugin' => 'ev_shop',
				'controller' => 'variants',
				'action' => 'manage'
			);
		}

		$actions = array(
			'Pricing' => array(
				'cell' => array(
					'class' => 'action pricing'
				),
				'link' => array(
					'url' => $variantManageUrl,
					'text' => '<i class="fa fa-money"></i>',
					'options' => array(
						'escape' => false
					)
				)
			)
		) + parent::_adminIndexActions();

		return $actions;
	}

	/**
	 * form fields -check for brands setting and auto remove
	 */
	protected function _adminFormFields() {
		$fields = parent::_adminFormFields();

		$fields['Product.description']['type'] = 'html';

		if (! Configure::read('EvShop.showBrands')) {
			unset($fields['Product.brand_id']);
		}

		$fields['tabs'] = [];

		if (!empty(Configure::read('EvShop.showAttributes'))) {
			$fields = $this->_adminAddAttributes($fields);
		}

		// if a user with admin permissions is viewing the admin, move the
		// google product feed entries to a new tab. Otherwise, remove from form
		if ($this->Permissions->hasSuperAdminPermission($this->Auth->user())) {
			// quickly update the google product feed description field to plain_text
			$fields['Product.google_product_feed_description']['type'] = 'text_plain';

			$fields['tabs']['Google Product Feed'] = [
				'Product.google_product_feed_title' => $fields['Product.google_product_feed_title'],
				'Product.google_product_feed_description' => $fields['Product.google_product_feed_description']
			];
		} else {
			// remove from the form
			unset(
				$fields['Product.google_product_feed_title'],
				$fields['Product.google_product_feed_description']
			);
		}

		return $this->_discardTabbedFormFields($fields);
	}

/**
 * adds the product attributes to fields
 *
 * @param array $fields fields to be displayed in admin
 * @return array
 */
	protected function _adminAddAttributes($fields) {
		$this->loadModel('EvShop.ProductAttributeGroup');
		$productAttributeGroups = $this->ProductAttributeGroup->find('all', [
			'contain' => [
				'ProductAttribute'
			]
		]);

		if (!empty($this->request->data['ProductAttribute'])) {
			$attributeValues = Hash::combine($this->request->data['ProductAttribute'], '{n}.product_attribute_group_id', '{n}.id');
		}

		$productAttributeFields = [];
		foreach ($productAttributeGroups as $attributeGroup) {
			$groupId = $attributeGroup['ProductAttributeGroup']['id'];
			$productAttributeFields['ProductAttribute.ProductAttribute.' . $groupId] = [
				'label' => $attributeGroup['ProductAttributeGroup']['name'],
				'type' => 'select',
				'options' => Hash::combine($attributeGroup['ProductAttribute'], '{n}.id', '{n}.name'),
				'default' => !empty($attributeValues[$groupId]) ? $attributeValues[$groupId] : null
			];
		}
		return ArrayUtil::addAfter($fields, 'Product.template_id', $productAttributeFields );
	}

/**
 * admin populate lookups
 */
	protected function _adminPopulateLookups() {
		$Model = $this->{$this->modelClass};
		parent::_adminPopulateLookups();

		if (Configure::read('EvShop.showBrands')) {
			$this->set(
				'brands',
				$Model->Brand->getForProductAdminDropDown()
			);
		}
	}

/**
 * Defines the buttons in the toolbar displayed on an admin_form.
 *
 * @param int $id The id of the record in the form.
 * @return array Form toolbar.
 */
	protected function _adminFormToolbar($id = null) {
		$Model = $this->{$this->modelClass};
		$modelAlias = $Model->alias;

		$actions = [];

		if (!empty($id)) {
			$actions['Copy'] = [
				'url' => ['action' => 'copy', $id],
				'class' => 'toolbar__btn--copy'
			];


			$actions['Edit Variants'] = [
				'url' => [
					'plugin' => 'ev_shop',
					'controller' => 'variants',
					'action' => 'manage',
					$id
				],
				'class' => 'toolbar__btn--edit'
			];
		}

		return array_merge($actions, parent::_adminFormToolbar($id));
	}

/**
 * Copies an item - uses a custom copy method as this is too complex for the copyable behaviour
 * to handle.
 *
 * @param int $id Id of item to copy.
 * @return void
 */
	public function admin_copy($id) {
		$Model = $this->{$this->modelClass};
		$modelAlias = $Model->alias;

		if (!empty($this->request->data)) {
			$Model->setCopyData($this->request->data);
			$copyResult = $Model->copy($id);

			if ($copyResult) {
				$this->Session->setFlash(
					[
						'title' => InflectorExt::humanize($Model->displayName) . " copied",
						'description' => 'Item has been successfully copied!'
					],
					'flash_success'
				);

				return $this->redirect(array('action' => 'edit', $Model->id));
			} else {
				$this->Session->setFlash(
					[
						'title' => 'Copy failed',
						'description' => 'Failed to copy ' . InflectorExt::humanize($Model->displayName)
					],
					'flash_fail'
				);

				return $this->redirect(array('action' => 'edit', $id));
			}
		}

		$this->request->data = $Model->findById($id);

		$this->set('title_for_layout', 'Copy ' . InflectorExt::humanize($Model->displayName));

		// Set the default template, this can be overridden in controllers using $this->view
		// or $this->render().
		$this->view = 'EvShop.Products/admin_copy';
	}

	/**
	 * redefine admin edit to setup inject form
	 * for variants
	 */
	public function admin_edit($id = null) {
		$Model = $this->{$this->modelClass};

		$this->toInject('components', 'EvShop.Categories');
		$this->toInject('helpers', 'EvShop.Categories');

		if (Configure::read('EvShop.showVariants')) {
			$this->toInject('components', 'EvShop.Variants');
			$this->toInject('helpers', 'EvShop.Variants');
		}

		// If the product is being created or if the "modify pricing after saving" checkbox was checked
		// then redirect the variant manage page. The _adminEditSaveRedirect will do the redirecting.
		$this->manageVariantAfterSave = false;
		if (empty($id)) {
			$this->manageVariantAfterSave = true;
		} elseif (!empty($this->request->data['pricing'])) {
			$this->manageVariantAfterSave = true;
		}

		// Set flag to know we are saving from the admin
		if (($this->request->is('post') || $this->request->is('put'))) {
			$this->adminSaveOptions['savingFromAdmin'] = true;
		}

		parent::admin_edit($id);

		// Bring in Variant fields
		$this->set('variantFields', $this->_adminVariantManageFields());

		// Bring in currencies
		if (CakePlugin::loaded('EvCurrency')) {
			$this->Currencies->injectAdminForm(array(), $Model, $id);

			$Currencies = EvClassRegistry::init('EvCurrency.Currency');
			$defaultCurrency = $Currencies->find('first', array(
				'conditions' => array(
					'is_default' => '1'
				)
			));

			if (! empty($defaultCurrency)) {
				$this->set('defaultCurrency', $defaultCurrency['Currency']['id']);
			}
		}

		//If this is a submitted form merge in the request data
		if (($this->request->is('post') || $this->request->is('put')) && !empty($Model->validationErrors)) {
			// Merge the latest data into the readForEdit which has more info about variant options
			$existing = $Model->readForEdit($id);
			$this->request->data = Hash::merge($existing, $this->request->data);
		}

		//Pull the options out of the request data and organise by option group
		$chosenOptions = Hash::combine($this->request->data,
			'Variant.{n}.Option.{n}.id',
			'Variant.{n}.Option.{n}.name',
			'Variant.{n}.Option.{n}.OptionGroup.name'
		);

		//Need to get the filters out
		$optionGroups = Hash::combine($this->request->data,
			'Variant.{n}.Option.{n}.OptionGroup.name',
			'Variant.{n}.Option.{n}.OptionGroup'
		);

		//Put the filters into the options array
		foreach ($optionGroups as $groupName => $groupData) {
			if (array_key_exists($groupName, $chosenOptions)) {
				$extraGroupData = $this->_getSelectedOptionGroupDataForImageEdit($groupData);
				$chosenOptions[$groupName] = $chosenOptions[$groupName] + $extraGroupData;
			}
		}

		if (!empty($this->viewVars['metaData'])) {
			if (isset($this->viewVars['metaData']['MetaData.title'])) {
				$this->viewVars['metaData']['MetaData.title']['displayInfo'] = 'Tokens can be used to add dynamic info to product meta data. To use a token add the data you want to show inside curly brackets. For example if you want to add the product price the token would be {Product.price}.';
			}

			if (isset($this->viewVars['metaData']['MetaData.description'])) {
				$this->viewVars['metaData']['MetaData.description']['displayInfo'] = 'Tokens can be used to add dynamic info to product meta data. To use a token add the data you want to show inside curly brackets. For example if you want to add the product price the token would be {Product.price}.';
			}
		}

		$this->set('chosenOptions', $chosenOptions);
		$this->view = 'EvShop./Products/admin_form';
	}

/**
 * Determines the redirect URL for admin_edit after saving. Extend in your controller to add
 * additional functionality.
 *
 * @param int $id Primary key
 * @return array|string Redirect route
 */
	protected function _adminEditSaveRedirect($id) {
		if (!empty($this->manageVariantAfterSave)) {
			if (in_array('EvShopVariants', App::objects('Controller'))) {
				$this->adminRedirect = array(
					'plugin' => false,
					'controller' => 'ev_shop_variants',
				);
			} else {
				$this->adminRedirect = array(
					'plugin' => 'ev_shop',
					'controller' => 'variants'
				);
			}

			$this->adminRedirect['action'] = 'manage';
			$this->adminRedirectRequiresId = true;
		}

		return parent::_adminEditSaveRedirect($id);
	}

	protected function _getSelectedOptionGroupDataForImageEdit($data) {
		$chosenData = [];

		$chosenData['option_group_id'] = $data['id'];

		return $chosenData;
	}

/**
 * Define an array of actions that can be performed on multiple items on the
 * admin listing page.
 *
 * @return array
 */
	protected function _adminMultiEditActions() {
		$actions = parent::_adminMultiEditActions();
		if (!empty(Configure::read('EvShop.batchManageStock'))) {
			$actions['ManageStock'] = 'Manage Stock';
		}
		return $actions;
	}

/**
 * Saves the product Ids and redirects the the stock page to avoid the default redirect back to the index
 * @return null
 */
	protected function _processMultiEditManageStock($productIds) {
		$this->Session->write('EvShop.batchEditStock.productIds', $productIds);
		$this->redirect([
			'action' => 'stock'
		]);
	}

/**
 * Takes multiple products and presents stock levels in a grid
 * @return null
 */
	public function admin_stock() {
		//
		$Inventory = EvClassRegistry::init('EvInventory.Inventory');
		if (!empty($this->request->data['Inventory'])) {

			$Inventory->saveMany($this->request->data['Inventory']);

			if (empty($this->request->data('return'))) {
				$this->redirect([
					'action' => 'index'
				]);
			}
		}

		$productIds = $this->Session->read('EvShop.batchEditStock.productIds', []);
		$Model = $this->{$this->modelClass};

		// First find any variants that don't have an inventory saved and create one.
		// This should never be the case but this code is here to guard against the situation.
		// This is easier than trying to create a lot of different inventories from a form and linking them to variants
		$variantsWithoutInventories = $Model->Variant->find('list', [
			'joins' => [
				[
					'table' => 'ev_inventory_inventories',
					'alias' => 'Inventory',
					'type' => 'LEFT',
					'conditions' => [
						'Inventory.model' => 'EvShop.Variant',
						'Inventory.model_id = Variant.id'
					]
				]
			],
			'conditions' => [
				'Inventory.id' => null,
				'Variant.product_id' => $productIds
			]
		]);

		// Create inventories for those that are missing
		$newInventories = [];
		foreach ($variantsWithoutInventories as $variantId => $variantName) {
			$newInventories[] = [
				'Inventory' => [
					'model' => 'EvShop.Variant',
					'model_id' => $variantId,
					'stock' => 0,
					'warning_level' => 0,
					'warning_action' => [],
					'oos_action' => []
				]
			];
		}
		$Inventory->saveMany($newInventories);

		$variantParams = $Model->Variant->getStockVariantParams($productIds);

		$variants = $Model->Variant->find('all', $variantParams);

		// Group in a way that makes it easier to process
		$products = [];
		foreach ($variants as $variant) {
			$productId = $variant['Product']['id'];
			$variantId = $variant['Variant']['id'];
			$optionGroupId = $variant['OptionGroup']['id'];
			$optionId = $variant['Option']['id'];

			if (empty($products[$productId])) {
				$products[$productId] = $variant['Product'];
			}

			if (empty($products[$productId]['Variant'][$variantId])) {
				$products[$productId]['Variant'][$variantId] = $variant['Variant'];
				$products[$productId]['Variant'][$variantId]['Inventory'] = $variant['Inventory'];
			}

			if (empty($products[$productId]['Variant'][$variantId]['OptionGroup'][$optionGroupId])) {
				$products[$productId]['Variant'][$variantId]['OptionGroup'][$optionGroupId] = $variant['OptionGroup'];
			}
			$products[$productId]['Variant'][$variantId]['OptionGroup'][$optionGroupId]['Option'] = $variant['Option'];
		}

		// Rebuild the array in a format best suited for rendering
		$productStock = [];
		foreach ($products as $key => $product) {
			$allOptionGroups = Hash::combine($product['Variant'], '{n}.OptionGroup.{n}.id', '{n}.OptionGroup.{n}');

			// Sort by sequence
			usort($allOptionGroups, function ($a, $b) {
				return $a['sequence'] - $b['sequence'];
			});

			$allOptionGroups = array_values($allOptionGroups);

			// Get all options for each option group for table headings
			foreach ($allOptionGroups as $ogKey => $optionGroup) {
				$allOptionGroups[$ogKey]['Option'] = Hash::combine($product, 'Variant.{n}.OptionGroup.' . $optionGroup['id'] . '.Option.id', 'Variant.{n}.OptionGroup.' . $optionGroup['id'] . '.Option');
			}

			// If there's more than 2 options groups split the products into separate tables.
			// The separated product names and their option ids will be stored in the split field
			$tableOptionGroups = $allOptionGroups;
			$numOptionGroups = count($allOptionGroups);
			$splitProducts = [];
			while ($numOptionGroups > 2) {
				$groupToSplit = array_shift($tableOptionGroups);
				if (empty($splitProducts)) {
					foreach ($groupToSplit['Option'] as $option) {
						$splitProduct = [
							'name' => $product['name'] . ' (' . $groupToSplit['name'] . ': ' . $option['name'] . ')',
							'ids_prefix' => $option['id']
						];
						$splitProducts[] = $splitProduct;
					}
				} else {
					$newSplitProducts = [];
					foreach ($splitProducts as $splitProduct) {
						foreach ($groupToSplit['Option'] as $option) {
							$newSplitProduct = $splitProduct;
							$newSplitProduct['name'] .= ' (' . $groupToSplit['name'] . ': ' . $option['name'] . ')';
							$newSplitProduct['ids_prefix'] .= '.' . $option['id'];
							$newSplitProducts[] = $newSplitProduct;
						}
					}
					$splitProducts = $newSplitProducts;
				}
				$numOptionGroups--;
			}

			// Key variants by their option ids to make them easier to search when rendering
			$productVariants = [];
			foreach ($product['Variant'] as $vKey => $variant) {
				$selectionIds = [];
				if (!empty($allOptionGroups)) {
					foreach ($allOptionGroups as $optionGroup) {
						$selectionIds[] = $variant['OptionGroup'][(int)$optionGroup['id']]['Option']['id'];
					}
				} else {
					$selectionIds[] = 0;
				}
				$productVariants[implode('.', $selectionIds)] = [
					'name' => $variant['name'],
					'Inventory' => $variant['Inventory'],
				];
			}

			$productStock[] = array_merge($product, array(
				'Variant' => $productVariants,
				'TableOptionGroup' => $tableOptionGroups,
				'Split' => $splitProducts
			));
		}

		// Set this to pre-populate the stock fields
		$this->request->data['Inventory'] = Hash::combine($variants, '{n}.Inventory.id', '{n}.Inventory');

		$this->set('products', $productStock);
		$this->view = 'EvShop.admin_stock';
	}

}
