<?php

App::uses('AppComponent', 'Controller/Component');

class ProductsComponent extends AppComponent {

/**
 * Returns paginated, or none paginated, array of products to show on the
 * app template
 *
 * @param array $params The base paginator settings array to query against
 * @param bool $isPaginated Defines whether to process a paginated query
 * @param array $filterOptionParams An array of filter option selects to query against
 * @return array Contains array of results based on the supplied params and filter selections
 */
	public function listing($params = [], $isPaginated = true, $filterOptionParams = null) {
		//Initialise the product model to find/paginate on
		$Product = EvClassRegistry::init('EvShop.Product');

		//Get the database source so we can use it to build sub-queries
		$database = $Product->getDataSource();

		//get the original virtual fields so they can be set back to normal after listing
		$productVirtualFields = $Product->virtualFields;

		//Add virtual fields to the product listing. By default, price and max_price are added
		$this->_addVirtualFieldsToProductListing($Product, $filterOptionParams);

		//Get the variant sub-query. By Default it returns all variants that match filter options ordered by price
		$variantSubQuery = $this->_getVariantSubQueryForProductListing(
			$database,
			$filterOptionParams
		);

		$defaults = [
			'fields' => [
				'Product.*',
				'Variant.id',
				'Variant.product_id',
				'Variant.name',
				'Variant.sku',
				'Variant.gtin',
				'Variant.lowest_price',
				'Variant.rrp_price',
				'Variant.was_price',
			],
			'joins' => [
				[
					'table' => '(' . $variantSubQuery . ')',
					'alias' => 'Variant',
					'conditions' => [
						'Variant.product_id = Product.id'
					],
				],
			],
			'group' => 'Product.id',
			'conditions' => [
				'Product.is_active' => true
			],
			'contain' => [
				'ListingImage'
			]
		];

		// only include variant imagery when assigned in the plugin config
		if (
			! Configure::check('EvShop.hasVariantImagery') ||
			Configure::read('EvShop.hasVariantImagery') === true
		) {
			$defaults = $this->_addVariantImagesToListing(
				$Product,
				$database,
				$defaults,
				$filterOptionParams
			);
		}

		$defaults = $this->_addAdditionalParamsToListing(
			$Product,
			$defaults,
			$filterOptionParams
		);

		$params = array_merge_recursive($defaults, $params);

		if (!empty($params['order']) && is_array($params['order'])) {
			$params['order'] = implode(', ', $params['order']);
		}

		$behaviors = $Product->Behaviors->loaded();

		// remove the meta behaviour to prevent it being
		// called on each of the product listing items
		if (is_array($behaviors) && in_array('Meta', $behaviors)) {
			$Product->removeBehavior('Meta');
		}

		if ($isPaginated && ! empty($this->_controller)) {
			$this->_controller->Paginator->settings = $params;
			$result = $this->_controller->Paginator->paginate(
				$Product
			);
		} else {
			$result = $Product->find('all', $params);
		}

		if (
			! Configure::check('EvShop.hasVariantImagery') ||
			Configure::read('EvShop.hasVariantImagery') === true
		) {
			$result = $this->_containVariantImagesInListing($result, $filterOptionParams);
		}

		//Setting virtual fields back to default
		$Product->virtualFields = $productVirtualFields;

		// Reload Meta's behavior
		$Product->Behaviors->load('Meta');

		return $result;
	}

/**
 * Get a featured product listing
 *
 * @param array $params   Query / paginate params
 * @param bool  $paginate Paginate results or not?
 * @return array Array of results
 */
	public function featured($params = array(), $paginate = false) {
		$params = Hash::merge(
			array(
				'conditions' => array(
					'Product.is_featured' => 1
				)
			),
			$params
		);

		return $this->listing($params, $paginate);
	}

/**
 * Returns a listing of product that match the supplied category id and filter options
 *
 * @param int|array $categoryId Category ID or array of category IDs
 * @param array $params query / paginate params
 * @param bool $isPaginated Paginate results or not?
 * @param array $filterOptionParams An array of filter option selects to query against
 * @return array Array of results
 */
	public function listingByCategory(
		$categoryId = null,
		$params = [],
		$isPaginated = true,
		$filterOptionParams = null
	) {
		// add product category based params to the listing query
		$params['joins'][] = [
			'table' => 'ev_shop_categories_products',
			'alias' => 'CategoriesProduct',
			'conditions' => [
				'CategoriesProduct.product_id = Product.id'
			]
		];

		$params['conditions']['CategoriesProduct.category_id'] = $categoryId;

		return $this->listing($params, $isPaginated, $filterOptionParams);
	}

/**
 * Returns a listing of product that match the supplied brand id and filter options
 *
 * @param int $brandId The unique brand id to query against
 * @param array $params query / paginate params
 * @param bool $isPaginated Paginate results or not?
 * @param array $filterOptionParams An array of filter option selects to query against
 * @return array Array of results
 */
	public function listingByBrand(
		$brandId = null,
		$params = [],
		$isPaginated = true,
		$filterOptionParams = null
	) {
		$defaults = array(
			'conditions' => array(
				'Product.brand_id' => $brandId
			)
		);

		// if it's many brands, and the group by and
		// also contain the brands so we know which is which
		if (is_array($brandId)) {
			$defaults['contain'][] = 'Brand';
		}

		$params = Hash::merge(
			$defaults,
			$params
		);

		return $this->listing($params, $isPaginated, $filterOptionParams);
	}

/**
 * Add virtual fields to the query model to attach data that isn't included in the Model.
 * Used by default to assign the price of the cheapest variant to it's product.
 *
 * @param model &$Model    The model that is being queried on that the virtual fields will be assigned to.
 * @param array $orderData An array containing any filters or sorts that are to be applied to the query.
 * @return void.
 */
	protected function _addVirtualFieldsToProductListing(&$Model, $orderData) {
		$Model->virtualFields['price'] = 'Variant.lowest_price';
		$Model->virtualFields['max_price'] = 'MAX(Variant.lowest_price)';
		$Model->virtualFields['was_price_ex_vat'] = 'Variant.was_price';
		$Model->virtualFields['rrp_price_ex_vat'] = 'Variant.rrp_price';
	}

/**
 * A default sub-query included in the product listing that returns all the variants ordered by their price,
 * sale price is used if available. Uses the "_addAdditionalParamsToVariantSubQuery" to filter variants by
 * their options.
 *
 * @param object $DBSource  The database source of the project, used to create the sub-query.
 * @param array  $orderData An array containing the relevant data to filter and sort the variants by.
 * @return string The sql statement that generates the table for the sub-query.
 */
	protected function _getVariantSubQueryForProductListing($DBSource, $orderData) {
		//Get the variant model to build a query on it
		$Variant = EvClassRegistry::init('EvShop.Variant');

		$variantMinPriceJoin = [
			'alias' => 'Variant',
			'table' => 'ev_shop_variants',
			'conditions' => [
				'Variant.id = VariantPricing.variant_id',
				'Variant.is_active' => 1,
			]
		];

		if (isset($orderData['productIds'])) {
			// If we specify a list of products, we only need to find minimum prices for those products.
			// This will also speed up the main variant query as this is the first table linked against so
			// it will discard any product not in this list
			$variantMinPriceJoin['conditions']['Variant.product_id'] = $orderData['productIds'];
		}

		if (isset($orderData['variantIds'])) {
			$variantMinPriceJoin['conditions']['Variant.id'] = $orderData['variantIds'];
		}

		// This creates another lookup table inside the subquery that matches products to their lowest price
		// We can use this to only fetch the variants with the lowest price.
		$productMinPriceSubQueryParams = [
			'table' => 'ev_shop_variant_pricings',
			'alias' => 'VariantPricing',
			'fields' => [
				'Variant.product_id',
				'MIN(VariantPricing.lowest_price) as min_price'
			],
			'joins' => [
				$variantMinPriceJoin
			],
			'group' => 'Variant.product_id'
		];

		if (CakePlugin::loaded('EvCurrency')) {
			$productMinPriceSubQueryParams['conditions']['VariantPricing.currency_id'] = CakeSession::read('EvCurrency.currencyId');
		}

		//Add any extra filtering or ordering to variants
		$productMinPriceSubQueryParams = $this->_addAdditionalParamsToVariantSubQuery($productMinPriceSubQueryParams, $orderData);

		$productMinPriceSubQuery = $Variant->getDataSource()->buildStatement( $productMinPriceSubQueryParams, $Variant );

		//Set up the basic parms needed for the variant sub-query
		//Basic params join variant pricing and order by price
		$variantSubQueryParams = [
			'fields' => [
				'Variant.*',
				'VariantPricing.lowest_price AS lowest_price',
				'VariantPricing.rrp AS rrp_price',
				'IF(VariantPricing.sale_price < VariantPricing.price, VariantPricing.price, VariantPricing.sale_price) AS was_price'
			],
			'table' => $DBSource->fullTableName($Variant),
			'alias' => 'Variant',
			'joins' => [
				[
					'table' => '(' . $productMinPriceSubQuery . ')',
					'alias' => 'ProductMinPrice',
					'conditions' => [
						'ProductMinPrice.product_id = Variant.product_id'
					],
				],
				[
					'table' => 'ev_shop_variant_pricings',
					'alias' => 'VariantPricing',
					'conditions' => [
						'VariantPricing.variant_id = Variant.id',
						'VariantPricing.lowest_price = ProductMinPrice.min_price'
					],
				],
			],
			'conditions' => [
				'Variant.is_active' => true
			],
			'group' => 'Variant.id',
			'order' => 'VariantPricing.lowest_price ASC'
		];

		//Add any extra filtering or ordering to variants wrapper
		$variantSubQueryParams = $this->_addAdditionalParamsToVariantSubQueryWrapper($variantSubQueryParams, $orderData);

		//Build the variant sub-query
		return $DBSource->buildStatement($variantSubQueryParams, $Variant);
	}

/**
 * Add extra parameters that aren't part of the default variant pricing sub-query. Override this if you need to make
 * changes/additions to the default variant pricing sub query.
 *
 * @param array $currentParams An array containing the current parameters used for the variant sub-query.
 * @param array $orderData     An array containing data passed to be used for adding relevant parameters.
 * @return array The modified query parameters.
 */
	protected function _addAdditionalParamsToVariantSubQuery($currentParams, $orderData) {
		if (isset($orderData['option']) && !empty($orderData['option'])) {
			$currentParams = $this->_addOptionFiltersToVariantSubQuery($currentParams, $orderData['option']);
		}

		if (CakePlugin::loaded('EvCurrency')) {
			$currentParams['conditions']['VariantPricing.currency_id'] = CakeSession::read('EvCurrency.currencyId');
		}

		if (isset($orderData['variantIds'])) {
			$currentParams['conditions']['Variant.id'] = $orderData['variantIds'];
		}

		return $currentParams;
	}

/**
 * Add extra parameters that aren't part of the default variant sub-query. Override this if you need to make
 * changes/additions to the default variant sub query.
 *
 * @param array $currentParams An array containing the current parameters used for the variant sub-query.
 * @param array $orderData     An array containing data passed to be used for adding relevant parameters.
 * @return array The modified query parameters.
 */
	protected function _addAdditionalParamsToVariantSubQueryWrapper($currentParams, $orderData) {
		if (isset($orderData['variantIds'])) {
			$currentParams['conditions']['Variant.id'] = $orderData['variantIds'];
		}

		return $currentParams;
	}

/**
 * Adds a specific set of parameters to the variant sub-query to filter out variants by there options.
 * The options to filter by are provided in the orderOptions variable listed under their optionGroup.
 * Multiple joins are created for each option group to enforce that each variant meets the criteria.
 * Joins are named with the option group included to prevent clashes.
 *
 * @param array $currentParams An array containing the current parameters used for the variant sub-query.
 * @param array $orderOptions  An array containing data passed to be used for adding relevant parameters.
 * @return array The modified query parameters.
 */
	protected function _addOptionFiltersToVariantSubQuery($currentParams, $orderOptions) {
		//Foreach option group we need to join on a separate set of option_variants and options, aliased with the
		//name of the option group to prevent clashes
		foreach ($orderOptions as $optionGroup => $options) {

			$groupName = preg_replace('/[^A-Za-z0-9]/', '', $optionGroup);

			//Add the option_variant join
			$currentParams['joins'][] = [
				'table' => 'ev_shop_options_variants',
				'alias' => 'OptionVariantFor' . $groupName,
				'conditions' => [
					'OptionVariantFor' . $groupName . '.variant_id = Variant.id'
				]
			];

			//Add the option join
			$currentParams['joins'][] = [
				'table' => 'ev_shop_options',
				'alias' => 'OptionFor' . $groupName,
				'conditions' => [
					'OptionFor' . $groupName . '.id = OptionVariantFor' . $groupName . '.option_id',
					'OptionFor' . $groupName . '.name' => $options
				]
			];

			// Make sure we only have the options in the correct group
			$currentParams['joins'][] = [
				'table' => 'ev_shop_option_groups',
				'alias' => 'OptionGroup' . $groupName,
				'conditions' => [
					'OptionGroup' . $groupName . '.id = OptionFor' . $groupName . '.option_group_id',
					'OptionGroup' . $groupName . '.name' => $optionGroup
				]
			];
		}

		return $currentParams;
	}

/**
 * Add fields and joins to the current listing query parameters to include variant images to the listing data.
 *
 * @param obj   $Product       The product model that will be queried on.
 * @param obj   $DBSource      The database source of the product model.
 * @param array $currentParams The current query parameters.
 * @param array $orderData     An array containing data passed to be used for adding relevant parameters.
 * @return array The current params with the variant image query parameters added.
 */
	protected function _addVariantImagesToListing($Product, $DBSource, $currentParams, $orderData) {
		$variantImagesSubQuery = $this->_getVariantImageSubQueryForProductListing(
			$DBSource,
			$orderData
		);

		$currentParams['fields'][] = 'Image.*';

		$currentParams['joins'][] = [
			'table' => '(' . $variantImagesSubQuery . ')',
			'alias' => 'Image',
			'type' => 'left',
			'conditions' => [
				'Image.model_id = Product.id'
			]
		];

		return $currentParams;
	}

/**
 * A default sub-query that attempts to find a variant image that fulfills the most criteria
 * provided in the orderData.
 *
 * @param object $DBSource  The database source of the project, used to create the sub-query.
 * @param array  $orderData An array containing the relevant data to filter and sort the variants by.
 * @return string The sql statement that generates the table for the sub-query.
 */
	protected function _getVariantImageSubQueryForProductListing($DBSource, $orderData) {
		//Start building the Image sub-query by initialising the Image Model
		$Image = EvClassRegistry::init('EvCore.Image');
		$Product = EvClassRegistry::init('EvShop.Product');

		// Only fetch the essential fields here - we will match them up to full image array later on when the list has been reduced
		$variantImageSubQueryParams = [
			'fields' => [
				'ImageInner.id',
				'ImageInner.model_id',
			],
			'conditions' => [
				'ImageInner.model' => $Product->alias
			],
			'table' => $DBSource->fullTableName($Image),
			'alias' => 'ImageInner',
			'group' => 'ImageInner.id',
			'order' => 'ImageInner.model_id ASC'
		];

		$variantImageSubQueryParams = $this->_addAdditionalParamsToVariantImageSubQuery($variantImageSubQueryParams, $orderData);

		//Build the variant image sub-query
		$mainSubQuery = $DBSource->buildStatement($variantImageSubQueryParams, $Image);

		//The variant image subquery is wrapped in a subquery so that the images can be grouped by the product id. This leaves
		//a single image for each product so when the model_id and product id is matched, only a single image will be selected
		//to returned as part of the image. Without this wrapper, multiple images have the potential to be returned, which
		//are selected randomly so there isn't any control on which image appears.
		$imageWrapperQuery = [
			'fields' => [
				'Image.id',
				'Image.model_id',
			],
			'table' => '(' . $mainSubQuery . ')',
			'alias' => 'Image',
			'group' => 'Image.model_id',
		];

		return $DBSource->buildStatement($imageWrapperQuery, $Image);
	}

/**
 * Add extra parameters that aren't part of the default variant image sub-query. Override this if you need to make
 * changes/additions to the default variant image sub query.
 *
 * @param array $currentParams An array containing the current parameters used for the variant image sub-query.
 * @param array $orderData     An array containing data passed to be used for adding relevant parameters.
 * @return array The modified query parameters.
 */
	protected function _addAdditionalParamsToVariantImageSubQuery($currentParams, $orderData) {
		if (isset($orderData['option']) && !empty($orderData['option'])) {
			$currentParams = $this->_addOptionFiltersToVariantImageSubQuery($currentParams, $orderData['option']);
		}

		return $currentParams;
	}

/**
 * Add the filtering joins to the variant image sub-query so that only images that meet the criteria provided
 * in the orderOptions are chosen. If multiple images are found that meet the criteria, the image that contains
 * the most selected options is selected.
 *
 * @param array $currentParams An array containing the current parameters used for the variant image sub-query.
 * @param array $orderOptions  An array containing data passed to be used for adding relevant parameters.
 * @return array The modified query parameters.
 */
	protected function _addOptionFiltersToVariantImageSubQuery($currentParams, $orderOptions) {
		//Add the joins to check by option
		$currentParams['joins'][] = [
			'table' => 'ev_shop_variant_image_options',
			'alias' => 'VariantImageOption',
			'conditions' => [
				'VariantImageOption.image_id = ImageInner.id'
			]
		];

		$imageOptionJoin = [
			'table' => 'ev_shop_options',
			'alias' => 'Option',
			'conditions' => [
				'Option.id = VariantImageOption.option_id'
			]
		];

		$variantImageFilters = [];
		foreach ($orderOptions as $optionGroup => $options) {
			//Options don't need to be separated by group so chuck them all in
			$variantImageFilters = Hash::merge($variantImageFilters, $options);
		}

		$imageOptionJoin['conditions']['Option.name'] = $variantImageFilters;

		$currentParams['joins'][] = $imageOptionJoin;

		$currentParams['order'] = 'ImageInner.model_id ASC, COUNT(Option.option_group_id) DESC, ImageInner.sequence ASC';

		return $currentParams;
	}

/**
 * Variant images only have their id returned from the initial listing. Find the full image data from the image id and
 * contain it in the product data.
 *
 * @param array $products The listing of products.
 * @param array $filterOptionParams An array of filter option selects to query against
 * @return array The modified product listing.
 */
	protected function _containVariantImagesInListing($products, $filterOptionParams) {
		// Find the images and match them up
		$Image = EvClassRegistry::init('EvCore.Image');
		$imageIds = Hash::extract($products, '{n}.Image.id');

		$images = $Image->find('all', [
			'conditions' => [
				'Image.id' => $imageIds
			]
		]);
		$images = Hash::combine($images, '{n}.Image.id', '{n}.Image');

		foreach ($products as &$product) {
			if (!empty($product['Image']['id'])) {
				$product['Image'] = $images[$product['Image']['id']];
			}
		}

		return $products;
	}

/**
 * Check if specific ordering data is provided and if so call the function that adds the correct parameters to the
 * query. Extend to add add your own parameters to the listing query.
 *
 * @param array &$queryModel   The current parameters of the query to change. Normally the defaults set in listing().
 * @param array $currentParams The ordering data (direction and currencyId if it is set).
 * @param Model $orderData     The model that is being queried on, listing uses Product.
 * @return array The modified query parameters.
 */
	protected function _addAdditionalParamsToListing(&$queryModel, $currentParams, $orderData) {
		//Go through the passed ordering parameters and add it to the pagination query
		if (isset($orderData['price']) && !empty($orderData['price'])) {
			$currentParams = $this->_addPriceParamsToListing($currentParams, $orderData['price'], $queryModel);
		}

		if (isset($orderData['brand']) && !empty($orderData['brand'])) {
			$currentParams = $this->_addBrandParamsToListing($currentParams, $orderData['brand'], $queryModel);
		}

		if (!empty(Configure::read('EvShop.filterOnAttributes')) && !empty($orderData['attribute'])) {
			$currentParams = $this->_addAttributeParamsToListing($currentParams, $orderData['attribute'], $queryModel);
		}

		return $currentParams;
	}

/**
 * Adds joins and order to the pagination query parameters. Defaults to ordering
 * in ascending order. The ordering will chose the lowest price between the price and sale_price
 * to order by.
 *
 * @param array $currentParams The current parameters of the query to change. Normally the defaults set in listing().
 * @param array $orderPrice    The ordering data (direction and currencyId if it is set).
 * @param Model &$queryModel   The model that is being queried on, listing uses Product.
 * @return array The modified query parameters.
 */
	protected function _addPriceParamsToListing($currentParams, $orderPrice, &$queryModel) {
		$orderDirection = 'ASC';
		if (isset($orderPrice['direction'])) {
			$orderDirection = $orderPrice['direction'];
		}

		$currentParams['order'] = "Product.price $orderDirection";

		return $currentParams;
	}

/**
 * Adds joins and conditions to the listing params so that products are selected only if they belong to a single
 * brand. Brands are provided throught the order data as a list of the ids that the products need to match to.
 *
 * @param array $currentParams The current parameters of the query to change. Normally the defaults set in listing().
 * @param array $orderBrands   The ordering data (list of brand ids).
 * @param Model &$queryModel   The model that is being queried on, listing uses Product.
 * @return array The modified query parameters.
 */
	protected function _addBrandParamsToListing($currentParams, $orderBrands, &$queryModel) {
		$currentParams['joins'][] = [
			'table' => 'ev_shop_brands',
			'alias' => 'FilteredBrand',
			'conditions' => [
				'FilteredBrand.id = Product.brand_id'
			]
		];

		if (Configure::read('EvShop.showSubBrandProducts')) {
			$orderBrands = EvClassRegistry::init('EvShop.Brand')->appendChildren($orderBrands);
		}

		$currentParams['conditions']['FilteredBrand.name'] = $orderBrands;

		return $currentParams;
	}

/**
 * Adds joins and conditions to the listing params so that products are selected only if they have the given attributes.
 *
 * @param array $currentParams   The current parameters of the query to change. Normally the defaults set in listing().
 * @param array $orderAttributes The ordering data (list of attributes with values).
 * @param Model &$queryModel     The model that is being queried on, listing uses Product.
 * @return array The modified query parameters.
 */
	protected function _addAttributeParamsToListing($currentParams, $orderAttributes, &$queryModel) {
		if (!empty($orderAttributes)) {
			foreach ($orderAttributes as $groupName => $attributeNames) {

				$safeGroupName = preg_replace('/[^A-Za-z0-9]/', '', $groupName);

				$currentParams['joins'][] = [
					'table' => 'ev_shop_product_attributes_products',
					'alias' => 'FilteredAttributesProductsFor' . $safeGroupName,
					'conditions' => [
						'FilteredAttributesProductsFor' . $safeGroupName . '.product_id = Product.id'
					]
				];
				$currentParams['joins'][] = [
					'table' => 'ev_shop_product_attributes',
					'alias' => 'FilteredAttributeFor' . $safeGroupName,
					'conditions' => [
						'FilteredAttributesProductsFor' . $safeGroupName . '.product_attribute_id = FilteredAttributeFor' . $safeGroupName . '.id',
						'FilteredAttributeFor' . $safeGroupName . '.name' => $attributeNames,

					]
				];
				$currentParams['joins'][] = [
					'table' => 'ev_shop_product_attribute_groups',
					'alias' => 'FilteredAttributeGroupFor' . $safeGroupName,
					'conditions' => [
						'FilteredAttributeGroupFor' . $safeGroupName . '.id = FilteredAttributeFor' . $safeGroupName . '.product_attribute_group_id',
						'FilteredAttributeGroupFor' . $safeGroupName . '.name' => $groupName
					]
				];
			}
		}

		return $currentParams;
	}

/**
 * Adds the _incTax fields to the query results. Checks that the EvTax plugin is loaded if not then
 * it is assumed that the tax will be added manually from whatever peeps be using.
 *
 * @param array $products The results of the product query containing each product to add tax to.
 * @return array products with tax added to their prices.
 */
	protected function _addTaxToProducts($products) {
		if (CakePlugin::loaded('EvTax')) {
			$TaxLevelModel = EvClassRegistry::init('EvTax.TaxLevel');
			$VariantPricingModel = EvClassRegistry::init('EvShop.VariantPricing');

			if (! empty($products)) {
				foreach ($products as $key => $item) {
					$taxLevel = array();
					if (isset($item['Product']['tax_level_id']) && $item['Product']['tax_level_id'] > 0) {

						$taxLevel = $TaxLevelModel->find('first', array(
							'conditions' => array(
								'id' => $item['Product']['tax_level_id']
							)
						));

						if (!empty($taxLevel)) {
							$taxRate = $taxLevel['TaxLevel']['rate'];
							if (isset($item['Product']['price']) && !empty($item['Product']['price'])) {
								$priceIncTax = $VariantPricingModel->addTaxToPrice($item['Product']['price'], $taxRate);
								$products[$key]['Product']['price_incTax'] = $priceIncTax;
							}

							if (isset($item['Product']['max_price']) && !empty($item['Product']['max_price'])) {
								$priceIncTax = $VariantPricingModel->addTaxToPrice($item['Product']['max_price'], $taxRate);
								$products[$key]['Product']['max_price_incTax'] = $priceIncTax;
							}
						}
					}
				}
			}
		}

		return $products;
	}

/**
 * Get a set of options to filter listing pages by. By default all the options will be retrieved, but if you want to limit
 * selection by category or featured or something else then you can pass extra find parameters in.
 *
 * This will also use product attributes if they are enabled in the config.
 *
 * @param array $extraParams            Parameters that will be passed into the main find that gets the options.
 *                                      Variants, options and option groups are joined by default so any of their
 *                                      fields can be accessed.
 * @param bool  $includeBrands          Set to true if you want to include a list of brands to filter on. Retrieved
 *                                      brands are based on the products that are found, so if you are limiting filters
 *                                      to categories or featured then only brands that match those products will be
 *                                      returned.
 * @param array $brandParams            Parameters that will be passed into the brand find. Used to limit brands
 *                                      further than the matched products or if other data is required.
 * @param bool  $includeCategories      True if the categories should be included as part of the filters. Defaults to
 *                                      false.
 * @param arram $productCategoryParams  Additional query parameters to use when included the categories.
 * @param array $extraOptionGroupParams Additional query parameters to use when getting the option groups for filters.
 * @return array An array containing each filter group and the options for each group and any further
 *               data added to options.
 */
	public function getOptionFiltersForListing(
		$extraParams = [],
		$includeBrands = false,
		$brandParams = [],
		$includeCategories = false,
		$productCategoryParams = [],
		$extraOptionGroupParams = []
	) {
		$optionGroups = $this->_addOptionsToOptionsForListing(
			$extraParams,
			$extraOptionGroupParams
		);

		if (!empty(Configure::read('EvShop.filterOnAttributes'))) {
			$optionGroups = $this->_addAttributesToOptionsForListing($optionGroups, $extraParams);
		}

		$optionGroups = $this->_addExtraOptionGroupsToOptionsForListing(
			$optionGroups,
			$extraParams
		);

		if ($includeBrands) {
			$brandOptions = $this->_addBrandsToOptionsForListing([], $brandParams);

			if (!empty($brandOptions['Option'])) {
				$optionGroups[] = $brandOptions;
			}
		}

		// add product categories to the product index filter options
		if ($includeCategories) {
			$productIds = Hash::extract($productOptions, '{n}.Product.id');
			$optionGroups[] = $this->_addProductCategoriesToOptionsForListing(
				$productIds
			);
		}

		//Tidy up the array and sort
		$optionGroups = array_values($optionGroups);
		$optionGroups = Hash::sort($optionGroups, '{n}.name');

		return $optionGroups;
	}

/**
 * Get the options for a filter to be used on a listing. The options are formatted into option groups.
 *
 * @param array $extraParams            Array of custom query parameters.
 * @param array $extraOptionGroupParams Array of custom query parameters for the option groups.
 * @return array Option groups.
 */
	protected function _addOptionsToOptionsForListing($extraParams, $extraOptionGroupParams) {
		$Product = EvClassRegistry::init('EvShop.Product');

		$OptionGroup = EvClassRegistry::init('EvShop.OptionGroup');
		$optionGroupsList = Cache::remember('option_groups', function () use ($OptionGroup, $extraParams, $extraOptionGroupParams) {
			$defaultOptionGroupParams = [
				'conditions' => [
					'OptionGroup.on_front_end' => true
				],
				'fields' => [
					'OptionGroup.id',
					'OptionGroup.name',
					'OptionGroup.tooltip'
				]
			];

			$optionGroupParams = $OptionGroup->mergeQueryParams(
				$defaultOptionGroupParams,
				$extraOptionGroupParams
			);

			if (!empty($extraParams[$OptionGroup->alias])) {
				$optionGroupParams = $OptionGroup->mergeQueryParams(
					$optionGroupParams,
					$extraParams[$OptionGroup->alias]
				);
			}

			$optionGroups = $OptionGroup->find('all', $optionGroupParams);

			return Hash::combine($optionGroups, '{n}.OptionGroup.id', '{n}.OptionGroup');
		});

		$defaultParams = [
			'joins' => [
				[
					'table' => 'ev_shop_variants',
					'alias' => 'Variant',
					'conditions' => [
						'Variant.product_id = Product.id',
						'Variant.is_active' => true,
					]
				],
				[
					'table' => 'ev_shop_options_variants',
					'alias' => 'OptionVariant',
					'conditions' => [
						'OptionVariant.variant_id = Variant.id'
					]
				],
				[
					'table' => 'ev_shop_options',
					'alias' => 'Option',
					'conditions' => [
						'Option.id = OptionVariant.option_id',
						'Option.option_group_id' => array_keys($optionGroupsList)
					]
				],
			],
			'conditions' => [
				'Product.is_active' => true,
			],
			'fields' => [
				'Option.*',
				'Product.id',
			],
			'order' => 'Option.sequence, Option.name',
			'group' => 'Option.id'
		];

		$params = array_merge_recursive($defaultParams, $extraParams);

		$productOptions = $Product->find('all', $params);

		$optionGroups = [];

		// Get unique options
		$options = Hash::combine($productOptions, '{n}.Option.id', '{n}.Option');

		//Loop through each option and add it to it's corresponding option group.
		foreach ($options as $option) {
			$optionGroupId = $option['option_group_id'];
			$option = $this->_addExtraDataToOptionsForListing($option);

			if (!isset($optionGroups[$optionGroupId])) {
				$newOptionGroup = [
					'name' => $optionGroupsList[$optionGroupId]['name'],
					'option_group_id' => $optionGroupId,
					'tooltip' => $optionGroupsList[$optionGroupId]['tooltip'],
				];

				$newOptionGroup = $this->_addExtraDataToOptionGroupsForListing(
					$optionGroupsList[$optionGroupId],
					$newOptionGroup
				);

				$optionGroups[$optionGroupId] = $newOptionGroup;
			}

			$optionGroups[$optionGroupId]['Option'][] = $option;
		}

		return $optionGroups;
	}

/**
 * Add any additional data to each option found from the main find. This method is called from inside a loop
 * so every option will obtain any data added here.
 *
 * @param array $option The returned data from the database of the current option.
 * @return array        The option array with any added or manipulated data.
 */
	protected function _addExtraDataToOptionsForListing($option) {
		return $option;
	}

/**
 * Add any additional data to each option group found from the main find. This method is called from inside a loop
 * so every option group will obtain any data added here.
 *
 * @param array $optionGroup        The returned data from the database of the current option group. Will also contain
 *                                  the options that belong to this option group.
 * @param array $currentOptionGroup The current option group that is being added to the listing.
 * @return array The option group array with any added or manipulated data.
 */
	protected function _addExtraDataToOptionGroupsForListing($optionGroup, $currentOptionGroup) {
		return $currentOptionGroup;
	}

/**
 * Add any additional option groups to the filters. Any options that may not have a database object could be added
 * here or if you need options from a non ev shop table. This method could also be used to structure the data
 * differently if you need to.
 *
 * @param array $optionGroups The current set of option groups and their options.
 * @param array $extraParams  Extra query parameters to use when finding extra option groups to add to the listing.
 * @return 					  The modified set of option groups and their options.
 */
	protected function _addExtraOptionGroupsToOptionsForListing($optionGroups, $extraParams) {
		return $optionGroups;
	}

/**
 * Get the brands that match the current products and create an option group for them so they can be added
 * to the option group filters.
 *
 * @param array $productIds A list of products ids that will be used to find matching brands
 * @param array $params     Additional params passed through from the initial getOptionFiltersForListing
 *                          call.
 * @return A filter group containing the brands as options to be added to the option groups
 */
	protected function _addBrandsToOptionsForListing($productIds, $params = []) {
		$Product = EvClassRegistry::init('EvShop.Product');

		$brands = [
			'name' => 'Brands'
		];

		$defaultParams = [
			'joins' => [
				[
					'table' => 'ev_shop_brands',
					'alias' => 'Brand',
					'conditions' => [
						'Brand.id = Product.brand_id',
					]
				]
			],
			'fields' => [
				'Brand.*',
			],
			'conditions' => [
				'Brand.is_active' => true,
			],
			'group' => 'Brand.id'
		];

		if (Configure::read('EvShop.brandsHaveParents') == true) {
			$defaultParams['joins'][] = [
				'table' => 'ev_shop_brands',
				'alias' => 'ParentBrand',
				'type' => 'LEFT',
				'conditions' => [
					'ParentBrand.id = Brand.parent_id',
				]
			];
			$defaultParams['fields'][] = 'ParentBrand.*';
			$defaultParams['conditions'][] = [
				'OR' => [
					'ParentBrand.id IS NULL',
					'ParentBrand.is_active' => true
				]
			];
		} else {
			// Only fetch parent brands
			$defaultParams['joins'][0]['conditions']['OR'] = [
				'Brand.parent_id IS NULL',
				['Brand.parent_id' => ''],
				['Brand.parent_id' => '0']
			];
		}

		$params = array_merge_recursive($defaultParams, $params);

		$optionBrands = $Product->find('all', $params);

		if (empty($optionBrands)) {
			return $brands;
		}

		// Sort brands so child brands appear under the parents.
		// Parent brands will be created first...
		foreach ($optionBrands as $optionBrand) {
			if (empty($optionBrand['ParentBrand']['id'])) {
				$brands['Option'][$optionBrand['Brand']['id']] = $this->_addExtraDataToBrandForListing($optionBrand['Brand']);
			} elseif (empty($brands['Option'][$optionBrand['ParentBrand']['id']])) {
				$brands['Option'][$optionBrand['ParentBrand']['id']] = $this->_addExtraDataToBrandForListing($optionBrand['ParentBrand']);
			}
		}
		if (Configure::read('EvShop.brandsHaveParents') == true) {
			// ...then children added
			foreach ($optionBrands as $optionBrand) {
				if (!empty($optionBrand['ParentBrand']['id'])) {
					$brands['Option'][$optionBrand['ParentBrand']['id']]['children'][] = $optionBrand['Brand']['name'];
				}
			}
		}

		// Convert back to an indexed array instead of a keyed array for backwards compatability
		$brands['Option'] = array_values($brands['Option']);
		$this->_addExtraDataToBrandGroupForListing($brands);

		return $brands;
	}

/**
 * Add any extra data to each brand option. This method is called in a loop so any data added here will be added
 * to every brand option.
 *
 * @param array $currentBrand The current brand array containing data from the database.
 * @return array The current brand with any additional of modified data.
 */
	protected function _addExtraDataToBrandForListing($currentBrand) {
		return $currentBrand;
	}

/**
 * Add any extra data to the brand option group.
 *
 * @param array $currentBrandGroup The current brand filter group. Also contains the brand options.
 * @return array The current brand filter group with any additional or modified data.
 */
	protected function _addExtraDataToBrandGroupForListing($currentBrandGroup) {
		return $currentBrandGroup;
	}

/**
 * Adds product category options to the filter options.
 *
 * @param array $productIds Contains product ids that will be used to find matching categories
 * @param array $params Additional params passed through from the calling method
 * @return array Containing product categories as options to be added to the option groups
 */
	protected function _addProductCategoriesToOptionsForListing(
		$productIds = [],
		$params = []
	) {
		// add product category based params to the listing query
		$defaultParams = [
			'conditions' => [
				'Product.id' => $productIds,
				'Product.is_active' => true
			],
			'fields' => [
				'ProductCategory.*'
			],
			'group' => [
				'ProductCategory.id'
			],
			'joins' => [
				[
					'table' => 'ev_shop_categories_products',
					'alias' => 'CategoriesProduct',
					'conditions' => [
						'CategoriesProduct.product_id = Product.id'
					]
				],
				[
					'table' => 'ev_shop_categories',
					'alias' => 'ProductCategory',
					'conditions' => [
						'ProductCategory.id = CategoriesProduct.category_id',
						'ProductCategory.is_active' => true
					]
				]
			]
		];

		$params = array_merge_recursive($defaultParams, $params);

		$data = EvClassRegistry::init('EvShop.Product')->find('all', $params);
		$productCategoryFilterOptions = Hash::extract($data, '{n}.ProductCategory');

		return [
			'name' => 'Category',
			'Option' => $productCategoryFilterOptions
		];
	}

/**
 * Add attributes to the listing filter. Attribute groups are cached to speed up the query. The attribute groups can
 * have their query params modified by providing query parameters under the "ProductAttributeGroup" key in $extraParams.
 *
 * @param array $optionGroups All of the current option groups for the listing.
 * @param array $extraParams The params to select which products to get the attributes from.
 * @return void.
 */
	protected function _addAttributesToOptionsForListing($optionGroups, $extraParams) {
		$Product = EvClassRegistry::init('EvShop.Product');

		$AttributeGroup = EvClassRegistry::init('EvShop.ProductAttributeGroup');
		$attributeGroupsList = Cache::remember('ev_shop_attribute_groups', function () use ($AttributeGroup, $extraParams) {
			$attributeGroupParams = [
				'conditions' => [
					$AttributeGroup->alias . '.is_active' => true,
					$AttributeGroup->alias . '.show_in_filters' => true
				],
				'fields' => [
					$AttributeGroup->alias . '.id',
					$AttributeGroup->alias . '.name',
					$AttributeGroup->alias . '.system_name',
					$AttributeGroup->alias . '.show_in_filters',
					$AttributeGroup->alias . '.sequence',
				]
			];

			if (!empty($extraParams[$AttributeGroup->alias])) {
				$attributeGroupParams = $AttributeGroup->mergeQueryParams(
					$attributeGroupParams,
					$extraParams[$AttributeGroup->alias]
				);
			}

			$attributeGroups = $AttributeGroup->find('all', $attributeGroupParams);

			return Hash::combine(
				$attributeGroups, '{n}.' . $AttributeGroup->alias . '.id',
				'{n}.' . $AttributeGroup->alias
			);
		});

		$attributeParams = [
			'joins' => [
				[
					'table' => 'ev_shop_variants',
					'alias' => 'Variant',
					'conditions' => [
						'Variant.product_id = Product.id',
						'Variant.is_active' => true,
					]
				],
				[
					'table' => 'ev_shop_product_attributes_products',
					'alias' => 'ProductAttributeProduct',
					'conditions' => [
						'ProductAttributeProduct.product_id = Product.id'
					]
				],
				[
					'table' => 'ev_shop_product_attributes',
					'alias' => 'ProductAttribute',
					'conditions' => [
						'ProductAttributeProduct.product_attribute_id = ProductAttribute.id',
						'ProductAttribute.product_attribute_group_id' => array_keys($attributeGroupsList),
					]
				],
			],
			'fields' => [
				'ProductAttribute.*',
				'Product.id',
			],
			'group' => 'ProductAttribute.id',
			'conditions' => [
				'Product.is_active' => 1
			],
		];

		$attributeParams = array_merge_recursive($attributeParams, $extraParams);

		$attributes = $Product->find('all', $attributeParams);

		//Loop through each attribute and add it to it's corresponding attribute group.
		$attributeGroups = [];
		foreach ($attributes as $attribute) {
			$attributeGroupId = $attribute['ProductAttribute']['product_attribute_group_id'];

			//Prefix the attribute group id with a string so that they don't clash with the option groups.
			$attributeGroupKey = 'attr:' . $attributeGroupId;

			if (
				!isset($attributeGroups[$attributeGroupKey])
				&& isset($attributeGroupsList[$attributeGroupId])
			) {
				$newAttributeGroup = $attributeGroupsList[$attributeGroupId];

				$attributeGroups[$attributeGroupKey] = $newAttributeGroup;
			}

			//Attributes are still added as options so views can just loop through the data.
			$attributeGroups[$attributeGroupKey]['Option'][] = $attribute['ProductAttribute'];
		}

		// Only include attributes that have more than one available option.
		foreach ($attributeGroups as $key => $attributeGroup) {
			$optionGroups = $this->_addAttributeGroupToOptionGroupsForListing($optionGroups, $key, $attributeGroup);
		}

		return $optionGroups;
	}

/**
 * Modify an attribute group before it is added into the option groups for listing array.
 *
 * @param array  $optionGroups   The current array of option groups to return for the listing.
 * @param string $key            The key to use when adding the attributeGroup to the option groups.
 * @param array  $attributeGroup The array of the attribute group currently being modified.
 * @return array The option groups with the added attribute group.
 */
	protected function _addAttributeGroupToOptionGroupsForListing($optionGroups, $key, $attributeGroup) {
		$optionGroups[$key] = $attributeGroup;

		return $optionGroups;
	}

/**
 * Processes the supplied filter parameters and returns a route array, allowing
 * us to redirect to a bookmarkable URL
 *
 * @param array $request Array containing the current request parameters
 * @return string Containing the route to redirect to
 */
	public function getRedirectFromFilters($request = []) {
		// Filters have been applied (on the index page),
		// redirect to include the named parameters
		$newRoute = $request->params;

		//Take out the passedData and assign it to the new route
		foreach ($request->params['pass'] as $passedData) {
			$newRoute[] = $passedData;
		}
		unset($newRoute['pass']);

		//Take out the named parameters and assign them to the new route
		foreach ($request->params['named'] as $namedKey => $namedData) {
			$newRoute[$namedKey] = $namedData;
		}
		unset($newRoute['named']);

		//Take out the query data and add it to the new route
		foreach ($request->query as $namedKey => $namedData) {
			$newRoute['?'][$namedKey] = $namedData;
		}
		unset($newRoute['named']);

		$addedFilter = false;

		//Take out the passed in filter data and assign it to the new route
		if (! empty($request->data['EvShopFilter'])) {
			foreach ($request->data['EvShopFilter'] as $filterGroup => $filterOption) {
				if (is_array($filterOption)) {
					foreach ($filterOption as $optionName => $selected) {
						$urlFilterGroup = $filterGroup;
						$urlOptionName = $urlFilterGroup . '-' . $optionName;

						$urlOptionName = InflectorExt::slugUrlSafePreserveChars($urlOptionName, true);

						if ($selected != '0') {
							$newRoute[$urlOptionName] = $urlFilterGroup;
							$addedFilter = true;
						} else {
							if (isset($newRoute[$urlOptionName])) {
								unset($newRoute[$urlOptionName]);
							}
						}
					}
				} else {

					// Dont filter by price if set to "Any"
					if ($filterGroup == 'min_price' && empty($filterOption)) {
						unset($newRoute['min_price']);
						continue;
					}

					if ($filterGroup == 'max_price' && empty($filterOption)) {
						unset($newRoute['max_price']);
						continue;
					}

					$filterOption = InflectorExt::slugUrlSafePreserveChars(
						$filterOption,
						true
					);

					$urlFilterGroup = $filterGroup;
					$urlFilterOption = $filterOption;
					$newRoute[$urlFilterGroup] = $urlFilterOption;
					$addedFilter = true;
				}
			}
		}

		if ($addedFilter && isset($newRoute['page'])) {
			unset($newRoute['page']);
		}

		return Router::url($newRoute);
	}

/**
 * Processes the supplied filter options and returns an array of params to use
 * in the Products paginate call
 *
 * @param array $filterOptions The filter options to use in the paginate settings
 * @param int $currencyId The currency id used for processing pricing
 * @return array Containing the filter option specific paginate settings
 */
	public function getFilterOptionsForPaginate($filterOptions = [], $currencyId = null) {
		$returnParams = [];
		$returnPriceOrder = [];
		$returnOptionFilter = [];
		$returnAttributeFilter = [];
		$returnBrandFilter = [];
		$returnPriceFilter = [];

		if (!empty(Configure::read('EvShop.filterOnAttributes'))) {
			$this->_controller->loadModel('EvShop.ProductAttributeGroup');
			$attributeGroupNames = $this->_controller->ProductAttributeGroup->find('list', [
				'fields' => 'name',
				'conditions' => [
					'ProductAttributeGroup.is_active' => 'true',
					'ProductAttributeGroup.show_in_filters' => 'true'
				]
			]);
		}

		if (! empty($filterOptions) && is_array($filterOptions)) {
			//Remove limit
			if (array_key_exists('limit', $filterOptions)) {
				$returnParams['limit'] = $filterOptions['limit'];

				unset($filterOptions['limit']);
			}

			//Remove Order
			if (array_key_exists('productOrder', $filterOptions)) {
				$filterOrder = $filterOptions['productOrder'];

				unset($filterOptions['productOrder']);
			}

			//Remove Paging
			if (array_key_exists('page', $filterOptions)) {
				$filterPage = $filterOptions['page'];

				unset($filterOptions['page']);
			}

			if (array_key_exists('price_range', $filterOptions)) {
				$filterPriceRange = $filterOptions['price_range'];

				// Need to sort it out a bit cause all the values are split up
				// by spaces when interpreted from the url
				$filterPriceRange = explode(' ', $filterPriceRange);
				// Ranges should have been input using the correct format. So assuming it is
				// in the correct format, if there are 4 elements then we have a range
				// (lower price - higher price), otherwise just a minimum
				if (count($filterPriceRange) == 4) {
					//Need to combine the pairs of elements to get the actual price values
					$returnPriceFilter['minimum'] = array_shift($filterPriceRange) . '.' . array_shift($filterPriceRange);
					$returnPriceFilter['maximum'] = array_shift($filterPriceRange) . '.' . array_shift($filterPriceRange);
					$returnPriceFilter['type'] = '><';
				} else {
					//Just a minimum so only has one price value
					$returnPriceFilter['minimum'] = array_shift($filterPriceRange) . '.' . array_shift($filterPriceRange);
					$returnPriceFilter['type'] = '<';
				}

				if ($currencyId !== null) {
					$returnPriceFilter['currency_id'] = $currencyId;
				}

				unset($filterOptions['price_range']);
			}

			//Check if brand filters exist and apply if they do
			if (array_key_exists('Brands', $filterOptions)) {
				foreach ($filterOptions['Brands'] as $brandIndex => $brand) {
					$returnBrandFilter[] = $brand;
				}

				// Remove brands from the filter options so we can loop through
				// without making a mess
				unset($filterOptions['Brands']);
			}

			// Separate attributes from options.
			if (!empty($attributeGroupNames)) {
				foreach ($attributeGroupNames as $attributeGroupName) {
					if (!empty($filterOptions[$attributeGroupName])) {
						$returnAttributeFilter[$attributeGroupName] = $filterOptions[$attributeGroupName];
						unset($filterOptions[$attributeGroupName]);
					}
				}
			}

			// Whatever filters are left should just be the selected options in
			// their option groups
			$returnOptionFilter = $filterOptions;
		}

		if (isset($filterOrder)) {
			$filterOrder = strtolower($filterOrder);

			//if the ordering is on prices then that needs to happen here
			if (strpos($filterOrder, 'price') !== false) {
				switch ($filterOrder) {
					case 'price asc':
						$returnPriceOrder['direction'] = 'ASC';
						break;
					case 'price desc':
						$returnPriceOrder['direction'] = 'DESC';
						break;
				}

				if ($currencyId !== null) {
					$returnPriceOrder['currency_Id'] = $currencyId;
				}
			} else {
				switch ($filterOrder) {
					case 'name asc':
						$returnParams['order'] = 'Product.name ASC';
						break;
					case 'name desc':
						$returnParams['order'] = 'Product.name DESC';
						break;
					case 'created asc':
						$returnParams['order'] = 'Product.created ASC';
						break;
					case 'created desc':
						$returnParams['order'] = 'Product.created DESC';
						break;
				}
			}
		}

		return [
			'params' => $returnParams,
			'price' => $returnPriceOrder,
			'brand' => $returnBrandFilter,
			'option' => $returnOptionFilter,
			'attribute' => $returnAttributeFilter,
			'price_range' => $returnPriceFilter
		];
	}

/**
 * Loops through the supplied named parameters and humanizes strings and
 * reinstates decimal points where required
 *
 * @param array $namedParams Contains the named request parameters to clean
 * @return array Containing a sanatized collection of named request params
 */
	public function sanatizeParams($namedParams = []) {
		$newParamArray = [];

		if (! empty($namedParams)) {
			// These filters can be kept as they are and don't need to be sanatized
			$keepFilters = [
				'limit',
				'order',
				'page',
				'productOrder',
				'min_price',
				'max_price'
			];

			foreach ($namedParams as $paramName => $paramValue) {
				if (in_array($paramName, $keepFilters)) {
					$newParamArray[$paramName] = InflectorExt::slugUrlSafePreserveChars(
						$paramValue,
						false
					);
				} else {
					$paramName = InflectorExt::slugUrlSafePreserveChars($paramName, false);

					// Filter options are presented as Group-Option => Group
					// Ignore any that are not in this format
					if (strpos($paramName, $paramValue . '-') === 0) {
						$paramName = substr( $paramName, strlen($paramValue) + 1 );
						$newParamArray[$paramValue][] = $paramName;
					}
				}
			}
		}

		return $newParamArray;
	}

/**
 * Extracts and returns the lowest variant pricing value from the supplied Variant array
 *
 * @param array $variantData A hasMany array of variants to extract the lowest price from.
 * @return decimal Containing the lowest price found
 */
	public function extractLowestVariantPrice($variantData = []) {
		$price = 0;

		$data = Hash::extract($variantData, '{n}.VariantPricing.{n}.lowest_price');

		if (! empty($data)) {
			$price = min($data);
		}

		return $price;
	}
}
