<?php

App::uses('AppComponent', 'Controller/Component');

class CategoryOverridesComponent extends AppComponent {

/**
 * The cached hash data
 *
 * @var array
 */
	private $__hashDataCache;

/**
 * Override the content and meta data of a category based on options selected in a filter.
 *
 * @param array $data The category data to override.
 * @return array Overridden data.
 */
	public function override($data) {
		if (empty(Configure::read('EvShop.useCategoryOverrides'))) {
			return $data;
		}

		if (empty($this->_controller->Category)) {
			$this->_controller->loadModel('EvShop.Category');
		}

		if (empty($data[$this->_controller->Category->alias]['id'])) {
			return $data;
		}

		if (empty($this->_controller->CategoryOverride)) {
			$this->_controller->loadModel('EvShop.CategoryOverride');
		}

		$override = $this->_controller->Category->CategoryOverride->findOverride($this->_getHashData($data));

		$override = $this->_restrictByRequest($data, $override);

		if (empty($override)) {
			return $data;
		}

		$data = $this->_mergeOverride($data, $override);

		return $data;
	}

/**
 * Check to see if there is an override for the given category based on options selected in a filter.
 *
 * @param array $data The category data to override.
 * @return bool.
 */
	public function hasOverride($data) {
		return $this->_controller->Category->CategoryOverride->hasOverride($this->_getHashData($data));
	}

/**
 * Take the data from the category and request data to find an override.
 *
 * @param array $data Category data.
 * @return array Data to hash.
 */
	protected function _getHashData($data) {
		if (!empty($this->__hashDataCache)) {
			return $this->__hashDataCache;
		}

		$data = [
			'categoryId' => !empty($data[$this->_controller->Category->alias]['id']) ?
				$data[$this->_controller->Category->alias]['id'] :
				null,
			'brandId' => $this->_getFilterBrand(),
			'options' => $this->_getFilterOptions(),
			'attributes' => $this->_getFilterAttributes(),
		];

		$this->__hashDataCache = $data;

		return $data;
	}

/**
 * Get the id of the brand from the filter. Overrides can only be associated with a single brand so if multiple brands
 * have been selected in the filter then an override can't be applied.
 *
 * As filters can be different per project, extend to apply an override.
 *
 * @return int The id of the brand to override by.
 */
	protected function _getFilterBrand() {
		return 0;
	}

/**
 * Get a list of options from the filter.
 *
 * As filters can be different per project, extend to apply an override.
 *
 * @return array List of option ids.
 */
	protected function _getFilterOptions() {
		return [];
	}

/**
 * Get a list of attributes from the filter.
 *
 * As filters can be different per project, extend to apply an override.
 *
 * @return array List of attribute ids.
 */
	protected function _getFilterAttributes() {
		return [];
	}

/**
 * Restrict an override by the current request. Restrictions should be added here only if they can't be put in the
 * override hash due to always being present but unknown before the request is made. For example: pagination page,
 * route, etc.
 *
 * @param array $data     Category data to be overridden.
 * @param array $override Category override data.
 * @return array Restricted override.
 */
	protected function _restrictByRequest($data, $override) {
		$categoryOverrideAlais = $this->_controller->Category->CategoryOverride->alias;

		if (empty($override[$categoryOverrideAlais]['pagination_page'])) {
			return $override;
		}

		$paginationPage = $this->_controller->request->param('named.page');
		$paginationPageOverride = $override[$categoryOverrideAlais]['pagination_page'];

		if (empty($paginationPage) && $paginationPageOverride === 1) {
			//Override has been restricted to the first page of pagination. If no page is requested, assume it is the first page.
			return $override;
		}

		if ($paginationPage === $paginationPageOverride) {
			return $override;
		}

		return [];
	}

/**
 * Merge the override data into the category data. If meta fields have been overridden then merge them into the meta
 * data fields.
 *
 * @param array $data     The category data to override
 * @param array $override The data to override with.
 * @return array The overridden category data.
 */
	protected function _mergeOverride($data, $override) {
		$fieldBlacklist = $this->_mergeFieldBlackList();

		foreach ($override['CategoryOverride'] as $field => $value) {
			if (empty($value)) {
				//If the override value hasn't been set then don't override the original data.
				continue;
			}

			$alias = 'Category';

			if (strpos($field, 'meta_') === 0) {
				//This is a meta field
				$alias = 'MetaData';
				$field = preg_replace('/meta_/', '', $field, 1);
			}

			if (in_array($alias . '.' . $field, $fieldBlacklist)) {
				continue;
			}

			$data[$alias][$field] = $value;
		}

		return $data;
	}

/**
 * A list of fields to not merge even if overridden values have been provided.
 *
 * @return array.
 */
	protected function _mergeFieldBlackList() {
		return [
			'Category.id',
		];
	}
}
