<?php

App::uses('EvShopAppController', 'EvShop.Controller');

class CategoryOverridesController extends EvShopAppController {

/**
 * Used to populate form drop down selects.
 *
 * @return void.
 */
	protected function _adminPopulateLookups() {
		$Model = $this->{$this->modelClass};

		$this->set(
			'categories',
			$Model->Category->generateTreeList(
				null,
				null,
				null,
				'--'
			)
		);

		$this->set(
			'brands',
			$Model->Brand->find('list')
		);
	}

/**
 * {@inheritDoc}
 */
	protected function _adminFilterFields() {
		$filterFields = parent::_adminFilterFields();

		$Model = $this->{$this->modelClass};
		$alias = $Model->alias;

		$categoryFilterField = [
			$alias . '.category_id' => [
				'type' => 'select',
				'label' => 'Category',
				'compare' => [
					$alias . '.category_id' => '%s',
				],
			],
		];

		$filterFields = ArrayUtil::addAfter($filterFields, $alias . '.id', $categoryFilterField);

		return $filterFields;
	}

/**
 * Defines the buttons in the toolbar displayed on an admin_index page.
 *
 * @return array
 */
	protected function _adminIndexToolbar() {
		$Model = $this->{$this->modelClass};
		$modelAlias = $Model->alias;

		$actions = parent::_adminIndexToolbar();

		$actions['Back To Categories'] = [
			'url' => [
				'plugin' => 'ev_shop',
				'controller' => 'categories',
				'action' => 'index',
			],
			'class' => 'toolbar__btn toolbar__btn--listing'
		];

		return $actions;
	}

/**
 * Pagination settings for admin_index.
 *
 * @return array Pagination settings.
 */
	protected function _adminIndexPaginate() {
		$Model = $this->{$this->modelClass};

		$paginate = parent::_adminIndexPaginate();

		$paginate['contain']['Category'] = [];
		$paginate['contain']['Brand'] = [];

		// Join Options
		$paginate['joins'][] = [
			'type' => 'left',
			'table' => 'ev_shop_category_overrides_options',
			'alias' => 'CategoryOverridesOption',
			'conditions' => [
				'CategoryOverridesOption.category_override_id = ' . $Model->alias . '.id',
			],
		];

		$paginate['joins'][] = [
			'type' => 'left',
			'table' => 'ev_shop_options',
			'alias' => 'Option',
			'conditions' => [
				'Option.id = CategoryOverridesOption.option_id',
			],
		];

		$Model->virtualFields['options'] = 'GROUP_CONCAT(Option.name)';

		// Join Product Attributes
		$paginate['joins'][] = [
			'type' => 'left',
			'table' => 'ev_shop_category_overrides_product_attributes',
			'alias' => 'CategoryOverridesProductAttribute',
			'conditions' => [
				'CategoryOverridesProductAttribute.category_override_id = ' . $Model->alias . '.id',
			],
		];

		$paginate['joins'][] = [
			'type' => 'left',
			'table' => 'ev_shop_product_attributes',
			'alias' => 'ProductAttribute',
			'conditions' => [
				'ProductAttribute.id = CategoryOverridesProductAttribute.product_attribute_id',
			],
		];

		$Model->virtualFields['product_attributes'] = 'GROUP_CONCAT(ProductAttribute.name)';

		return $paginate;
	}

/**
 * Defines the columns displayed in the admin_index results table using a columns whitelist.
 *
 * @return array Index columns
 */
	protected function _adminIndexColumns() {
		$alias = $this->{$this->modelClass}->alias;

		$columns = parent::_adminIndexColumns();

		$categoryColumn = [
			'Category.name' => [
				'type' => 'String',
				'label' => 'Category',
			],
		];

		$columns = ArrayUtil::addBefore($columns, $alias . '.name', $categoryColumn);

		$additionalColumns = [
			'Brand.name' => [
				'type' => 'string',
				'label' => 'Brand',
			],
			$alias . '.options' => [
				'type' => 'string',
				'label' => 'Options'
			],
			$alias . '.product_attributes' => [
				'type' => 'string',
				'label' => 'Attributes'
			],
		];

		$columns = ArrayUtil::addBefore($columns, $alias . '.created', $additionalColumns);

		return $columns;
	}

/**
 * Defines the buttons in the toolbar displayed on an admin_edit page.
 *
 * @param int $id The id of the record being edited.
 * @return array
 */
	protected function _adminFormToolbar($id = null) {
		$Model = $this->{$this->modelClass};
		$modelAlias = $Model->alias;

		$actions = parent::_adminFormToolbar($id);

		$actions['Back To Overrides'] = [
			'url' => [
				'plugin' => 'ev_shop',
				'controller' => 'category_overrides',
				'action' => 'index',
			],
			'class' => 'toolbar__btn toolbar__btn--listing'
		];

		return $actions;
	}

/**
 * Defines the fields displayed in an admin_form for this model. Defaults to all fields in the db table.
 *
 * @return array Form fields.
 */
	protected function _adminFormFields() {
		$Model = $this->{$this->modelClass};
		$alias = $Model->alias;

		$fields = parent::_adminFormFields();

		if (isset($fields[$alias . '.override_hash'])) {
			unset($fields[$alias . '.override_hash']);
		}

		//Conditions
		if (isset($fields[$alias . '.pagination_page'])) {
			$fields[$alias . '.pagination_page']['min'] = 1;
			$fields[$alias . '.pagination_page']['displayInfo'] = 'Provide a page number (1 or greater) if you want to limit the overridden content to only display on a specific paginated page. If left blank then the category override will appear on all paginated pages.';

			$fields['tabs']['Conditions'][$alias . '.pagination_page'] = $fields[$alias . '.pagination_page'];
			unset($fields[$alias . '.pagination_page']);
		}

		if (isset($fields[$alias . '.brand_id'])) {
			$fields['tabs']['Conditions'][$alias . '.brand_id'] = $fields[$alias . '.brand_id'];
			unset($fields[$alias . '.brand_id']);
		}

		// Options
		$optionGroups = $Model->Option->OptionGroup->find('all', ['contain' => 'Option']);

		foreach ($optionGroups as $group) {
			if (!empty($group['Option'])) {
				$fields['tabs']['Options']['Option.' . $group['OptionGroup']['id']] = [
					'type' => 'select',
					'options' => Hash::combine($group['Option'], '{n}.id', '{n}.name'),
					'label' => $group['OptionGroup']['name'],
				];
			}
		}

		//Product Attributes
		$productAttributeGroups = $Model->ProductAttribute->ProductAttributeGroup->find(
			'all',
			[
				'contain' => 'ProductAttribute',
			]
		);

		foreach ($productAttributeGroups as $group) {
			if (!empty($group['ProductAttribute'])) {
				$fields['tabs']['Attributes']['ProductAttribute.' . $group['ProductAttributeGroup']['id']] = [
					'type' => 'select',
					'options' => Hash::combine($group['ProductAttribute'], '{n}.id', '{n}.name'),
					'label' => $group['ProductAttributeGroup']['name'],
				];
			}
		}

		return $fields;
	}
}
