<?php

App::uses('EvShopAppModel', 'EvShop.Model');
App::uses('PurchasableInterface', 'EvCheckout.Lib');

class ProductVariant extends EvShopAppModel implements PurchasableInterface {

	public $belongsTo = array(
		"Product" => array(
			'className' => 'EvShop.Product'
		)
	);

	public $hasMany = array(
		'ProductAttributeOptionProductVariant' => array(
			'className' => 'EvShop.ProductAttributeOptionProductVariant'
		)
	);

	public $hasAndBelongsToMany = array(

		"ProductAttributeOption" => array(
			'className' => 'EvShop.ProductAttributeOption', 
			'joinTable' => 'ev_shop_product_attribute_options_product_variants'
		)

	);


	public $validate = array(
		'name' => array(
			'maxLength' => array(
				'rule' => array('maxLength', 250),
				'message' => 'No more tham 250 characters long',
				'allowEmpty' => true
			)
		),
		'rrp' => array(
			'required' => array(
				'rule' => 'notEmpty',
				'message' => 'Required'
			),
			'maxLength' => array(
				'rule' => array('maxLength', 16),
				'message' => 'No more tham 16 characters long'
			),
			'numeric' => array(
				'rule' => 'numeric',
				'message' => 'Numerical values only'
			)
		),
		'price' => array(
			'required' => array(
				'rule' => 'notEmpty',
				'message' => 'Required'
			),
			'maxLength' => array(
				'rule' => array('maxLength', 16),
				'message' => 'No more tham 16 characters long'
			),
			'numeric' => array(
				'rule' => 'numeric',
				'message' => 'Numerical values only'
			)
		),
		'stock' => array(
			'naturalNumber' => array(
				'rule' => array('naturalNumber', true),
				'message' => 'Integers only',
				'allowEmpty' => true
			)
		)
	);


/**
 * Generates a unique variant product key using the given options 
 * @param array $variantOptions An array of the variant options in the form array(array('id' => '..', 'name' => '..'), array('id' => '..', 'name' => '..'))
 * @return string An ordered list of the option ids ordered and joined by "-" e.g. "3-5-9-21"
 */
	public function generateKeyFromVariantOptions($variantOptions) {

		$variantOptionIds = Hash::extract($variantOptions, '{n}.id');

		if(!empty($variantOptionIds)) {
			
			sort($variantOptionIds);
			return implode("-", $variantOptionIds);

		} else {

			return "no-attributes";

		}


	}

/**
 * Returns a list of variants for the given product indexed by the product option ids 
 * @see ProductVariant::generateKeyFromVariantOptions()
 * @param   $productId 
 * @return array An array of product variants indexed using ProductVariant::generateKeyFromVariantOptions() in a find('all') format
 */
	public function getProductVariantsIndexedByAttributeOptions($productId) {

		//Get all product variants for the given product and their product attribute options

		$variants = $this->find('all', array(
				'conditions' => array(
					'ProductVariant.product_id' => $productId
				), 
				'contain' => array(
					'ProductAttributeOption' => array(
						'order' => 'ProductAttributeOption.product_attribute_id'
					)
				)
			)
		);

		$indexedArray = array();

		foreach($variants as $variant) {

			$variantArray = $variant['ProductVariant'];
			unset($variant['ProductVariant']);

			$variantArray = array_merge($variantArray, $variant);

			$indexedArray[$this->generateKeyFromVariantOptions($variant['ProductAttributeOption'])] = $variantArray;
		}

		return $indexedArray;

	}


/**
 * Returns a find('all') array with all the available product variants for the given product
 * Available is defined by the following criteria
 *	 1. The variant has all of the attributes that are currently attached to the product 
 *	 2. The variant has a price which is not empty
 * 
 * @param $productId (int) - Id of the product you wish to fetch variants for
 * @param $frontEnd If set to true will only bring out those available for purchase (i.e. price of )
 * @param $query (array) - Any additional parameters you wish to pass into the query
 * @return (array) 	find('all') format array of Product Variants 
 */
	public function getAvailableProductVariants($productId, $query = array()) {

		$findQuery = array_merge($query, array(
			'conditions' =>	array(
				'ProductVariant.product_id' => $productId, 
				'ProductVariant.price <>' => ''
			), 
			'contain' => array(
				'ProductAttributeOption' => array(
					'ProductAttribute'
				)
			)
		));

		$productVariants = $this->find('all', $findQuery);

		return $productVariants;

	}


/** 
 * Finds a product variant based on the product attributes passed in
 * @param int $productId The parent product
 * @param array $productAttributeOptions An array of product attribute options the product variant should have 
 * @return array ProductVariant find('first')
 */
	public function findByProductAttributeOptions($productId, $productAttributeOptions) {
		
		$params = array(
			'conditions' => array(
				'ProductVariant.product_id' => $productId, 
				'ProductVariant.price !=' => null
			),
			'joins' => array(
				array(
					'table' => 'ev_shop_product_attribute_options_product_variants', 
					'alias' => 'ProductAttributeOptionProductVariant', 
					'type' => 'LEFT',
					'conditions' => array(
						'ProductAttributeOptionProductVariant.product_variant_id = ProductVariant.id', 
						'ProductAttributeOptionProductVariant.product_attribute_option_id' => $productAttributeOptions
					)
				)
			), 
			'group' => 'ProductVariant.id HAVING COUNT(ProductAttributeOptionProductVariant.id) >= "'. count($productAttributeOptions). '"'
		); 

		return $this->find('first', $params); 

	}

/**
 * Finds and prepares the data to add to the basket. It will check that at least
 * one item is in stock when EvCheckout.track_stock is enabled.
 * 
 * @param integer $productVariantId
 * @return array or false when no product variant is found.
 * 
 * @todo  Rewrite more efficiently 
 */
	public function prepareForBasket($productVariantId) {

		$productVariant = $this->findById($productVariantId);

		// Get distinguishing product attributes, i.e. the ones the user has selected on the product page
		$distinguishingProductAttributes = $this->ProductAttributeOption->find('list', array(
				'fields' => array(
					'ProductAttributeOption.product_attribute_id',
					// 'ProductAttributeOption.name'
				),
				'conditions' => array(), 
				'joins' => array(
					array(
						'table' => 'ev_shop_product_attribute_options_product_variants', 
						'alias' => 'ProductAttributeOptionProductVariant', 
						'type' => 'LEFT',
						'conditions' => array(
							'ProductAttributeOptionProductVariant.product_attribute_option_id = ProductAttributeOption.id', 
						)
					),
					array(
						'table' => 'ev_shop_product_variants', 
						'alias' => 'ProductVariant', 
						'type' => 'RIGHT',
						'conditions' => array(
							'ProductAttributeOptionProductVariant.product_variant_id = ProductVariant.id',
							'ProductVariant.product_id' => $productVariant['ProductVariant']['product_id']
						)
					)
				), 
				'group' => 'ProductAttributeOption.product_attribute_id HAVING COUNT(DISTINCT ProductAttributeOption.id) > 1'
			)
		);

		$params = array(
			'conditions' => array(
				$this->alias.'.id' => $productVariantId
			),
			'contain' => array(
				'Product', 
				'ProductAttributeOption' => array(
					'ProductAttribute', 
					'conditions' => array(
					),
					'order' => 'ProductAttributeOption.sequence ASC'
				)
			)
		);

		if ($distinguishingProductAttributes) {
			
			$params['contain']['ProductAttributeOption']['conditions']['ProductAttributeOption.product_attribute_id'] = Hash::extract($distinguishingProductAttributes, '{n}');
		
		} else {
			
			$params['contain']['ProductAttributeOption']['conditions']['ProductAttributeOption.product_attribute_id'] = array();

		}

		// Check if we need to observe stock levels when adding to the basket.
		// There must be at least one item in stock before we proceed. Exact 
		// quantities will get handled when adding to the basket and adjusted
		// accordingly.
		$trackStock = Configure::read('EvCheckout.track_stock');
		if ($trackStock === true) {
			$params['conditions']['OR'][$this->alias.'.stock >'] = 0;
			$params['conditions']['OR'][$this->alias.'.unlimited_stock'] = true;
		}

		$productVariant = $this->find('first', $params);

		if (empty($productVariant)) {
			// No product variant available.
			return false;
		}


		// Checkout required all items to be passed with price excluding VAT so if our site prices 
		// include VAT we need to amend them 
		if (Configure::read('EvShop.prices_include_vat')) {

			// Calculate VAT
			$vatRate = $this->Product->VatRate->findById($productVariant['Product']['vat_rate_id']);
			if (!empty($vatRate)) {
				$unitCost = $productVariant[$this->alias]['price'] / (1 + ($vatRate['VatRate']['rate'] / 100));
			} else {
				$unitCost = $productVariant[$this->alias]['price'];
			}
			
		} else {

			$unitCost = $productVariant[$this->alias]['price'];

		}

		// Get the plugin name, because this could be extended we don't hardcode it
		if ($this->plugin) {
			$model = $this->plugin . '.' . $this->alias;
		} else {
			$model = $this->alias;
		}

		// Get product description
		if (!empty($productVariant['ProductAttributeOption'])) {

			$description = implode(", ", Hash::extract($productVariant['ProductAttributeOption'], '{n}.name'));

		} else {

			$description = " - ";

		}

		return array(
			'product_id' => $productVariant[$this->alias]['product_id'],
			'model' => $model, 
			'model_id' => $productVariant[$this->alias]['id'], 
			'unit_cost' => $unitCost,
			'name' => $productVariant['Product']['name'], 
			'vat_rate_id' => $productVariant['Product']['vat_rate_id'], 
			'description' => $description,
			'stock' => $productVariant[$this->alias]['stock'],
			'unlimited_stock' => $productVariant[$this->alias]['unlimited_stock']
		);

	}


/**
 * Callback from the EvCheckout.Order to adjust stock after purchase
 */
	public function adjustStock($orderItem, $modifier = -1) {

		$productVariantId = $orderItem['model_id'];
		$quantity = $orderItem['quantity'];

		$this->id = $productVariantId;

		$productVariant = $this->findById($productVariantId);

		if (!empty($productVariant)) {

			if ($productVariant['ProductVariant']['stock'] !== null && $productVariant['ProductVariant']['unlimited_stock'] === false) {

				$this->updateAll(array('stock' => 'stock + ' . ($modifier * $quantity)), array($this->alias . '.id' => $productVariantId));

			}

		}

		return;

	}


/**
 * Removes any product variants that are no longer applicable with the current combinations
 * of attributes. This happens when the variants of a product are changed.
 * @todo Rewrite into a single SQL query
 */
	public function cleanUpRedundantVariants($productId) {

		//First get all the product variants which dont have the same number of attributes as they should
		//But the attrbutes dont line up 
		$invalidVariants = $this->ProductAttributeOptionProductVariant->find('all', array(
				'fields' => 'product_variant_id',
				'conditions' => array(
					'product_attribute_option_id NOT IN (
						SELECT product_attribute_option_id 
						FROM ev_shop_product_attribute_options_products 
						WHERE ev_shop_product_attribute_options_products.product_id = "'.$productId.'"
					)'
				),
				'group' => 'product_variant_id'
			)
		);

		// debug($this->getLastQuery());

		//Secondly get all the product variants where they have the incorrect number of attributes
		$invalidVariants2 = $this->find('all', array(
			'joins' => array(
				array(
					'table' => 'ev_shop_product_attribute_options_product_variants',
					'alias' => 'ProductAttributeOptionProductVariant',
					'type' => 'LEFT',
					'conditions' => array(
						'ProductAttributeOptionProductVariant.product_variant_id = ProductVariant.id'
					)
				),
				array(
					'table' => 'ev_shop_product_attribute_options',
					'alias' => 'ProductAttributeOption',
					'type' => 'LEFT',
					'conditions' => array(
						'ProductAttributeOption.id = ProductAttributeOptionProductVariant.product_attribute_option_id'
					)
				)
			),
			'conditions' => array(
				'ProductVariant.product_id' => $productId
			),
			'group' => 'ProductVariant.id HAVING COUNT(DISTINCT(product_attribute_id)) <> (SELECT COUNT(DISTINCT product_attribute_id) FROM ev_shop_product_attribute_options AS ProductAttributeOption RIGHT JOIN ev_shop_product_attribute_options_products AS ProductProductAttributeOption ON ProductProductAttributeOption.product_attribute_option_id = ProductAttributeOption.id WHERE product_id = "'.$productId.'" GROUP BY product_id)'
			
		));


		//Extract the IDs
		$invalidVariantIds = Hash::extract($invalidVariants, '{n}.ProductAttributeOptionProductVariant.product_variant_id');
		$invalidVariant2Ids = Hash::extract($invalidVariants2, '{n}.ProductVariant.id');

		// debug($this->getLastQuery());
		
		//And delete
		$this->deleteAll(array(
			'ProductVariant.id' => $invalidVariantIds, 
			'ProductVariant.product_id' => $productId
		));
		$this->deleteAll(array(
			'ProductVariant.id' => $invalidVariant2Ids, 
			'ProductVariant.product_id' => $productId
		));


	}

}