<?php

App::uses('AppHelper', 'View/Helper');

App::uses('InflectorExt', 'EvInflector.Lib');

class ProductsHelper extends AppHelper {

	public function __construct(View $View, $settings = array()) {
		$this->helpers[] = 'Number';

		$this->helpers['Form'] = array(
			'className' => 'EvForm.FormExt'
		);

		$this->helpers['Html'] = array(
			'className' => 'HtmlExt'
		);

		parent::__construct($View, $settings);
	}

/**
 * Checks the supplied values and returns a boolean values based on
 * whether the filter field should be marked as checked / selected
 *
 * @param string $productOptionGroupName
 * @param string $productOptionName
 * @return boolean
 */
	protected function _isFilterSelected(
		$productOptionGroupName = null,
		$productOptionName = null
	) {
		$selected = false;

		//Check if this group has any selected filters
		if (in_array($productOptionGroupName, $this->request->params['named'])) {
			//change name to url safe name for numbers
			if (preg_match('/(^\d*\.\d*$)|(^\d*$)/', $productOptionName)) {
				$urlSafeProductOptionName = str_replace('.', '_', $productOptionName);
				if (preg_match('/^\d*$/', $productOptionName)) {
					//integer number, need to add the _0
					$urlSafeProductOptionName .= '_0';
				}
			} else {
				$urlSafeProductOptionName = InflectorExt::slugUrlSafePreserveChars($productOptionName, true);
			}

			$selected = array_key_exists(
				$urlSafeProductOptionName,
				$this->request->params['named']
			);
		}

		return $selected;
	}

/**
 * Contructs and returns unordered list element containing available
 * colour options ready to manipulate with CSS and Javascript
 *
 * @param array $optionGroup Contains the colour options to format
 * @param string $filterId The string to use as part of the filter checkboxes
 * @return string Contains the HTML ul element to d©2isplay on he DOM
 */
	protected function _buildColourFilterGroupFields(
		$optionGroup = [],
		$filterId = ''
	) {
		$filterOptionsHtml = '';

		if (! empty($optionGroup['FilterGroupOption'])) {
			foreach ($optionGroup['FilterGroupOption'] as $filterOptionItem) {
				$groupSelected = true;

				$optionItemHtml = $this->Html->tag('div', null, [
					'class' => 'truefilters hidden'
				]);

				foreach ($filterOptionItem['Option'] as $productOption) {
					$isChecked = $this->_isFilterSelected(
						$optionGroup['name'],
						$productOption['name']
					);

					$productOptionSafeName = InflectorExt::slugUrlSafePreserveChars(
						$productOption['name'],
						true
					);

					$checkboxId = $optionGroup['name'] . $productOptionSafeName . $filterId;

					if ($isChecked === false) {
						$groupSelected = false;
					}

					$optionItemHtml .= $this->Form->checkbox(
						'EvShopFilter.' . $optionGroup['FilterGroup']['name'] . '.' . $productOptionSafeName,
						[
							'id' => $checkboxId,
							'class' => 'required js-auto-submit',
							'value' => $productOption['name'],
							'checked' => $isChecked
						]
					);

					// Form->label() mangles the custom ID passed in
					$optionItemHtml .= $this->Html->tag('label', $productOption['name'], [
						'for' => $checkboxId
					]);
				}

				$optionItemHtml .= $this->Html->tag('/div');

				$optionItemHtml .= $this->Form->checkbox(false, [
					'class' => 'group-filter-check hidden',
					'id' => 'filterGf' . $filterOptionItem['id'],
					'checked' => $groupSelected,
					'value' => ''
				]);

				$optionItemHtml .= $this->Html->tag('label', null, [
					'for' => 'filterGf' . $filterOptionItem['id']
				]);

				$optionItemHtml .= $this->Html->tag(
					'div',
					$this->Html->tag('div', ''),
					[
						'class' => 'color-pick__item',
						'style' => 'background-color: #' . $filterOptionItem['data'] . ';'
					]
				);

				// wrap the $optionItemHtml markup in a li, ready for the DOM
				$filterOptionsHtml .= $this->Html->tag('li', $optionItemHtml);
			}
		}

		// return an ul element containing the contructions li elements
		return $this->Html->tag('ul', $filterOptionsHtml, ['class' => 'color-pick large']);
	}

/**
 * Contructs and return element containing filter group fields
 *
 * @param array $optionGroup Contains array of options to convert to HTML
 * @param string $filterId The unique string to use wihin checkbox elements
 * @return string Containing the filter group fields to show on the DOM
 */
	protected function _buildFilterGroupFields($optionGroup = [], $filterId = '') {
		$filterGroupFieldsHtml = '';

		foreach ($optionGroup['FilterGroupOption'] as $filterOptionItem) {
			$groupSelected = true;

			$productOptionsHtml = '';

			foreach ($filterOptionItem['Option'] as $productOption) {
				$isChecked = $this->_isFilterSelected(
					$optionGroup['name'],
					$productOption['name']
				);

				$productOptionSafeName = InflectorExt::slugUrlSafePreserveChars(
					$productOption['name'],
					true
				);

				$checkboxId = $optionGroup['name'] . $productOptionSafeName . $filterId;

				if ($isChecked === false) {
					$groupSelected = false;
				}

				$productOptionsHtml .= $this->Form->checkbox(
					'EvShopFilter.' . $optionGroup['FilterGroup']['name'] . '.' . $productOptionSafeName,
					[
						'id' => $checkboxId,
						'class' => 'required js-auto-submit',
						'value' => $productOption['name'],
						'checked' => $isChecked
					]
				);

				// Form->label() mangles the custom ID passed in
				$productOptionsHtml .= $this->Html->tag(
					'label',
					$productOption['name'],
					['for' => $checkboxId]
				);
			}

			$filterGroupHtml = $this->Html->tag('div', $productOptionsHtml, [
				'class' => 'truefilters hidden'
			]);

			$filterGroupHtml .= $this->Form->checkbox(false, [
				'class' => 'group-filter-check',
				'id' => 'filterGf' . $filterOptionItem['id'],
				'checked' => $groupSelected,
				'value' => ''
			]);

			$filterGroupHtml .= $this->Html->tag('label', $filterOptionItem['name'], [
				'for' => 'filterGf' . $filterOptionItem['id']
			]);

			$filterGroupFieldsHtml .= $this->Html->tag('div', $filterGroupHtml, [
				'class' => 'checkbox'
			]);
		}

		return $filterGroupFieldsHtml;
	}

/**
 * Contructs and returns form checkbox elements to use within the filter fields DOM
 *
 * @param array $optionGroup Contains array of options to convert to HTML
 * @param string $filterId The unique string to use wihin checkbox elements
 * @return string Containing the filter group fields to show on the DOM
 */
	protected function _buildFilterFields($optionGroup = [], $filterId = '') {
		$filterFieldsHtml = '';

		foreach ($optionGroup['Option'] as $option) {
			$isChecked = $this->_isFilterSelected(
				$optionGroup['name'],
				$option['name']
			);

			$optionSafeName = InflectorExt::slugUrlSafePreserveChars($option['name'],
				true
			);

			$checkboxId = $optionGroup['name'] . $optionSafeName . $filterId;

			$checkbox = $this->Form->checkbox(
				'EvShopFilter.' . $optionGroup['name'] . '.' . $optionSafeName,
				[
					'id' => $checkboxId,
					'class' => 'js-auto-submit',
					'value' => $option['name'],
					'checked' => $isChecked
				]
			);

			// Form->label() mangles the custom ID passed in
			$label = $this->Html->tag('label', $option['name'], [
				'for' => $checkboxId
			]);

			$filterFieldsHtml .= $this->Html->tag('div', $checkbox . $label, [
				'class' => 'checkbox',

			]);
		}

		return $this->Html->tag('div', $filterFieldsHtml, [
			'class' => 'menu__section__content collapse',
			'id' => 'menu-collapse-content-sub-' . $optionGroup['name'],
			'aria-expanded' => 'false'
		]);
	}

/**
 * Loops around supplied array of price options and updated the value of each
 * item to show a currency representation of the price
 *
 * @param array $priceOptions An array of price options to format
 * @return array Contains price options with formatted array values
 */
	public function formatPriceOptions($priceOptions = []) {
		if (! empty($priceOptions)) {
			foreach ($priceOptions as $key => $option) {
				$priceOptions[$key] = $this->Number->currency($option, null, ['places' => 0]);
			}
		}
		return $priceOptions;
	}

/**
 * for a product with single variant, extract the requests data
 *
 * @param 	string 		The column to extract
 * @param 	array 		The Data array
 * @return 	mixed 		The extracted data or bool false if multiple variants
 */
	public function singleVariantData($item, $data) {
		if (count($data['Variant']) > 1) {
			return false;
		}

		return Hash::get($data, 'Variant.0.' . $item);
	}

/**
 * get the option groups for the variants
 *
 * @param  	array 	Product Data array
 * @return 	array 	Array of option groups
 */
	public function getOptionGroups($data) {
		if (! empty($data['Variant']['0']['Option'])) {
			return Hash::combine($data['Variant'], '{n}.Option.{n}.OptionGroup.id', '{n}.Option.{n}.OptionGroup.name');
		}

		return array();
	}

/**
 * given the variant array and the option group
 * return the option name
 *
 * @param 	array 	Variant Array
 * @param 	int 	Option group Id
 * @return 	string 	Option value
 */
	public function getOption($Variant, $groupId) {
		return Hash::get($Variant, 'Option.' . $groupId . '.name', '-');
	}

/**
 * Given an array of VariantPricings, the pricing that matches the current currency ID will be found
 * and an array will be returned with the price. If the variant pricing contains a sale_price then
 * that will be returned as well as the lowest_price and the original price returned as the price.
 * @param  array $pricings An array of variant pricings for a specific variant
 * @return array           An array containing the price and the sale_price if it exists.
 */
	public function getVariantPrice($pricings) {
		$possiblePricings = Hash::combine($pricings, '{n}.currency_id', '{n}');
		$currencyId = CakeSession::read('EvCurrency.currencyId');

		$pricing = $possiblePricings[$currencyId];

		if (isset($pricing['sale_price']) && !empty($pricing['sale_price'])) {
			$originalPrice = $pricing['price'];
			if (Configure::read('EvShop.displayTaxAsVat')) {
				if (isset($pricing['price_incTax']) && !empty($pricing['price_incTax'])) {
					$originalPrice = $pricing['price_incTax'];
				}
			}
			return [
				'price' => $originalPrice,
				'sale_price' => $pricing['lowest_price']
			];
		} else {
			return [
				'price' => $pricing['lowest_price'],
			];
		}
	}

/**
 * Given a set of variant images, remove the restricted images so that only the generic images are left.
 * @param  array $allVariantImages An array of variant images
 * @return array                  An array without an restricted images in
 */
	public function getGenericVariantImages($allVariantImages) {
		foreach ($allVariantImages as $imageKey => $image) {
			if (isset($image['is_restricted']) && $image['is_restricted']) {
				unset($allVariantImages[$imageKey]);
			}
		}

		return $allVariantImages;
	}

/**
 * Given a set of variant images, remove the generic images so that only the restricted images are left.
 * Also remove any restricted images that don't have any options associated with them.
 * @param  array $allVariantImages An array of variant image
 * @return array                   An array without any generic images in,
 */
	public function getRestrictedVariantImages($allVariantImages) {
		foreach ($allVariantImages as $imageKey => $image) {
			if (!isset($image['is_restricted']) || !$image['is_restricted']) {
				unset($allVariantImages[$imageKey]);
			}

			if (!isset($image['VariantImageOption']) || empty($image['VariantImageOption'])) {
				unset($allVariantImages[$imageKey]);
			}
		}

		return $allVariantImages;
	}

/**
 * Given a product variant and a set of variant images, find all the images that are applicable to be displayed
 * for the variant. The variant images need to have options associated with them otherwise they will not be
 * returned.
 * @param  array $variant       An array containing the information about a specific variant
 * @param  array $variantImages An array of variant images.
 * @return array                An array of variant images that are applicable to the specified variant.
 */
	public function getRestrictedImagesForVariant($variant, $variantImages) {
		$imageArray = [];

		if (isset($variant['Option']) && !empty($variant['Option'])) {
			$allVariantOptions = Hash::extract($variant['Option'], '{n}.id');

			$restrictedImages = $this->getRestrictedVariantImages($variantImages);

			foreach ($restrictedImages as $image) {
				$currentGroupsChecked = [];

				foreach ($image['VariantImageOption'] as $imageOption) {
					if (!isset($currentGroupsChecked[$imageOption['option_group_id']])) {
						$currentGroupsChecked[$imageOption['option_group_id']] = false;
					}

					if (in_array($imageOption['option_id'], $allVariantOptions)) {
						$currentGroupsChecked[$imageOption['option_group_id']] = true;
					}
				}

				if (!in_array(false, $currentGroupsChecked)) {
					$imageArray[] = $image;
				}
			}
		}

		return $imageArray;
	}

/**
 * Basic GETTER for getting sale price of the product
 * It is better than accessing them directly as this way it can be changed without affecting the template
 * @param $product  Product array
 * @param $incVat Defaults to false, if set to true INC vat price is returned
 * @return  formatted currency Sale price
 */
	public function getSalePrice($product, $incVat = false) {
		$currencies = $this->_View->viewVars['currencies'];
		if (!empty($product['was_price_ex_vat'])) {
			if ($incVat && !empty($product['was_price'])) {
				return $this->Number->currency($product['was_price'], $currencies[CakeSession::read('EvCurrency.currencyId')]);
			} else {
				return $this->Number->currency($product['was_price_ex_vat'], $currencies[CakeSession::read('EvCurrency.currencyId')]);
			}
		} else {
			return false;
		}
	}

/**
 * Basic GETTER for getting RRP price of the product
 * It is better than accessing them directly as this way it can be changed without affecting the template
 * @param $product  Product array
 * @param $incVat Defaults to false, if set to true INC vat price is returned
 * @return  formatted currency RRP price
 */
	public function getRrpPrice($product, $incVat = false) {
		$currencies = $this->_View->viewVars['currencies'];
		if ($incVat) {
			return $this->Number->currency($product['rrp_price'], $currencies[CakeSession::read('EvCurrency.currencyId')]);
		} else {
			return $this->Number->currency($product['rrp_price_ex_vat'], $currencies[CakeSession::read('EvCurrency.currencyId')]);
		}
	}

/**
 * Basic GETTER for getting price of the product
 * It is better than accessing them directly as this way it can be changed without affecting the template
 * @param $product  Product array
 * @return  formatted currency price
 */
	public function getPrice($product) {
		$currencies = $this->_View->viewVars['currencies'];
		return $this->Number->currency($product['price'], $currencies[CakeSession::read('EvCurrency.currencyId')]);
	}

/**
 * Basic GETTER for getting max price of the product
 * It is better than accessing them directly as this way it can be changed without affecting the template
 * @param $product  Product array
 * @return  formatted currency price
 */
	public function getMaxPrice($product) {
		$currencies = $this->_View->viewVars['currencies'];
		return $this->Number->currency($product['max_price'], $currencies[CakeSession::read('EvCurrency.currencyId')]);
	}

/**
 * Returns true if there is a WAS price to the product
 * @param  Product  $product Product array
 * @return boolean          Returns true if the item is on sale
 */
	public function isSalePrice($product) {
		return ($product['was_price_ex_vat'] > 0);
	}

/**
 * Constructs and returns the html for the products filter form, usually used as
 * part of the sidebar displayed on prouct index templates
 *
 * @param array $filterOptions Contains the filter options to covert to filter elements
 * @param array $priceOptions Contains the price options to covert to price elements
 * @param string $filterTitle The title of the filter element
 * @param string $filterId The id string to use for the collapse functionality
 * @param array $clearLink The array to use when producing the clear link, contains name and url
 * @return string Contains the HTML to display on the template
 */
	public function buildFilterForm(
		$filterOptions = [],
		$priceOptions = [],
		$filterTitle = 'Filter Products',
		$filterId = 'Filter',
		$clearLink = []
	) {
		$filterHtml = $this->Html->tag('div', null, ['class' => 'menu']);

		if (!empty($filterTitle)) {
			$filterTitleLink = $this->Html->link(
				$filterTitle,
				'#',
				[
					'class' => 'menu__heading',
					'data-toggle' => 'collapse',
					'data-target' => '#menu-collapse-content-' . $filterId
				]
			);

			$filterHtml .= $this->Html->tag('div', $filterTitleLink, [
				'class' => 'menu__header'
			]);
		}

		// Collapse block open
		$filterHtml .= $this->Html->tag('div', null, [
			'aria-expanded' => false,
			'class' => 'menu__content collapse',
			'id' => 'menu-collapse-content-' . $filterId
		]);

		$filterHtml .= $this->Form->create('EvShopFilter', [
			'inputDefaults' => [
				'div' => 'form-group',
				'label' => [
					'class' => 'control-label'
				],
				'wrapInput' => '',
				'class' => 'form-control'
			],
			'id' => 'Filter' . $filterId,
			'class' => ''
		]);

		if (! empty($filterOptions)) {
			foreach ($filterOptions as $optionGroup) {
				if (!empty($optionGroup['Option'])) {
					// Group container open
					$filterHtml .= $this->Html->tag('div', null, ['class' => 'menu__section menu__section--filter-' . InflectorExt::slug(strtolower($optionGroup['name']))]);
					// Menu heading open
					$filterHtml .= $this->Html->tag('div', null, [
						'class' => 'menu__sub-heading collapsed',
						'data-toggle' => 'collapse',
						'data-target' => '#menu-collapse-content-sub-' . $optionGroup['name']
					]);

					$filterHtml .= $optionGroup['name'];

					if (! empty($optionGroup['tooltip'])) {
						$filterHtml .= ' ' . $this->Html->tag('i', '', [
							'class' => 'fa fa-info-circle',
							'data-toggle' => 'tooltip',
							'data-placement' => 'right',
							'title' => $optionGroup['tooltip']
						]);
					}

					// Menu heading close
					$filterHtml .= $this->Html->tag('/div');

					if (isset($optionGroup['FilterGroup'])) {
						if (
							! empty($optionGroup['id']) &&
							$optionGroup['id'] == EvShopProduct::COLOUR_OPTION_GROUP
						) {
							$filterHtml .= $this->_buildColourFilterGroupFields(
								$optionGroup,
								$filterId
							);
						} else {
							$filterHtml .= $this->_buildFilterGroupFields(
								$optionGroup,
								$filterId
							);
						}
					} else {
						$filterHtml .= $this->_buildFilterFields(
							$optionGroup,
							$filterId
						);
					}

					// Group container close
					$filterHtml .= $this->Html->tag('/div');
				}
			}
		}

		// Add in the price filters, if populated
		if (! empty($priceOptions)) {
			// Group container open
			$filterHtml .= $this->Html->tag('div', null, ['class' => 'menu__section menu__section--filter-price']);

			// Menu heading open
			$filterHtml .= $this->Html->tag('div', __('Price'), [
				'class' => 'menu__sub-heading'
			]);

			//option body open
			$filterHtml .= $this->Html->tag('div', null, ['class' => 'price-select']);
			$priceOptionsFormatted = $this->formatPriceOptions( $priceOptions );
			$filterHtml .= $this->Form->select(
				'EvShopFilter.min_price',
				$priceOptionsFormatted,
				[
					'class' => 'required js-auto-submit',
					'empty' => 'Any',
				]
			);
			$filterHtml .= $this->Html->tag('span', 'to');
			$filterHtml .= $this->Form->select(
				'EvShopFilter.max_price',
				$priceOptionsFormatted,
				[
					'class' => 'required js-auto-submit',
					'empty' => 'Any'
				]
			);

			// Option body close
			$filterHtml .= $this->Html->tag('/div');

			// Group container close
			$filterHtml .= $this->Html->tag('/div');
		}

		// Add the clear options button

		if (!empty($clearLink)) {
			if (empty($clearLink['name'])) {
				$clearLink['name'] = 'Clear All Filters';
			}

			$clearFilterLink = $this->Html->link(
				$clearLink['name'],
				$clearLink['url'],
				[
					'class' => 'link--clear',
					'escape' => false
				]
			);

			$filterHtml .= $this->Html->tag('div', $clearFilterLink, [
				'class' => 'menu__section menu__section--clear-filters'
			]);
		}

		$filterHtml .= $this->Form->end();

		// Collapse block close
		$filterHtml .= $this->Html->tag('/div');

		// Main container close
		$filterHtml .= $this->Html->tag('/div');

		return $filterHtml;
	}

}
