<?php
App::uses('EvShopAppModel', 'EvShop.Model');
/**
 * EvShopVariantPricing Model
 *
 * @property Variant $Variant
 * @property Currency $Currency
 */
class VariantPricing extends EvShopAppModel {

/**
 * Validation rules
 *
 * @var array
 */
	public $validate = array(
		'rrp' => array(
			'numeric' => array(
				'rule' => 'numeric',
				'message' => 'Please enter a numeric value for RRP',
				'allowEmpty' => true
			),
			'greater' => array(
				'rule' => array('comparison', '>', 0),
				'message' => 'Please enter an RRP greater than 0.00'
			)
		),
		'price' => array(
			'numeric' => array(
				'rule' => 'numeric',
				'message' => 'Please enter a numeric value for price',
			),
			'greater' => array(
				'rule' => array('comparison', '>', 0),
				'message' => 'Please enter a price greater than 0.00'
			)
		),
	);

	// The Associations below have been created with all possible keys, those that are not needed can be removed

/**
 * belongsTo associations
 *
 * @var array
 */
	public $belongsTo = array(
		'Variant' => array(
			'className' => 'EvShop.Variant'
		)
	);

	/**
	 * redefine constructor to check for EvTax and EvCurrency
	 */
	public function __construct($id = false, $table = null, $ds = null) {
		if (CakePlugin::loaded('EvCurrency')) {
			$this->actsAs['EvCurrency.Currency'] = array(
				'formInject' => true
			);

			$this->belongsTo['Currency'] = array(
				'className' => 'EvCurrency.Currency',
				'foreignKey' => 'currency_id'
			);
		}

		if (CakePlugin::loaded('EvTax')) {
			$this->actsAs['EvTax.CalculateTax'] = array(
				'fields' => array(
					'rrp',
					'price',
					'sale_price'
				),
				'TaxRateModel' => 'EvShop.Product',
				'TaxRateModelId' => 'Variant.product_id'
			);
		}

		parent::__construct($id, $table, $ds);
	}

	public function beforeSave($options = array()) {
		$this->data = $this->updateLowestPrice($this->data);

		return true;
	}

	public function afterSave($created = false, $options = array()) {
		if (isset($this->data['VariantPricing']) && Configure::read('EvShop.autoManageCurrencies') && CakePlugin::loaded('EvCurrency') && !isset($this->data['VariantPricing'][0])) {
			// We're using both EvCurrency and the autoManageCurrencies setting. This means we want
			// to automatically update all the pricing for the non-default currency using the exchange
			// rate provided by EvCurrency. EvCurrency runs its own daily updater to get the latest
			// exchange rates. We then just send a save request to the main/default price records so
			// that we can force an update on the sub-currencies.
			$CurrencyModel = EvClassRegistry::init('EvCurrency.Currency');

			// Delete all other variant pricing records so we can update them.
			$this->deleteAll(array(
				'VariantPricing.variant_id' => $this->data['VariantPricing']['variant_id'],
				'VariantPricing.currency_id !=' => $this->data['VariantPricing']['currency_id']
			), false, false);

			// Load all currencies (except our active one) so we can work out what variant prices need creating
			$Currencies = $CurrencyModel->find('all', array(
				'conditions' => array(
					'is_active' => 1,
					'is_default' => 0
				)
			));

			if (! empty($Currencies)) {
				$variantPricingRecords = array();
				foreach ($Currencies as $currency) {

					$variantPricingRecords[] = array(
						'variant_id' => $this->data['VariantPricing']['variant_id'],
						'currency_id' => $currency['Currency']['id'],
						'rrp' => ($this->data['VariantPricing']['rrp'] * $currency['Currency']['exchange_rate']),
						'price' => ($this->data['VariantPricing']['price'] * $currency['Currency']['exchange_rate']),
						'sale_price' => ((!empty($this->data['VariantPricing']['sale_price'])) ? ($this->data['VariantPricing']['sale_price'] * $currency['Currency']['exchange_rate']) : null),
						'trade_price' => ((!empty($this->data['VariantPricing']['trade_price'])) ? ($this->data['VariantPricing']['trade_price'] * $currency['Currency']['exchange_rate']) : null)
					);
				}

				if (! empty($variantPricingRecords)) {
					foreach ($variantPricingRecords as $pricingKey => $pricingRecord) {
						//updateLowestPrice expects a single record under 'VariantPricing'
						$variantPricingRecords[$pricingKey] = $this->updateLowestPrice(['VariantPricing' => $pricingRecord])['VariantPricing'];
					}
					return $this->saveAll($variantPricingRecords, array('deep' => false, 'callbacks' => false));
				}
			}
		}

		return true;
	}

	public function updateLowestPrice($data) {
		$variantPricing = $data['VariantPricing'];

		if (!empty($variantPricing)) {
			$lowestPrice = $variantPricing['price'];

			if (isset($variantPricing['sale_price']) && $variantPricing['sale_price'] > 0) {
				$lowestPrice = $variantPricing['sale_price'];
			}

			if (CakePlugin::loaded('EvTax')) {
				//Add on the tax if possible
				$taxLevel = $this->Variant->find(
					'first',
					[
						'fields' => [
							'TaxLevel.*'
						],
						'joins' => [
							[
								'table' => 'ev_shop_products',
								'alias' => 'Product',
								'conditions' => [
									'Product.id = Variant.product_id'
								]
							],
							[
								'table' => 'ev_tax_tax_levels',
								'alias' => 'TaxLevel',
								'conditions' => [
									'TaxLevel.id = Product.tax_level_id'
								]
							]
						],
						'conditions' => [
							'Variant.id' => $variantPricing['variant_id'],
						]
					]
				);

				if (!empty($taxLevel)) {
					$lowestPrice = $this->addTaxToPrice($lowestPrice, $taxLevel['TaxLevel']['rate']);
				}
			}

			$data['VariantPricing']['lowest_price'] = $lowestPrice;
		}

		return $data;
	}
}
