<?php
App::uses('EvShopAppModel', 'EvShop.Model');
App::uses('ArrayUtil', 'EvCore.Lib');
App::uses('CakeNumber', 'Utility');

/**
 * Variant Model
 *
 * @property Product $Product
 * @property VariantPricing $VariantPricing
 * @property Option $Option
 */
class Variant extends EvShopAppModel {

	public $displayName = 'Pricing';

	/**
	 * Display field
	 *
	 * @var string
	 */
	public $displayField = 'name';

	/**
	 * Validation rules
	 *
	 * @var array
	 */
	public $validate = array(
		'product_id' => array(
			'numeric' => array(
				'rule' => array('numeric')
			),
		),
		'name' => array(
			'notEmpty' => array(
				'rule' => array('notBlank'),
				'message' => 'A variant name must be entered.'
			),
		)
	);

	//The Associations below have been created with all possible keys, those that are not needed can be removed

	/**
	 * belongsTo associations
	 *
	 * @var array
	 */
	public $belongsTo = array(
		'Product' => array(
			'className' => 'EvShop.Product'
		)
	);

	/**
	 * hasMany associations
	 *
	 * @var array
	 */
	public $hasMany = array(
		'VariantPricing' => array(
			'className' => 'EvShop.VariantPricing'
		)
	);

	/**
	 * hasAndBelongsToMany associations
	 *
	 * @var array
	 */
	public $hasAndBelongsToMany = array(
		'Option' => array(
			'className' => 'EvShop.Option',
			'joinTable' => 'ev_shop_options_variants'
		)
	);

	public $findMethods = array(
		'allAndRound' => true
	);

	/**
	 * redefine constructor to check for EvCurrency
	 */
	public function __construct($id = false, $table = null, $ds = null) {
		if (CakePlugin::loaded('EvInventory')) {
			$this->actsAs[] = 'EvInventory.Inventories';
		}

		if (CakePlugin::loaded('EvCurrency')) {
			$this->actsAs['EvCurrency.Currency'] = array(
				'formInject' => true
			);
		}

		parent::__construct($id, $table, $ds);
	}

	/**
	 * get all the data and then round the values
	 *
	 * @param 	string 	state - either before or after
	 * @param 	array 	array of the query params
	 * @param 	array 	Array of results (after only)
	 * @return 	array 	$query if state is before / $results if state is after
	 */
	protected function _findAllAndRound($state, $query, $results = array()) {
		if ($state === 'after') {
			foreach ($results as $key => $result) {

				if (! empty($result['VariantPricing'])) {
					foreach ($result['VariantPricing'] as $pricing) {
						if ($pricing['currency_id'] == CakeSession::read('EvCurrency.currencyId')) {

							$results[$key]['Variant']['rrp'] = CakeNumber::format($pricing['rrp'], 2);
							$results[$key]['Variant']['price'] = CakeNumber::format($pricing['price'], 2);
							$results[$key]['Variant']['sale_price'] = CakeNumber::format($pricing['sale_price'], 2);
						}
					}
				}
			}

			return $results;
		}

		return $query;
	}

	public function afterFind($results, $primary = false) {
		foreach ($results as $resultIndex => $result) {
			if (isset($result['VariantPricing'])) {
				$results[$resultIndex]['VariantPricing'] = Hash::combine($result['VariantPricing'], '{n}.currency_id', '{n}');
			}

			// Parse options in to a single field in the form "Option: value, Option: value...etc"
			if (isset($result['Option'])) {

				$optionsBulk = Hash::combine($result['Option'], '{n}.OptionGroup.name', '{n}.name');
				$optionsFlat = array();

				foreach ($optionsBulk as $optionKey => $optionValue) {
					$optionsFlat[] = $optionKey . ': ' . $optionValue;
				}

				$results[$resultIndex]['Variant']['option_desc'] = implode(', ', $optionsFlat);
			}
		}

		return $results;
	}

	public function readForEdit($id, $query = array()) {
		if (CakePlugin::loaded('EvTax')) {
			$query['contain'][] = 'Product.TaxLevel';
		}

		return parent::readForEdit($id, $query);
	}

	/**
	 * readForManage
	 * get all the variants by product ID for the admin
	 *
	 * @param 	int 	Product ID
	 * @return 	array 	Array of Variants / Options
	 */
	public function readForManage($productId) {
		$query = array(
			'conditions' => array(
				$this->name . '.product_id' => $productId
			),
			'order' => $this->name . '.sequence ASC',
			'contain' => array(
				'VariantPricing',
				'Option.OptionGroup'
			)
		);

		if (CakePlugin::loaded('EvTax')) {
			$query['contain'][] = 'Product.TaxLevel';
		}

		$results = $this->find('allAndRound', $query);

		foreach ($results as $resultIndex => $result) {
			$results[$resultIndex] = array_merge_recursive($result, Hash::extract($result, 'Variant'));
		}

		return $results;
	}

	/**
	 * readForProductView
	 * get all the variants by product ID for the view
	 *
	 * @param 	int 	Product ID
	 * @return 	array 	Array of Variants / Options
	 */
	public function readForProductView($productId) {
		$params = array(
			'fields' => '*',
			'conditions' => array(
				'Variant.product_id' => $productId,
				'Variant.is_active' => 1
			),
			'order' => 'Variant.sequence ASC',
			'contain' => array(
				'VariantPricing' => array(
					'conditions' => array(
						'VariantPricing.currency_id' => CakeSession::read('EvCurrency.currencyId')
					)
				)
			)
		);

		if (CakePlugin::loaded('EvTax')) {
			$TaxLevelModel = EvClassRegistry::init('EvTax.TaxLevel');

			$taxLevels = Cache::remember('tax_levels', function () use ($TaxLevelModel) {
				$taxLevels = array();

				$levels = $TaxLevelModel->find('all');

				if (! empty($levels)) {
					foreach ($levels as $level) {
						$taxLevels[$level['TaxLevel']['id']] = $level;
					}
				}

				return $taxLevels;
			});
		}

		// if Option data are being fetched as part of the query and the
		// Option model has imageSlots, contain Image within Option
		if (
			isset($params['contain']['Option']) &&
			! empty(EvClassRegistry::init('EvShop.Option')->imageSlots)
		) {
			$params['contain']['Option'][] = 'Image';
		}

		$data = $this->find('all', $params);

		if (! empty($data)) {
			$variantIds = Hash::extract($data, '{n}.Variant.id');

			// Generate an array to map variant ids back to the results array
			$variantMap = array();

			foreach ($data as $key => $variant) {
				$variantMap[$variant['Variant']['id']] = $key;

				// Whilst we're here, we'll add an empty options array to the
				// variants to prevent any backwards compatability issues (e.g
				// throwing undefined indexes)

				$data[$key]['Option'] = array();
			}

			// Start loading up options for the variants
			$OptionsModel = EvClassRegistry::init('EvShop.Option');

			$optionParams = array(
				'fields' => '*',
				'joins' => array(
					array(
						'table' => 'ev_shop_options_variants',
						'alias' => 'OptionsVariant',
						'conditions' => array(
							'OptionsVariant.variant_id' => $variantIds,
							'Option.id = OptionsVariant.option_id'
						)
					),
					array(
						'table' => 'ev_shop_option_groups',
						'alias' => 'OptionGroup',
						'conditions' => array(
							'Option.option_group_id = OptionGroup.id'
						)
					),
				),
				'order' => 'Option.sequence DESC',
				'callbacks' => false
			);

			if (! empty($OptionsModel->imageSlots)) {
				$optionParams['contain'][] = 'Image';
			}

			$options = $OptionsModel->find('all', $optionParams);

			if (! empty($options)) {
				// Reformat retrieved option data into the variant data
				foreach ($options as $option) {
					$variantId = $option['OptionsVariant']['variant_id'];
					if (array_key_exists($variantId, $variantMap) && $variantMap[$variantId] >= 0) {
						$variantKey = $variantMap[$option['OptionsVariant']['variant_id']];

						// Reformat the option data to match how it would be pulled if
						// this was done as a contain on the variant.
						$optionData = $option['Option'];
						$optionData['EvShopOptionsVariant'] = $option['OptionsVariant'];
						$optionData['OptionGroup'] = $option['OptionGroup'];

						$data[$variantKey]['Option'][] = $optionData;

					}
				}
			}
		}

		if (CakePlugin::loaded('EvTax')) {

			foreach ($data as $key => $variant) {

				if (isset($variant['Product']['tax_level_id'])
					&& $variant['Product']['tax_level_id'] > 0
					&& empty($taxLevel)
					&& ! empty($taxLevels[$variant['Product']['tax_level_id']])) {
					$taxLevel = $taxLevels[$variant['Product']['tax_level_id']];
				}
				if (isset($taxLevel['TaxLevel'])) {
					foreach ($variant['VariantPricing'] as $pricingKey => $pricing) {
						$variant['VariantPricing'][$pricingKey]['TaxLevel'] = $taxLevel['TaxLevel'];
					}
				}

				if (is_array($variant)) {
					$data[$key] = $this->VariantPricing->calculateTax($variant, $this->{$this->modelAlias});
				}
			}
		}

		// Rejiggle the array so it can be the same format
		// as if the original products models contained the data
		// also jiggle the options to use option group id as key so we can easily get when displaying
		return array_map(
			function ($variant) {

				$element = $variant['Variant'];
				unset($variant['Variant']);
				$element = ($element + $variant);

				if (! empty($element['Option'])) {
					$element['Option'] = Hash::combine(
						$element['Option'],
						'{n}.option_group_id',
						'{n}'
					);
				}

				return $element;
			},
			$data
		);
	}

	/**
	 * get listing for variants management
	 *
	 * @param 	int 	Product Id
	 * @param 	array 	custom array conditions if needed
	 * @return 	array 	results
	 */
	public function manageListing($productId, $query = array()) {
		$params = array(
			'conditions' => array(
				'Variant.product_id' => $productId
			),
			'order' => 'Variant.sequence ASC, Variant.id ASC',
			'contain' => array(
				'Option',
				'Product',
				'VariantPricing'
			)
		) + $query;

		if (CakePlugin::loaded('EvTax')) {
			$params['contain'][] = 'Product.TaxLevel';
		}

		$data = $this->find(
			'allAndRound',
			$params
		);

		// Reformat variant pricings to use the currency id as the array key
		foreach ($data as $variantKey => $variant) {
			if (isset($variant['VariantPricing']) && ! empty($variant['VariantPricing'])) {
				$newPricing = array();

				foreach ($variant['VariantPricing'] as $variantPricingKey => $variantPricing) {
					$newPricing[$variantPricing['currency_id']] = $variantPricing;
				}

				$data[$variantKey]['VariantPricing'] = $newPricing;
			}
		}

		// Due to relationships cake creates more queries to get the option groups
		// in the above find - manually sort option groups in only one extra query
		// process any options into a Group: Option, Group: Option string for displaying
		if (! empty($data['0']['Option'])) {
			$optionGroups = $this->Option->OptionGroup->find(
				'all',
				array(
					'callbacks' => false
				)
			);
			$optionGroups = Hash::combine($optionGroups, '{n}.OptionGroup.id', '{n}');

			// add the option group to each option
			$data = array_map(
				function ($variant) use ($optionGroups) {
					foreach ($variant['Option'] as $key => $option) {
						$variant['Option'][$key] += $optionGroups[$option['option_group_id']];
					}

					// create a readable description of all options on a variant
					$variant['optionDesc'] = Hash::combine(
						$variant,
						'Option.{n}.OptionGroup.name',
						'Option.{n}.name'
					);
					array_walk(
						$variant['optionDesc'],
						function (&$value, $key) {
							$value = $key . ': ' . $value;
						}
					);
					$variant['optionDesc'] = implode(', ', $variant['optionDesc']);

					return $variant;
				},
				$data
			);
		}

		return $data;
	}

	/**
	 * afterSave - send out variant saved event
	 *
	 */
	public function afterSave($created, $options = array()) {
		parent::afterSave($created, $options);

		// dispatch product saved event
		$this->getEventManager()->dispatch(
			new CakeEvent('EvShop.Model.Variant.saved', $this, array(
				'newItem' => $created
			))
		);
	}

	/**
	 * get any changes in the options and return the new variants
	 *
	 * @param 	array 	Current Options
	 * @param 	array 	New Options
	 * @param 	array 	default product array data
	 * @return 	array 	array of variants ready for DB insert
	 */
	public function processNewOptions($currentOptions, $newOptions, $product) {
		if (empty($newOptions)) {
			// options are empty but do we have an id number already?
			if (! empty($product['id']) && empty($currentOptions)) {
				// yes? it will already have a "variant"
				return array();
			}

			return $this->defaultInsertArray($product['name']);
		}

		$options = $this->getNewOptions(
			array(
				'CurrentOptions' => $currentOptions,
				'Options' => $newOptions
			)
		);

		$variants = array();

		if (! empty($options)) {
			foreach ($options as $optionIds) {
				$variants += $this->buildVariants(
					$optionIds
				);
			}
		}

		return $variants;
	}

	/**
	 * calculate any new options
	 *
	 * @param 	array 	Array of Options / CurrentOptions elements
	 * @return 	array 	Options to build variants for
	 */
	public function getNewOptions($data) {
		// check if the main option groups match
		$currentOptions = array_keys($data['CurrentOptions']);
		sort($currentOptions);
		$options = array_keys($data['Options']);
		sort($options);

		// rebuild all the selected options if they don't match
		if ($currentOptions !== $options) {
			return array($data['Options']);
		}

		$returnOptions = array();

		// the option groups are the same, check which options have been added
		// i.e. adding more colours
		foreach ($data['Options'] as $groupId => $optionIds) {
			foreach ($optionIds as $id => $value) {

				if (! isset($data['CurrentOptions'][$groupId][$id])) {
					$returnOptions[$groupId][$groupId][$id] = $id;
				}
			}
		}

		// add to the returning optins array
		// all the other groups that haven't changed so we can build all the new variant combinations
		if (! empty($returnOptions)) {
			foreach ($returnOptions as $section => $group) {
				$options = $data['Options'];

				unset($options[key($group)]);
				$returnOptions[$section] += $options;
			}
		}

		return $returnOptions;
	}

	/**
	 * take the multiple option arrays and build variants
	 *
	 * @param 	array 	array of option ids
	 * @return 	array 	array of variants ready for DB insertion
	 */
	public function buildVariants($options) {
		if (empty($options) || ! is_array($options)) {
			return array();
		}

		$Option = EvClassRegistry::init('EvShop.Option');
		$optionNames = $Option->getOptionNames(
			Hash::flatten($options)
		);

		$cartesian = ArrayUtil::cartesianProduct($options);

		return $this->generateInsertArray(
			$cartesian,
			$optionNames
		);
	}

	/**
	 * from the cartesian array generate db insert array
	 *
	 * @param 	array 	Cartesian product arrays
	 * @return 	array 	Array with variant name and option links
	 */
	public function generateInsertArray($cartesianProduct, $optionNames) {
		$variants = array();

		foreach ($cartesianProduct as $optionIds) {
			$name = array_intersect_key(
				$optionNames,
				array_flip($optionIds)
			);

			$variants[] = array(
				'name' => implode(' ', $name),
				'is_active' => 1,
				'Option' => array(
					'Option' => $optionIds
				)
			);
		}

		return $variants;
	}

	/**
	 * generate a default Insert Array for a product with no options
	 * variant = product name
	 *
	 * @param 	string 	Product name
	 * @return 	array 	Array with variant data for insert
	 */
	public function defaultInsertArray($productName) {
		return array(
			array(
				'name' => $productName,
				'is_active' => 1,
			)
		);
	}

	/**
	 * calculate the changes to options that require variants to be deleted
	 *
	 * @param 	array 	Array of Options / CurrentOptions elements
	 * @return 	array 	Options to remove
	 */
	public function getDeletedOptions($data) {
		// check if the main option groups match
		$currentOptions = array_keys($data['CurrentOptions']);
		sort($currentOptions);
		$options = array_keys($data['Options']);
		sort($options);

		// flag for deletion if not
		if ($currentOptions !== $options) {
			return Hash::flatten($data['CurrentOptions']);
		}

		// the option groups are the same, check which options have been removed
		$deletedOptions = array();

		foreach ($data['CurrentOptions'] as $groupId => $optionIds) {
			foreach ($optionIds as $id => $value) {

				if (! isset($data['Options'][$groupId][$id])) {
					$deletedOptions[] = $id;
				}
			}
		}

		return $deletedOptions;
	}

	/**
	 * process the deleted options and get the variants to delete
	 *
	 * @param 	array 	Current Options
	 * @param 	array 	New Options
	 * @param 	int 	product Id
	 * @return 	array 	Array of all the variants to delete
	 */
	public function processDeletedOptions($currentOptions, $newOptions, $productId) {
		$options = $this->getDeletedOptions(
			array(
				'CurrentOptions' => $currentOptions,
				'Options' => $newOptions
			)
		);

		/*
			if options are empty a few situations could be happening.
			- if productId is empty then likely it's a new product options is empty as there is nothing to delete = return null
			- productId is not empty to either
			- - both currentOptions and newOptions were equal, meaning no options changed = return null
			- - or currentOptions is empty and newOptions has been checked (In this instance we need to delete the single variant created for the product) = proceed to getting variant id for delete
		*/
		if (empty($options)) {
			if (empty($productId)) {
				return null;
			}

			if (
				(empty($currentOptions) && empty($newOptions)) ||
				(! empty($currentOptions) && ! empty($newOptions))
			) {
				return null;
			}
		}

		// if we're here proceed and get the variants to delete
		$params = array(
			'conditions' => array(
				'Variant.product_id' => $productId

			),
			'callbacks' => false
		);

		// only join options if options are involved
		// could be a single variant product (i.e. one with no options)
		if (! empty($options)) {
			$params['joins'] = array(
				array(
					'table' => 'ev_shop_options_variants',
					'alias' => 'OptionsVariant',
					'conditions' => array(
						'Variant.id = OptionsVariant.variant_id'
					)
				)
			);
			$params['conditions']['OptionsVariant.option_id'] = $options;
		}

		return $this->find(
			'list',
			$params
		);
	}

	/**
	 * get the unit price of the variant
	 * used by EvBasket
	 *
	 * @param 	int 	$itemId 	itemId
	 * @return 	float|bool
	 */
	public function getUnitPrice($itemId) {
		$Variant = $this->find(
			'first',
			array(
				'conditions' => array(
					'Variant.id' => $itemId
				),
				'contain' => array(
					'VariantPricing' => array(
						'conditions' => array(
							'currency_id' => CakeSession::read('EvCurrency.currencyId')
						)
					)
				)
			)
		);
		if (empty($Variant)) {
			return false;
		}

		if (empty($Variant['VariantPricing'])) {
			return false;
		}

		// Work out what (pre-tax) price the customer is paying. This could be any of the following:
		// - price
		// - sale price
		// - trade price

		//Find the most appropriate VariantPricing
		$currencyId = CakeSession::read('EvCurrency.currencyId');

		$possiblePricings = Hash::combine($Variant['VariantPricing'], '{n}.currency_id', '{n}');
		if (!empty($possiblePricings[$currencyId])) {
			$VariantPricing = $possiblePricings[$currencyId];

			// First, check if the customer is a trade customer, and if trade pricing is enabled
			if (CakeSession::read('EvShop.isTrade') && Configure::read('SiteSetting.ev_shop.enable_trade_pricing') == '1'):
				return (float)$VariantPricing['trade_price'];
			endif;

			// If we get to this point, we're not using trade pricing so fall back to standard pricing checks
			$salePrice = (float)$VariantPricing['sale_price'];
			if (! empty($salePrice)) {
				return $VariantPricing['sale_price'];
			}

			return $VariantPricing['price'];
		} else {
			return false;
		}
	}

	/**
	 * get the tax rate for the variant
	 * used by EvBasket
	 *
	 * @param 	int 	$itemId 	item ID
	 * @return 	float
	 */
	public function getTaxRate($itemId) {
		if (! CakePlugin::loaded('EvTax')) {
			return 0;
		}

		$VariantPricing = $this->VariantPricing->find(
			'first',
			array(
				'conditions' => array(
					'VariantPricing.variant_id' => $itemId,
					'currency_id' => CakeSession::read('EvCurrency.currencyId')
				),
				'contain' => 'Variant.Product'
			)
		);

		if (isset($VariantPricing[0]['VariantPricing'])) {
			$VariantPricing = $VariantPricing[0];
		}

		if (empty($VariantPricing) || empty($VariantPricing['Variant']['Product']['tax_level_id'])) {
			return 0;
		}

		$TaxLevel = EvClassRegistry::init('EvTax.TaxLevel');
		$taxRate = $TaxLevel->fieldOnly(
			'rate',
			array(
				'TaxLevel.id' => $VariantPricing['Variant']['Product']['tax_level_id']
			)
		);

		// No tax rate set
		if (is_null($taxRate)) {
			return 0;
		}

		// If the customer is a trade customer, and trade pricing is enabled, AND tax is disabled
		// on trade accounts then we'll want to return zero.
		if (CakeSession::read('EvShop.isTrade') && Configure::read('SiteSetting.ev_shop.enable_trade_pricing') == '1'
			&& Configure::read('SiteSetting.ev_shop.disable_tax_on_trade_pricing') == '1'):
			return 0;
		endif;

		return $taxRate;
	}

}
