<?php
App::uses('EvShopAppModel', 'EvShop.Model');
App::uses('CakeNumber', 'Utility');
/**
 * Product Model
 *
 * @property Brand $Brand
 * @property Variant $Variant
 * @property Category $Category
 */
class Product extends EvShopAppModel {

	public $actsAs = array(
		'Routable.Routable' => array(
			'config' => 'EvShop',
			'alias' => 'product/:displayField'
		),
		'MetaData.Meta',
		'EvTemplates.Template' => array(
			'formInject' => true,
			'restrictTo' => 'Product'
		)
	);

	/**
	 * Display field
	 *
	 * @var string
	 */
	public $displayField = 'name';

	/**
	 * Validation rules
	 *
	 * @var array
	 */
	public $validate = array(
		'name' => array(
			'notEmpty' => array(
				'rule' => array('notBlank'),
				'message' => 'A product name must be entered'
			),
		),
	);

	//The Associations below have been created with all possible keys, those that are not needed can be removed

	/**
	 * belongsTo associations
	 *
	 * @var array
	 */
	public $belongsTo = array(
		'Brand' => array(
			'className' => 'EvShop.Brand'
		),
	);

	/**
	 * hasMany associations
	 *
	 * @var array
	 */
	public $hasMany = array(
		'Variant' => array(
			'className' => 'EvShop.Variant',
			'cascade' => true,
			'dependent' => true
		),
		'CategoriesProduct' => array(
			'className' => 'EvShop.CategoriesProduct',
			'order' => 'CategoriesProduct.sequence ASC'
		),
	);

	/**
	 * image slots
	 */
	public $imageSlots = array(
		'listing' => array(
			'slots' => 1,
			'fields' => false
		)
	);

	/**
	 * redefine constructor to check for EvTax
	 */
	public function __construct($id = false, $table = null, $ds = null) {
		if (CakePlugin::loaded('EvTax')) {
			$this->actsAs['EvTax.Taxable'] = array(
				'formInject' => true
			);

			$this->belongsTo['TaxLevel'] = array(
				'className' => 'EvTax.TaxLevel'
			);
		}

		parent::__construct($id, $table, $ds);
	}

	/**
	 * before we save anything process the options into variants
	 * also process the category array
	 * also process the variant images
	 */
	public function beforeBeforeSave($data, $options) {
		$Variant = EvClassRegistry::init('EvShop.Variant');

		if (empty($data['CurrentOptions'])) {
			$data['CurrentOptions'] = array();
		}
		if (empty($data['Options'])) {
			$data['Options'] = array();
		}

		$data['Options'] = array_map(
			function ($options) {
				return array_combine(
					$options,
					$options
				);
			},
			$data['Options']
		);

		//Match the current options array with the selected options array
		$data['CurrentOptions'] = array_map(
			function ($options) {
				return array_combine(
					$options,
					$options
				);
			},
			$data['CurrentOptions']
		);

		// Take a copy of any current Variant data so it doesn't get overwritten later
		$originalVariants = array();
		if (isset($data['Variant'])) {
			$originalVariants = $data['Variant'];
		}

		$this->variantsToDelete = $Variant->processDeletedOptions($data['CurrentOptions'], $data['Options'], $data[$this->alias][$this->primaryKey]);
		$data['Variant'] = $Variant->processNewOptions($data['CurrentOptions'], $data['Options'], $data[$this->alias]);
		unset($data['Options']);

		// Merge the original into the list if it is not on the list to delete
		// This stops old redundant variants being recreated as variant save happens after product afterSave deletes the variants from variantsToDelete
		$idsToDelete = !empty($this->variantsToDelete) ? array_keys($this->variantsToDelete) : [];
		if (!empty($originalVariants)) {
			foreach ($originalVariants as $variant) {
				if (!in_array($variant['id'], $idsToDelete)) {
					$data['Variant'][] = $variant;
				}
			}
		}

		// process the categories
		$Category = EvClassRegistry::init('EvShop.Category');
		$this->categoriesToDelete = $Category->processDeletedCategories($data['CategoriesProduct']);

		// process the variant images
		if (isset($this->variantImageSlots)) {
			$VariantImage = EvClassRegistry::init('EvShop.VariantImage');
			$data = $VariantImage->processVariantImageOptions($data, $this->variantImageSlots);
		}

		return $data;
	}

	/**
	 * afterSave - delete any variants to be deleted
	 *
	 */
	public function afterSave($created, $options = array()) {
		parent::afterSave($created, $options);

		if (! empty($this->variantsToDelete)) {
			$Variant = EvClassRegistry::init('EvShop.Variant');

			$Variant->deleteAll(
				array(
					'Variant.id' => array_keys($this->variantsToDelete)
				)
			);
		}

		if (! empty($this->categoriesToDelete)) {
			$CategoriesProduct = EvClassRegistry::init('EvShop.CategoriesProduct');

			$CategoriesProduct->deleteAll(
				array(
					'CategoriesProduct.id' => $this->categoriesToDelete
				)
			);
		}

		// dispatch product saved event
		$this->getEventManager()->dispatch(
			new CakeEvent('EvShop.Model.Product.saved', $this, array(
				'created' => $created,
				'product' => $this->data['Product']
			))
		);
	}

	/**
	 * afterFind - Calculate tax prices for each variant
	 *
	 */

	public function afterFind($results, $primary = false) {
		if (!empty($results)) {
			if (CakePlugin::loaded('EvTax')) {
				$results = $this->calculateTaxPrices($results);
			}
		}

		return $results;
	}

	/**
	 * readForEdit - bring out the variants and calculate the options
	 *
	 */
	public function readForEdit($id, $query = array()) {
		$query['contain']['Variant'] = [
			'fields' => [
				'Variant.id'
			]
		];

		$query['contain']['CategoriesProduct'] = array('Category');

		$data = parent::readForEdit($id, $query);

		$data['CategoriesProduct'] = Hash::combine(
			$data['CategoriesProduct'],
			'{n}.category_id',
			'{n}'
		);

		if (! empty($data['Variant'])) {
			// when there is one variant this can be edited directly from the
			// admin edit tempalte - fetch the relevant data
			//
			// when more than one vatiant is available a link is shown and
			// the variants cannot be edited directly from the template so
			// we only need product option ids, grouped by option groups
			if (count($data['Variant']) == 1) {
				$data['Variant'] = $this->Variant->readForManage($id);

				$data['Options'] = Hash::combine(
					$data['Variant'],
					'{n}.Option.{n}.id',
					'{n}.Option.{n}.id',
					'{n}.Option.{n}.option_group_id'
				);
			} else {
				// Prep options array for in-page editing
				$data['Options'] = $this->Variant->Option->readForProductEdit($id);
			}
		}

		return $data;
	}

	/**
	 * readForView - Do the Variants separately so the callbacks will be run
	 *
	 * @param integer $id ID of row to edit
	 * @param array $params The db query array - can be used to pass in additional parameters such as contain
	 * @return array
	 */
	public function readForView($id, $query = array()) {
		$query['contain']['CategoriesProduct'] = array('Category');

		$behaviors = $this->CategoriesProduct->Behaviors->loaded();
		array_walk($behaviors, [$this->CategoriesProduct, 'removeBehavior']);

		if (Configure::read('EvShop.showBrands')) {
			$query['contain'][] = 'Brand';

			$behaviors = $this->Brand->Behaviors->loaded();
			array_walk($behaviors, [$this->Brand, 'removeBehavior']);
		}

		$data = parent::readForView($id, $query);

		if (empty($data)) {
			return false;
		}

		// perform this on it's own rather then contain so the inventory behaviours / cakllbacks can be triggered
		$data['Variant'] = $this->Variant->readForProductView($id);

		if (! empty($data['Variant'])) {
			$data['Options'] = Hash::combine($data['Variant'], '{n}.Option.{n}.id', '{n}.Option.{n}.id', '{n}.Option.{n}.option_group_id');
		}

		if (! empty($data['CategoriesProduct'])) {
			$data['CategoriesProduct'] = Hash::combine($data['CategoriesProduct'], '{n}.category_id', '{n}');
		}

		$data = $this->defaultMetaDescription($data);

		return $data;
	}

	/**
	 * unless the meta description has been overridden
	 * set seo product meta description
	 *
	 * @param 	array 	$data 	Array of product data
	 * @return 	array 	$data 	Updated product data array with updated meta description
	 */
	public function defaultMetaDescription($data) {
		if (! isset($data['MetaData']['description']) || empty($data['MetaData']['description'])) {
			$variantCount = count($data['Variant']);
			$siteTitle = Configure::read('SiteSetting.general.site_title');

			if ($variantCount > 1) {
				$prices = Hash::extract($data, 'Variant.{n}.price');
				foreach ($prices as $key => $price) {
					if (empty($price) || $price == '0.000000') {
						unset($prices[$key]);
					}
				}
				sort($prices);

				if (! empty($prices['0']) && (float)$prices['0'] > 0) {
					$price = CakeNumber::currency($prices['0']);

					$description = "Order " . $data['Product']['name'] . " from only " . $price . " from " . $siteTitle . " now!";
				}
			} elseif ($variantCount === 1) {
				$currencies = Configure::read('currencies');
				if (! isset($data['Variant'][0]['VariantPricing'][0]['price'])) {
					$price = 0;
				} else {
					$price = CakeNumber::currency($data['Variant'][0]['VariantPricing'][0]['price'], $currencies[CakeSession::read('EvCurrency.currencyId')]);
				}
				$description = "Order " . $data['Product']['name'] . " for only " . $price . " from " . $siteTitle . " now!";
			}

			if (! empty($description)) {
				$data['MetaData']['description'] = $description;
			}
		}

		return $data;
	}

	/**
	 * Calculate the prices with tax for each variant
	 *
	 * @param array $result The results from the find
	 * @return array $result Updated variant pricing to include tax prices
	 */

	public function calculateTaxPrices($result) {
		$TaxLevelModel = EvClassRegistry::init('EvTax.TaxLevel');
		$VariantPricingModel = EvClassRegistry::init('EvShop.VariantPricing');

		if (! empty($result)) {
			foreach ($result as $key => $item) {
				$taxLevel = array();

				if (isset($item['Product']['tax_level_id']) && $item['Product']['tax_level_id'] > 0) {

					$taxLevel = $TaxLevelModel->find('first', array(
						'conditions' => array(
							'id' => $item['Product']['tax_level_id']
						)
					));

					// As well as the variants add tax to the product 'summary' prices
					if (isset($item['Product']['was_price_ex_vat'])) {
						if (isset($taxLevel['TaxLevel']['rate'])) {
							$result[$key]['Product']['was_price'] = $VariantPricingModel->addTaxToPrice($item['Product']['was_price_ex_vat'], $taxLevel[$TaxLevelModel->alias]['rate']);
						} else {
							$result[$key]['Product']['was_price'] = $item['Product']['was_price_ex_vat'];
						}
					}
					if (isset($item['Product']['rrp_price_ex_vat'])) {
						if (isset($taxLevel['TaxLevel']['rate'])) {
							$result[$key]['Product']['rrp_price'] = $VariantPricingModel->addTaxToPrice($item['Product']['rrp_price_ex_vat'], $taxLevel[$TaxLevelModel->alias]['rate']);
						} else {
							$result[$key]['Product']['rrp_price'] = $item['Product']['rrp_price_ex_vat'];
						}
					}

					if (isset($item['Variant']) && ! empty($item['Variant'])) {
						foreach ($item['Variant'] as $vKey => $variant) {

							if (! empty($taxLevel['TaxLevel'])) {

								if (isset($variant['VariantPricing']) && ! empty($variant['VariantPricing'])) {

									foreach ($variant['VariantPricing'] as $pricingKey => $pricing) {
										$variant['VariantPricing'][$pricingKey]['TaxLevel'] = $taxLevel['TaxLevel'];
									}
								}
							}

							if (isset($result[$key]['Variant'][$vKey])) {
								$result[$key]['Variant'][$vKey] = $VariantPricingModel->calculateTax($variant, 'Variant');
							}
						}
					}
				}
			}
		}

		return $result;
	}
}
