<?php
App::uses('EvShopAppController', 'EvShop.Controller');
App::uses('Folder', 'Utility');
App::uses('File', 'Utility');
App::uses('InflectorExt', 'EvInflector.Lib');

class ProductsController extends EvShopAppController {

	/**
	 * redefine to allow the index / view page through the auth component!
	 */
	public function beforeFilter() {
		$this->adminActions[] = 'admin_attention';

		parent::beforeFilter();

		$this->Auth->allow(array('index', 'view'));
	}

	protected function _adminFilterFields() {
		$fields = parent::_adminFilterFields();
		$CategoryModel = EvClassRegistry::init('EvShop.Category');
		$fields['CategoriesProduct.category_id'] = array(
			'label' => 'Category',
			'type' => 'select',
			'options' => $CategoryModel->getForDropDown(),
			'compare' => array(
				'CategoriesProduct.category_id' => "%s"
			)
		);

		return $fields;
	}

	protected function _adminIndexPaginate() {
		$paginate = parent::_adminIndexPaginate();

		$paginate['joins'][] = array(
			'table' => 'ev_shop_categories_products',
			'alias' => 'CategoriesProduct',
			'type' => 'LEFT',
			'conditions' => array(
				'Product.id = CategoriesProduct.product_id',
			)
		);

		$paginate['group'] = 'Product.id';

		return $paginate;
	}

	/**
	 * product index page
	 * list all products out
	 *
	 */
	public function index() {
		$this->view = 'EvShop./Fallbacks/Products/index';
		$pageId = Configure::read('EvShop.pageIds.product');

		// check if we have a specific content page set
		// if so - set the template
		if (! empty($pageId)) {
			$pageData = $this->assignPage($pageId);

			if (! empty($pageData)) {
				$this->view = $this->Tpl->getTemplate($pageData, 'Page');
			}
		}

		$this->set(
			'listing',
			$this->Products->listing(array(), false)
		);

		$this->Breadcrumb->addCrumb(__('All Products'));
	}

	/**
	 * product view details page
	 *
	 * @param 	int 	Product ID
	 */
	public function view($id) {
		$this->view = 'EvShop./Fallbacks/Products/view';

		$data = $this->{$this->modelClass}->readForView($id);
		if (empty($data)) {
			throw new NotFoundException();
		}

		// extract the lowest product variant price
		if (! empty($data['Variant'])) {
			$data['Product']['price'] = $this->Products->extractLowestVariantPrice(
				$data['Variant']
			);
		}

		$this->set('data', $data);
		$this->Meta->set($data);

		if (! empty($data)) {
			$dataTemplate = $this->Tpl->getTemplate($data);
			if ($dataTemplate != false) {
				$this->view = $dataTemplate;
			}
		}

		$this->_getProductBreadcrumb($data);
	}

	/**
	 * display all the products that don't have missing variant pricing,
	 * if a product has no variant pricings associated with it then they will not be displayed
	 * in listings.
	 * @return [type] [description]
	 */
	public function admin_attention() {
		$Model = $this->{$this->modelClass};
		$modelAlias = $Model->alias;

		$this->set('data', $Model->find(
			'all',
			[
				'joins' => [
					[
						'table' => 'ev_shop_categories_products',
						'alias' => 'CategoriesProduct',
						'type' => 'LEFT',
						'conditions' => array(
							'Product.id = CategoriesProduct.product_id',
						)
					],
					[
						'table' => 'ev_shop_variants',
						'alias' => 'Variant',
						'conditions' => [
							'Variant.product_id = Product.id',
						]
					],
					[
						'table' => 'ev_shop_variant_pricings',
						'alias' => 'VariantPricing',
						'type' => 'left',
						'conditions' => [
							'VariantPricing.variant_id = Variant.id'
						]
					]
				],
				'conditions' => [
					'AND' => [
						'Variant.is_active' => true,
						'VariantPricing.id IS NULL'
					]
				],
				'group' => 'Product.id',
			]
		));

		$this->_processMultiEdit();
		$this->_adminPopulateFilterLookups();
		$this->set('filter', $this->_adminFilterFields());
		$this->set('columns', $this->_adminIndexColumns());
		$this->set('toolbar', $this->_adminAttentionToolbar());
		$this->set('actions', $this->_adminIndexActions());
		$this->set('multiEditActions', $this->_adminMultiEditActions());

		$this->set('title_for_layout', InflectorExt::camelToPluralize($Model->displayName));

		$this->view = 'EvShop./Products/admin_attention';
	}

	/**
	 * Defines the toolbar buttons displayed in admin_index
	 *
	 * @return array
	 */
	protected function _adminIndexToolbar() {
		$toolbar = parent::_adminIndexToolbar();

		$toolbar['Products That Need Attention'] = [
			'url' => [
				'action' => 'admin_attention'
			],
			'class' => 'toolbar__btn--listing'
		];

		return $toolbar;
	}

	protected function _adminAttentionToolbar() {
		return $toolbar = [
			'Add New' => [
				'url' => [
					'action' => 'add'
				]
			],
			'All Products' => [
				'url' => [
					'action' => 'index'
				],
				'class' => 'toolbar__btn--listing'
			]
		];
	}

	/**
	 * add the variant management link
	 * prices / stock control (if active)
	 */
	protected function _adminIndexActions() {
		if (in_array('EvShopVariants', App::objects('Controller'))) {
			$variantManageUrl = array(
				'plugin' => false,
				'controller' => 'ev_shop_variants',
				'action' => 'manage'
			);
		} else {
			$variantManageUrl = array(
				'plugin' => 'ev_shop',
				'controller' => 'variants',
				'action' => 'manage'
			);
		}

		$actions = array(
			'Pricing' => array(
				'cell' => array(
					'class' => 'action pricing'
				),
				'link' => array(
					'url' => $variantManageUrl,
					'text' => '<i class="fa fa-money"></i>',
					'options' => array(
						'escape' => false
					)
				)
			)
		) + parent::_adminIndexActions();

		return $actions;
	}

	/**
	 * form fields -check for brands setting and auto remove
	 */
	protected function _adminFormFields() {
		$fields = parent::_adminFormFields();

		$fields['Product.description']['type'] = 'html';

		if (! Configure::read('EvShop.showBrands')) {
			unset($fields['Product.brand_id']);
		}

		return $fields;
	}

	/**
	 * admin populate lookups
	 */
	protected function _adminPopulateLookups() {
		$Model = $this->{$this->modelClass};
		parent::_adminPopulateLookups();

		if (Configure::read('EvShop.showBrands')) {
			$this->set(
				'brands',
				$Model->Brand->getForDropDown()
			);
		}
	}

	protected function _adminFormToolbar($id = null) {
		$Model = $this->{$this->modelClass};
		$modelAlias = $Model->alias;

		$actions['Copy'] = array(
			'url' => array('action' => 'copy', $id),
			'class' => 'toolbar__btn--copy'
		);

		$actions['Edit Variants'] = array(
			'url' => array(
				'plugin' => 'ev_shop',
				'controller' => 'variants',
				'action' => 'manage',
				$id
			),
			'class' => 'toolbar__btn--edit'
		);

		return array_merge($actions, parent::_adminFormToolbar($id));
	}

	/**
	 * Copies an item - uses a custom copy method as this is too complex for the copyable behaviour
	 * to handle.
	 *
	 * @param integer $id id of item to copy
	 * @return void
	 */
	public function admin_copy($id) {
		$Model = $this->{$this->modelClass};
		$modelAlias = $Model->alias;

		$CategoriesProduct = EvClassRegistry::init('EvShop.CategoriesProduct');
		$Variant = EvClassRegistry::init('EvShop.Variant');
		$VariantPricing = EvClassRegistry::init('EvShop.VariantPricing');
		$OptionsVariant = EvClassRegistry::init('EvShop.OptionsVariant');

		if (CakePlugin::loaded('EvInventory')) {
			$InventoryModel = EvClassRegistry::init('EvInventory.Inventory');
		}

		if (!empty($this->request->data)) {

			// Copy product
			$originalProduct = $Model->find('first', array(
				'conditions' => array(
					'id' => $id
				)
			));

			if (empty($originalProduct)) {
				$this->Session->setFlash(array(
					'title' => 'Copy failed',
					'description' => 'Failed to copy ' . InflectorExt::humanize($Model->displayName)
				), 'flash_fail');

				return $this->redirect(array('action' => 'edit', $id));
			}

			$newProduct = $originalProduct;
			unset($newProduct['Product']['id']);
			unset($newProduct['Product']['created']);
			unset($newProduct['Product']['modified']);

			$newProduct['Product']['name'] = $this->request->data['Product']['name'];
			$newProduct['Product']['is_active'] = false;

			$Model->save($newProduct);
			$newProductId = $Model->getInsertID();

			// Load original product categories
			$categories = $CategoriesProduct->find('all', array(
				'conditions' => array(
					'product_id' => $id
				)
			));

			if (! empty($categories)) {
				$newCategories = array();
				foreach ($categories as $key => $category) {
					$newCategories[$key] = $category;
					unset($newCategories[$key]['CategoriesProduct']['id']);
					$newCategories[$key]['CategoriesProduct']['product_id'] = $newProductId;
				}
				$CategoriesProduct->clear();
				$CategoriesProduct->saveMany($newCategories);
			}

			// Load the original product's variants
			$variantParams = array(
				'conditions' => array(
					'product_id' => $id
				),
				'contain' => array(
					'VariantPricing'
				)
			);

			if (CakePlugin::loaded('EvInventory')) {
				// Inventory plugin is in use, pull in inventory levels for each variant.
				$variantParams['contain'][] = 'Inventory';
			}

			$originalVariants = $Variant->find('all', $variantParams);
			$originalVariantIds = array();

			if (! empty($originalVariants)) {
				$newVariants = array();
				foreach ($originalVariants as $key => $variant) {
					$originalVariantIds[] = $variant['Variant']['id'];
					$newVariants[$key] = $variant;

					unset($newVariants[$key]['Variant']['id']);

					// Check if we need to change the variant name
					if ($newVariants[$key]['Variant']['name'] == $originalProduct['Product']['name'] || count($originalVariants) == 1) {
						$newVariants[$key]['Variant']['name'] = $this->request->data['Product']['name'];
					}

					$newVariants[$key]['Variant']['product_id'] = $newProductId;

					if (isset($newVariants[$key]['Inventory'])) {
						unset($newVariants[$key]['Inventory']);
					}

					$Variant->clear();
					$Variant->save($newVariants[$key]);
					$newVariantId = $Variant->getInsertID();

					// Variant Pricing
					if (! empty($variant['VariantPricing'])) {

						foreach ($variant['VariantPricing'] as $pricingKey => $pricing) {

							$newVariantPricing = array();
							$newVariantPricing['VariantPricing'] = $pricing;
							unset($newVariantPricing['VariantPricing']['id']);
							$newVariantPricing['VariantPricing']['variant_id'] = $newVariantId;
							$VariantPricing->clear();
							$VariantPricing->save($newVariantPricing, array('callbacks' => false));
						}
					}

					// Check if we need to copy the inventory levels (if inventory is enabled)
					if (CakePlugin::loaded('EvInventory')) {
						if ($this->request->data['copyInventory'] == '1') {
							if (isset($variant['Inventory']) && ! empty($variant['Inventory'])) {
								$newInventory = $variant['Inventory'];
								unset($newInventory['id']);
								$newInventory['model_id'] = $newVariantId;
								$newInventory['warning_action'] = json_encode($newInventory['warning_action']);
								$newInventory['oos_action'] = json_encode($newInventory['oos_action']);
								$InventoryModel->clear();
								$InventoryModel->save($newInventory);
							}
						}
					}
				}
			}

			// Load the original product's variant options
			$originalOptionVariants = $OptionsVariant->find('all', array(
				'conditions' => array(
					'id' => $originalVariantIds
				)
			));

			if (! empty($originalOptionVariant)) {
				$newOptionVariants = array();
				foreach ($originalOptionVariants as $key => $item) {
					$newOptionVariants[$key] = $item;
					unset($newOptionVariants[$key]['OptionsVariant']['id']);
				}
			}

			// Copy metadata over
			if (CakePlugin::loaded('MetaData')) {
				$MetaDataModel = EvClassRegistry::init('MetaData.MetaData');
				$originalMetaData = $MetaDataModel->find('first', array(
					'conditions' => array(
						'model' => 'Product',
						'model_id' => $id
					)
				));

				if (! empty($originalMetaData)) {

					$MetaDataModel->clear();
					$newMetaData = $originalMetaData['MetaData'];
					$newMetaData['id'] = $newProductId;
					$MetaDataModel->save($newMetaData);
				}
			}

			// Check if images need copying over
			if (isset($this->request->data['copyImages']) && $this->request->data['copyImages'] == '1') {

				$ImagesModel = EvClassRegistry::init('EvCore.Image');

				$originalImages = $ImagesModel->find('all', array(
					'conditions' => array(
						'model' => 'Product',
						'model_id' => $id
					)
				));

				if (! empty($originalImages)) {

					foreach ($originalImages as $originalImage) {
						$newImage = $originalImage;
						unset($newImage['Image']['id']);
						$newImage['Image']['model_id'] = $newProductId;

						$ImagesModel->clear();
						$ImagesModel->save($newImage);
						$newImageId = $ImagesModel->getInsertID();

						// Duplicate the entire original image directory
						$originalImgDir = new Folder(WWW_ROOT . 'files' . DS . 'image' . DS . $originalImage['Image']['id']);
						$newImgDir = WWW_ROOT . 'files' . DS . 'image' . DS . $newImageId;

						$originalImgDir->copy($newImgDir);
					}
				}
			}

			if (CakePlugin::loaded('EvRelatedItems') && isset($this->request->data['copyRelatedItems']) && $this->request->data['copyRelatedItems'] == '1') {
				$RelatedItemsModel = EvClassRegistry::init('EvRelatedItems.RelatedItem');

				$originalRelatedItems = $RelatedItemsModel->find('all', array(
					'conditions' => array(
						'model' => 'EvShop.Product',
						'model_id' => $id
					)
				));

				if (! empty($originalRelatedItems)) {
					$relatedItems = array();
					foreach ($originalRelatedItems as $key => $originalRelatedItem) {
						$relatedItems[$key] = $originalRelatedItem;
						unset($relatedItems[$key]['RelatedItem']['id']);
						$relatedItems[$key]['RelatedItem']['model_id'] = $newProductId;
					}
					$RelatedItemsModel->clear();
					$RelatedItemsModel->saveAll($relatedItems);
				}

			}

			$this->Session->setFlash(array(
				'title' => InflectorExt::humanize($Model->displayName) . " copied",
				'description' => 'Item has been successfully copied!'
			), 'flash_success');

			return $this->redirect(array('action' => 'edit', $newProductId));
		}

		$this->request->data = $Model->findById($id);

		if (isset($this->request->data[$modelAlias]['is_protected']) && $this->request->data[$modelAlias]['is_protected']) {

			return $this->redirect(array('action' => 'index'));
		}

		$this->set('title_for_layout', 'Copy ' . InflectorExt::humanize($Model->displayName));

		// Set the default template, this can be overridden in controllers using $this->view
		// or $this->render().
		$this->view = 'admin_copy';

		return;
	}

	/**
	 * redefine admin edit to setup inject form
	 * for variants
	 */
	public function admin_edit($id = null) {
		$Model = $this->{$this->modelClass};

		$this->toInject('components', 'EvShop.Categories');
		$this->toInject('helpers', 'EvShop.Categories');

		if (Configure::read('EvShop.showVariants')) {
			$this->toInject('components', 'EvShop.Variants');
			$this->toInject('helpers', 'EvShop.Variants');
		}

		// calculate the redirect to allow pricing redirect if needed
		if ($id === null) {
			$nextId = $Model->find(
				'first',
				array(
					'fields' => 'id',
					'order' => $Model->alias . '.id DESC',
					'callbacks' => false
				)
			);
			$nextId = Hash::get($nextId, $Model->alias . '.id');

			if ($nextId > 0) {
				$nextId++;
			} else {
				$nextId = 1;
			}

			if (in_array('EvShopVariants', App::objects('Controller'))) {
				$this->adminRedirect = array(
					'plugin' => false,
					'controller' => 'ev_shop_variants',
				);
			} else {
				$this->adminRedirect = array(
					'plugin' => 'ev_shop',
					'controller' => 'variants'
				);
			}

			$this->adminRedirect['action'] = 'manage';
			$this->adminRedirect[] = $nextId;
		} else {

			if (! empty($this->request->data['pricing'])) {
				if (in_array('EvShopVariants', App::objects('Controller'))) {
					$this->adminRedirect = array(
						'plugin' => false,
						'controller' => 'ev_shop_variants',
					);
				} else {
					$this->adminRedirect = array(
						'plugin' => 'ev_shop',
						'controller' => 'variants'
					);
				}

				$this->adminRedirect['action'] = 'manage';
				$this->adminRedirect[] = $id;
			}
		}

		parent::admin_edit($id);

		// Bring in Variant fields
		$this->set('variantFields', $this->_adminVariantManageFields());

		// Bring in currencies
		if (CakePlugin::loaded('EvCurrency')) {
			$this->Currencies->injectAdminForm(array(), $Model, $id);

			$Currencies = EvClassRegistry::init('EvCurrency.Currency');
			$defaultCurrency = $Currencies->find('first', array(
				'conditions' => array(
					'is_default' => '1'
				)
			));

			if (! empty($defaultCurrency)) {
				$this->set('defaultCurrency', $defaultCurrency['Currency']['id']);
			}
		}

		//On page load, set the currently selected variant options
		if (! $this->request->is('post') && ! $this->request->is('put')) {
			//Pull the options out of the request data and organise by option group
			$chosenOptions = Hash::combine($this->request->data,
				'Variant.{n}.Option.{n}.id',
				'Variant.{n}.Option.{n}.name',
				'Variant.{n}.Option.{n}.OptionGroup.name'
			);

			//Need to get the filters out
			$optionGroups = Hash::combine($this->request->data,
				'Variant.{n}.Option.{n}.OptionGroup.name',
				'Variant.{n}.Option.{n}.OptionGroup'
			);

			//Put the filters into the options array
			foreach ($optionGroups as $groupName => $groupData) {
				if (array_key_exists($groupName, $chosenOptions)) {
					$extraGroupData = $this->_getSelectedOptionGroupDataForImageEdit($groupData);
					$chosenOptions[$groupName] = $chosenOptions[$groupName] + $extraGroupData;
				}
			}

			$this->set('chosenOptions', $chosenOptions);
		}

		$this->view = 'EvShop./Products/admin_form';
	}

	protected function _getSelectedOptionGroupDataForImageEdit($data) {
		$chosenData = [];

		$chosenData['option_group_id'] = $data['id'];

		return $chosenData;
	}
}
