<?php
App::uses('Variant', 'Model');

/**
 * Variant Test Case
 *
 */
class VariantTest extends CakeTestCase {

/**
 * Fixtures
 *
 * @var array
 */
	public $fixtures = array(
		'plugin.EvShop.variant',
		'plugin.EvShop.product',
		'plugin.EvShop.brand',
		//'plugin.EvShop.tax_level',
		'plugin.EvShop.category',
		//'plugin.EvShop.categories_product',
		'plugin.EvShop.option',
		'plugin.EvShop.option_group',
		'plugin.EvShop.options_variant'
	);

/**
 * setUp method
 *
 * @return void
 */
	public function setUp() {
		parent::setUp();
		$this->Variant = ClassRegistry::init('EvShop.Variant');

		$this->options = array(
			1 => array(
				1 => 1,
				2 => 2
			),
			3 => array(
				4 => 4
			)
		);

		$this->defaultProductArray = array(
			'id' => 1,
			'name' => 'My Product',
			'description' => 'Lorem Ipsum',
			'is_active'
		);
	}

/**
 * tearDown method
 *
 * @return void
 */
	public function tearDown() {
		unset($this->Variant);

		parent::tearDown();
	}

	public function testProcessNewOptionsReturnsDefaultProductWhenNoOptionsPresentAndIsNewProduct() {
		$defaultProductArray = $this->defaultProductArray;
		unset($defaultProductArray['id']);

		$this->assertEquals(
			$this->Variant->processNewOptions(
				array(),
				array(),
				$defaultProductArray
			),
			array(
				array(
					'name' => 'My Product',
					'is_active' => 1
				)
			)
		);
	}

	public function testProcessNewOptionsReturnsEmptyArrayWhenNoOptionsPassedAndIsExistingProduct() {
		$defaultProductArray = $this->defaultProductArray;

		$this->assertEquals(
			$this->Variant->processNewOptions(
				array(),
				array(),
				$defaultProductArray
			),
			array()
		);
	}

	public function testProcessNewOptionsReturnsNoChangesWhenOptionsMatchOrWhenOptionsRemoved() {
		$this->assertEquals(
			$this->Variant->processNewOptions(
				$this->options,
				$this->options,
				$this->defaultProductArray
			),
			array()
		);

		$removedOptions = $this->options;
		unset($removedOptions[1][2]);

		$this->assertEquals(
			$this->Variant->processNewOptions(
				$this->options,
				$removedOptions,
				$this->defaultProductArray
			),
			array()
		);
	}

	public function testProcessNewOptionsReturnsVariantsWhenNewOptionsPassed() {
		$newOptions = $this->options;
		$newOptions[3][5] = 5;

		$this->assertEquals(
			$this->Variant->processNewOptions(
				$this->options,
				$newOptions,
				$this->defaultProductArray
			),
			array(
				array(
					'name' => 'Red Gloss',
					'is_active' => 1,
					'Option' => array(
						'Option' => array(
							5, 1
						)
					)
				),
				array(
					'name' => 'Blue Gloss',
					'is_active' => 1,
					'Option' => array(
						'Option' => array(
							5, 2
						)
					)
				)
			)
		);
	}

	public function testProcessNewOptionsRebuildsVariantsWhenNewOptionGroupPassed() {
		$newOptions = $this->options;
		$newOptions[2][3] = 3;
		ksort($newOptions);

		$this->assertEquals(
			$this->Variant->processNewOptions(
				$this->options,
				$newOptions,
				$this->defaultProductArray
			),
			array(
				array(
					'name' => 'Red Small Matte',
					'is_active' => 1,
					'Option' => array(
						'Option' => array(
							1, 3, 4
						)
					)
				),
				array(
					'name' => 'Blue Small Matte',
					'is_active' => 1,
					'Option' => array(
						'Option' => array(
							2, 3, 4
						)
					)
				)
			)
		);
	}

	public function testGetNewOptionsReturnsBlankArrayWithNoOptionChangesOrSomeOptionsRemoved() {
		$this->assertEquals(
			$this->Variant->getNewOptions(
				array(
					'CurrentOptions' => $this->options,
					'Options' => $this->options
				)
			),
			array()
		);

		$removedOptions = $this->options;
		unset($removedOptions[1][2]);

		$this->assertEquals(
			$this->Variant->getNewOptions(
				array(
					'CurrentOptions' => $this->options,
					'Options' => $removedOptions
				)
			),
			array()
		);
	}

	public function testGetNewOptionsReturnsCorrectArrayForNewOptions() {
		$newOptions = $this->options;
		$newOptions[3][5] = 5;

		$this->assertEquals(
			$this->Variant->getNewOptions(
				array(
					'CurrentOptions' => $this->options,
					'Options' => $newOptions
				)
			),
			array(
				3 => array(
					3 => array(
						5 => 5
					),
					1 => array(
						1 => 1,
						2 => 2
					)
				)

			)
		);
	}

	public function testGetNewOptionsReturnsCorrectArrayForNewOptionGroup() {
		$newOptions = $this->options;
		$newOptions[2][3] = 3;
		ksort($newOptions);

		$this->assertEquals(
			$this->Variant->getNewOptions(
				array(
					'CurrentOptions' => $this->options,
					'Options' => $newOptions
				)
			),
			array(
				$newOptions
			)
		);
	}

	public function testBuildVariantsReturnsCorrectArrayWithOptions() {
		$this->assertEquals(
			$this->Variant->buildVariants(
				array(
					2 => array(
						1, 2
					),
					3 => array(
						4, 5
					)
				),
				'My Product'
			),
			array(
				array(
					'name' => 'Red Matte',
					'is_active' => 1,
					'Option' => array(
						'Option' => array(
							1, 4
						)
					)
				),
				array(
					'name' => 'Red Gloss',
					'is_active' => 1,
					'Option' => array(
						'Option' => array(
							1, 5
						)
					)
				),
				array(
					'name' => 'Blue Matte',
					'is_active' => 1,
					'Option' => array(
						'Option' => array(
							2, 4
						)
					)
				),
				array(
					'name' => 'Blue Gloss',
					'is_active' => 1,
					'Option' => array(
						'Option' => array(
							2, 5
						)
					)
				)
			)
		);
	}

	public function testgenerateInsertArrayReturnsValidArrayForDB() {
		$this->assertEquals(
			$this->Variant->generateInsertArray(
				array(
					array(1, 4),
					array(1, 5),
					array(2, 4),
					array(2, 5)
				),
				array(
					'1' => 'Red',
					'2' => 'Blue',
					'4' => 'Matte',
					'5' => 'Gloss'
				)
			),
			array(
				array(
					'name' => 'Red Matte',
					'is_active' => 1,
					'Option' => array(
						'Option' => array(
							1, 4
						)
					)
				),
				array(
					'name' => 'Red Gloss',
					'is_active' => 1,
					'Option' => array(
						'Option' => array(
							1, 5
						)
					)
				),
				array(
					'name' => 'Blue Matte',
					'is_active' => 1,
					'Option' => array(
						'Option' => array(
							2, 4
						)
					)
				),
				array(
					'name' => 'Blue Gloss',
					'is_active' => 1,
					'Option' => array(
						'Option' => array(
							2, 5
						)
					)
				)
			)
		);
	}

	public function testdefaultInsertArrayGeneratesCorrectArrayForDB() {
		$this->assertEquals(
			$this->Variant->defaultInsertArray('My Product'),
			array(
				array(
					'name' => 'My Product',
					'is_active' => 1
				)
			)
		);
	}

	public function testGetDeletedOptionsReturnsBlankArrayWhenNoOptionsRemovedOrAdded() {
		$this->assertEquals(
			$this->Variant->getDeletedOptions(
				array(
					'CurrentOptions' => $this->options,
					'Options' => $this->options
				)
			),
			array()
		);

		$newOptions = $this->options;
		$newOptions[3][5] = 5;

		$this->assertEquals(
			$this->Variant->getDeletedOptions(
				array(
					'CurrentOptions' => $this->options,
					'Options' => $newOptions
				)
			),
			array()
		);
	}

	public function testGetDeletedOptionsReturnsArrayOfOptionsRemoved() {
		$removedOptions = $this->options;
		unset($removedOptions[1][2]);

		$this->assertEquals(
			$this->Variant->getDeletedOptions(
				array(
					'CurrentOptions' => $this->options,
					'Options' => $removedOptions
				)
			),
			array(2)
		);
	}

	public function testGetDeletedOptionsReturnsFlattenedArrayOfCurrentOptionsWhenNewGroupAdded() {
		$newOptions = $this->options;
		$newOptions[2][3] = 3;
		ksort($newOptions);

		$this->assertEquals(
			$this->Variant->getDeletedOptions(
				array(
					'CurrentOptions' => $this->options,
					'Options' => $newOptions
				)
			),
			array(
				'1.1' => 1,
				'1.2' => 2,
				'3.4' => 4
			)
		);
	}

	public function testProcessDeletedOptionsReturnsNullWhenNewOptionsAddedButItsNewProduct() {
		$this->assertNull(
			$this->Variant->processDeletedOptions(
				array(),
				$this->options,
				null
			)
		);
	}


	public function testProcessDeletedOptionsReturnsNullWhenNoOptionsRemovedOrAdded() {
		$this->assertNull(
			$this->Variant->processDeletedOptions(
				$this->options,
				$this->options,
				1
			)
		);

		$newOptions = $this->options;
		$newOptions[3][5] = 5;

		$this->assertNull(
			$this->Variant->processDeletedOptions(
				$this->options,
				$newOptions,
				'My Product'
			)
		);
	}

	public function testGetDeletedOptionsReturnsArrayOfVariantIdsWhenOptionRemoved() {
		$removedOptions = $this->options;
		unset($removedOptions[1][2]);

		$this->assertEquals(
			$this->Variant->processDeletedOptions(
				$this->options,
				$removedOptions,
				1
			),
			array(
				'2' => 'Blue Matte'
			)
		);
	}

	public function testGetDeletedOptionsReturnsArrayOfVariantIdsWhenOptionGroupIsChanged() {
		$newOptions = $this->options;
		$newOptions[2][3] = 3;
		ksort($newOptions);

		$this->assertEquals(
			$this->Variant->processDeletedOptions(
				$this->options,
				$newOptions,
				1
			),
			array(
				1 => 'Red Matte',
				2 => 'Blue Matte'
			)
		);

		$newOptions = $this->options;
		unset($newOptions[1]);

		$this->assertEquals(
			$this->Variant->processDeletedOptions(
				$this->options,
				$newOptions,
				1
			),
			array(
				1 => 'Red Matte',
				2 => 'Blue Matte'
			)
		);
	}
}
