<?php
App::uses('EvShopAppModel', 'EvShop.Model');
App::uses('ArrayUtil', 'EvCore.Lib');
App::uses('CakeNumber', 'Utility');

/**
 * Variant Model
 *
 * @property Product $Product
 * @property Option $Option
 */
class Variant extends EvShopAppModel {

	public $displayName = 'Pricing';

	/**
	 * Display field
	 *
	 * @var string
	 */
	public $displayField = 'name';

	/**
	 * Validation rules
	 *
	 * @var array
	 */
	public $validate = array(
		'product_id' => array(
			'numeric' => array(
				'rule' => array('numeric')
			),
		),
		'name' => array(
			'notEmpty' => array(
				'rule' => array('notBlank'),
				'message' => 'A variant name must be entered.'
			),
		),
		'price' => array(
			'notEmpty' => array(
				'rule' => array('notBlank')
			),
			'money' => array(
				'rule' => array('money')
			),
		),
	);

	//The Associations below have been created with all possible keys, those that are not needed can be removed

	/**
	 * belongsTo associations
	 *
	 * @var array
	 */
	public $belongsTo = array(
		'Product' => array(
			'className' => 'EvShop.Product'
		)
	);

	/**
	 * hasAndBelongsToMany associations
	 *
	 * @var array
	 */
	public $hasAndBelongsToMany = array(
		'Option' => array(
			'className' => 'EvShop.Option',
			'joinTable' => 'ev_shop_options_variants'
		)
	);

	public $findMethods = array(
		'allAndRound' => true
	);

	/**
	 * redefine constructor to check for EvTax
	 */
	public function __construct($id = false, $table = null, $ds = null) {
		if (CakePlugin::loaded('EvInventory')) {
			$this->actsAs[] = 'EvInventory.Inventories';
		}

		if (CakePlugin::loaded('EvTax')) {
			$this->actsAs['EvTax.CalculateTax'] = array(
				'fields' => array(
					'rrp',
					'price',
					'sale_price'
				),
				'TaxRateModel' => 'EvShop.Product',
				'TaxRateModelId' => 'Variant.product_id'
			);
		}

		parent::__construct($id, $table, $ds);
	}

	/**
	 * get all the data and then round the values
	 *
	 * @param 	string 	state - either before or after
	 * @param 	array 	array of the query params
	 * @param 	array 	Array of results (after only)
	 * @return 	array 	$query if state is before / $results if state is after
	 */
	protected function _findAllAndRound($state, $query, $results = array()) {
		if ($state === 'after') {
			foreach ($results as $key => $result) {
				$results[$key]['Variant']['rrp'] = CakeNumber::format($result['Variant']['rrp'], 2);
				$results[$key]['Variant']['price'] = CakeNumber::format($result['Variant']['price'], 2);
				$results[$key]['Variant']['sale_price'] = CakeNumber::format($result['Variant']['sale_price'], 2);
			}

			return $results;
		}

		return $query;
	}

	/**
	 * readForProductView
	 * get all the variants by product ID for the view
	 *
	 * @param 	int 	Product ID
	 * @return 	array 	Array of Variants / Options
	 */
	public function readForProductView($productId) {
		$data = $this->find(
			'all',
			array(
				'conditions' => array(
					'Variant.product_id' => $productId,
					'Variant.is_active' => 1
				),
				'order' => 'Variant.sequence ASC',
				'contain' => array(
					'Option' => array(
						'OptionGroup'
					)
				)
			)
		);

		// Rejiggle the array so it can be the same format
		// as if the original products models contained the data
		// also jiggle the options to use option group id as key so we can easily get when displaying
		return array_map(
			function ($variant) {

				$element = $variant['Variant'];
				unset($variant['Variant']);
				$element = ($element + $variant);

				if (! empty($element['Option'])) {
					$element['Option'] = Hash::combine(
						$element['Option'],
						'{n}.option_group_id',
						'{n}'
					);
				}

				return $element;
			},
			$data
		);
	}

	/**
	 * get listing for variants management
	 *
	 * @param 	int 	Product Id
	 * @param 	array 	custom array conditions if needed
	 * @return 	array 	results
	 */
	public function manageListing($productId, $query = array()) {
		$params = array(
			'conditions' => array(
				'Variant.product_id' => $productId
			),
			'order' => 'Variant.sequence ASC, Variant.id ASC',
			'contain' => array(
				'Option',
				'Product'
			)
		) + $query;

		$data = $this->find(
			'allAndRound',
			$params
		);

		// Due to relationships cake creates more queries to get the option groups
		// in the above find - manually sort option groups in only one extra query
		// process any options into a Group: Option, Group: Option string for displaying
		if (! empty($data['0']['Option'])) {
			$optionGroups = $this->Option->OptionGroup->find(
				'all',
				array(
					'callbacks' => false
				)
			);
			$optionGroups = Hash::combine($optionGroups, '{n}.OptionGroup.id', '{n}');

			// add the option group to each option
			$data = array_map(
				function ($variant) use ($optionGroups) {
					foreach ($variant['Option'] as $key => $option) {
						$variant['Option'][$key] += $optionGroups[$option['option_group_id']];
					}

					// create a readable description of all options on a variant
					$variant['optionDesc'] = Hash::combine(
						$variant,
						'Option.{n}.OptionGroup.name',
						'Option.{n}.name'
					);
					array_walk(
						$variant['optionDesc'],
						function (&$value, $key) {
							$value = $key . ': ' . $value;
						}
					);
					$variant['optionDesc'] = implode(', ', $variant['optionDesc']);

					return $variant;
				},
				$data
			);
		}

		return $data;
	}

	/**
	 * afterSave - send out variant saved event
	 *
	 */
	public function afterSave($created, $options = array()) {
		parent::afterSave($created, $options);

		// dispatch product saved event
		$this->getEventManager()->dispatch(
			new CakeEvent('EvShop.Model.Variant.saved', $this, array(
				'newItem' => $created
			))
		);
	}

	/**
	 * get any changes in the options and return the new variants
	 *
	 * @param 	array 	Current Options
	 * @param 	array 	New Options
	 * @param 	array 	default product array data
	 * @return 	array 	array of variants ready for DB insert
	 */
	public function processNewOptions($currentOptions, $newOptions, $product) {
		if (empty($newOptions)) {
			// options are empty but do we have an id number already?
			if (! empty($product['id']) && empty($currentOptions)) {
				// yes? it will already have a "variant"
				return array();
			}

			return $this->defaultInsertArray($product['name']);
		}

		$options = $this->getNewOptions(
			array(
				'CurrentOptions' => $currentOptions,
				'Options' => $newOptions
			)
		);

		$variants = array();

		if (! empty($options)) {
			foreach ($options as $optionIds) {
				$variants += $this->buildVariants(
					$optionIds
				);
			}
		}

		return $variants;
	}

	/**
	 * calculate any new options
	 *
	 * @param 	array 	Array of Options / CurrentOptions elements
	 * @return 	array 	Options to build variants for
	 */
	public function getNewOptions($data) {
		// check if the main option groups match
		$currentOptions = array_keys($data['CurrentOptions']);
		sort($currentOptions);
		$options = array_keys($data['Options']);
		sort($options);

		// rebuild all the selected options if they don't match
		if ($currentOptions !== $options) {
			return array($data['Options']);
		}

		$returnOptions = array();

		// the option groups are the same, check which options have been added
		// i.e. adding more colours
		foreach ($data['Options'] as $groupId => $optionIds) {
			foreach ($optionIds as $id => $value) {

				if (! isset($data['CurrentOptions'][$groupId][$id])) {
					$returnOptions[$groupId][$groupId][$id] = $id;
				}
			}
		}

		// add to the returning optins array
		// all the other groups that haven't changed so we can build all the new variant combinations
		if (! empty($returnOptions)) {
			foreach ($returnOptions as $section => $group) {
				$options = $data['Options'];

				unset($options[key($group)]);
				$returnOptions[$section] += $options;
			}
		}

		return $returnOptions;
	}

	/**
	 * take the multiple option arrays and build variants
	 *
	 * @param 	array 	array of option ids
	 * @return 	array 	array of variants ready for DB insertion
	 */
	public function buildVariants($options) {
		if (empty($options) || ! is_array($options)) {
			return array();
		}

		$Option = EvClassRegistry::init('EvShop.Option');
		$optionNames = $Option->getOptionNames(
			Hash::flatten($options)
		);

		$cartesian = ArrayUtil::cartesianProduct($options);

		return $this->generateInsertArray(
			$cartesian,
			$optionNames
		);
	}

	/**
	 * from the cartesian array generate db insert array
	 *
	 * @param 	array 	Cartesian product arrays
	 * @return 	array 	Array with variant name and option links
	 */
	public function generateInsertArray($cartesianProduct, $optionNames) {
		$variants = array();

		foreach ($cartesianProduct as $optionIds) {
			$name = array_intersect_key(
				$optionNames,
				array_flip($optionIds)
			);

			$variants[] = array(
				'name' => implode(' ', $name),
				'is_active' => 1,
				'Option' => array(
					'Option' => $optionIds
				)
			);
		}

		return $variants;
	}

	/**
	 * generate a default Insert Array for a product with no options
	 * variant = product name
	 *
	 * @param 	string 	Product name
	 * @return 	array 	Array with variant data for insert
	 */
	public function defaultInsertArray($productName) {
		return array(
			array(
				'name' => $productName,
				'is_active' => 1,
			)
		);
	}

	/**
	 * calculate the changes to options that require variants to be deleted
	 *
	 * @param 	array 	Array of Options / CurrentOptions elements
	 * @return 	array 	Options to remove
	 */
	public function getDeletedOptions($data) {
		// check if the main option groups match
		$currentOptions = array_keys($data['CurrentOptions']);
		sort($currentOptions);
		$options = array_keys($data['Options']);
		sort($options);

		// flag for deletion if not
		if ($currentOptions !== $options) {
			return Hash::flatten($data['CurrentOptions']);
		}

		// the option groups are the same, check which options have been removed
		$deletedOptions = array();

		foreach ($data['CurrentOptions'] as $groupId => $optionIds) {
			foreach ($optionIds as $id => $value) {

				if (! isset($data['Options'][$groupId][$id])) {
					$deletedOptions[] = $id;
				}
			}
		}

		return $deletedOptions;
	}

	/**
	 * process the deleted options and get the variants to delete
	 *
	 * @param 	array 	Current Options
	 * @param 	array 	New Options
	 * @param 	int 	product Id
	 * @return 	array 	Array of all the variants to delete
	 */
	public function processDeletedOptions($currentOptions, $newOptions, $productId) {
		$options = $this->getDeletedOptions(
			array(
				'CurrentOptions' => $currentOptions,
				'Options' => $newOptions
			)
		);

		/*
			if options are empty a few situations could be happening.
			- if productId is empty then likely it's a new product options is empty as there is nothing to delete = return null
			- productId is not empty to either
			- - both currentOptions and newOptions were equal, meaning no options changed = return null
			- - or currentOptions is empty and newOptions has been checked (In this instance we need to delete the single variant created for the product) = proceed to getting variant id for delete
		*/
		if (empty($options)) {
			if (empty($productId)) {
				return null;
			}

			if (
				(empty($currentOptions) && empty($newOptions)) ||
				(! empty($currentOptions) && ! empty($newOptions))
			) {
				return null;
			}
		}

		// if we're here proceed and get the variants to delete
		$params = array(
			'conditions' => array(
				'Variant.product_id' => $productId

			),
			'callbacks' => false
		);

		// only join options if options are involved
		// could be a single variant product (i.e. one with no options)
		if (! empty($options)) {
			$params['joins'] = array(
				array(
					'table' => 'ev_shop_options_variants',
					'alias' => 'OptionsVariant',
					'conditions' => array(
						'Variant.id = OptionsVariant.variant_id'
					)
				)
			);
			$params['conditions']['OptionsVariant.option_id'] = $options;
		}

		return $this->find(
			'list',
			$params
		);
	}

	/**
	 * get the unit price of the variant
	 * used by EvBasket
	 *
	 * @param 	int 	$itemId 	itemId
	 * @return 	float|bool
	 */
	public function getUnitPrice($itemId) {
		$Variant = $this->find(
			'first',
			array(
				'conditions' => array(
					'Variant.id' => $itemId
				)
			)
		);

		if (empty($Variant)) {
			return false;
		}

		$salePrice = (float)$Variant['Variant']['sale_price'];
		if (! empty($salePrice)) {
			return $Variant['Variant']['sale_price'];
		}

		return $Variant['Variant']['price'];
	}

	/**
	 * get the tax rate for the variant
	 * used by EvBasket
	 *
	 * @param 	int 	$itemId 	item ID
	 * @return 	float
	 */
	public function getTaxRate($itemId) {
		if (! CakePlugin::loaded('EvTax')) {
			return 0;
		}

		$Variant = $this->find(
			'first',
			array(
				'conditions' => array(
					'Variant.id' => $itemId
				),
				'contain' => array(
					'Product'
				)
			)
		);

		if (empty($Variant) || empty($Variant['Product']['tax_level_id'])) {
			return 0;
		}

		$TaxLevel = EvClassRegistry::init('EvTax.TaxLevel');
		$taxRate = $TaxLevel->fieldOnly(
			'rate',
			array(
				'TaxLevel.id' => $Variant['Product']['tax_level_id']
			)
		);

		if (is_null($taxRate)) {
			return 0;
		}

		return $taxRate;
	}

}
