<?php
App::uses('EvShopAppModel', 'EvShop.Model');
App::uses('CakeNumber', 'Utility');
/**
 * Product Model
 *
 * @property Brand $Brand
 * @property TaxLevel $TaxLevel
 * @property Variant $Variant
 * @property Category $Category
 */
class Product extends EvShopAppModel {

	public $actsAs = array(
		'Routable.Routable' => array(
			'config' => 'EvShop',
			'alias' => 'product/:displayField'
		),
		'MetaData.Meta',
		'EvTemplates.Template' => array(
			'formInject' => true,
			'restrictTo' => 'Product'
		)
	);

	/**
	 * Display field
	 *
	 * @var string
	 */
	public $displayField = 'name';

	/**
	 * Validation rules
	 *
	 * @var array
	 */
	public $validate = array(
		'name' => array(
			'notEmpty' => array(
				'rule' => array('notBlank'),
				'message' => 'A product name must be entered'
			),
		),
	);

	//The Associations below have been created with all possible keys, those that are not needed can be removed

	/**
	 * belongsTo associations
	 *
	 * @var array
	 */
	public $belongsTo = array(
		'Brand' => array(
			'className' => 'EvShop.Brand'
		)
	);

	/**
	 * hasMany associations
	 *
	 * @var array
	 */
	public $hasMany = array(
		'Variant' => array(
			'className' => 'EvShop.Variant',
			'cascade' => true,
			'dependent' => true
		),
		'CategoriesProduct' => array(
			'className' => 'EvShop.CategoriesProduct',
			'order' => 'CategoriesProduct.sequence ASC'
		)
	);

	/**
	 * image slots
	 */
	public $imageSlots = array(
		'listing' => array(
			'slots' => 1,
			'fields' => false
		)
	);

	/**
	 * redefine constructor to check for EvTax
	 */
	public function __construct($id = false, $table = null, $ds = null) {
		if (CakePlugin::loaded('EvTax')) {
			$this->actsAs['EvTax.Taxable'] = array(
				'formInject' => true
			);
		}

		parent::__construct($id, $table, $ds);
	}

	/**
	 * before we save anything process the options into variants
	 * also process the category array
	 *
	 */
	public function beforeBeforeSave($data, $options) {
		$Variant = EvClassRegistry::init('EvShop.Variant');

		if (empty($data['CurrentOptions'])) {
			$data['CurrentOptions'] = array();
		}
		if (empty($data['Options'])) {
			$data['Options'] = array();
		}

		$data['Options'] = array_map(
			function ($options) {
				return array_combine(
					$options,
					$options
				);
			},
			$data['Options']
		);

		$this->variantsToDelete = $Variant->processDeletedOptions($data['CurrentOptions'], $data['Options'], $data[$this->alias][$this->primaryKey]);
		$data['Variant'] = $Variant->processNewOptions($data['CurrentOptions'], $data['Options'], $data[$this->alias]);
		unset($data['Options']);

		// process the categories
		$Category = EvClassRegistry::init('EvShop.Category');
		$this->categoriesToDelete = $Category->processDeletedCategories($data['CategoriesProduct']);

		return $data;
	}

	/**
	 * afterSave - delete any variants to be deleted
	 *
	 */
	public function afterSave($created, $options = array()) {
		parent::afterSave($created, $options);

		if (! empty($this->variantsToDelete)) {
			$Variant = EvClassRegistry::init('EvShop.Variant');

			$Variant->deleteAll(
				array(
					'Variant.id' => array_keys($this->variantsToDelete)
				)
			);
		}

		if (! empty($this->categoriesToDelete)) {
			$CategoriesProduct = EvClassRegistry::init('EvShop.CategoriesProduct');

			$CategoriesProduct->deleteAll(
				array(
					'CategoriesProduct.id' => $this->categoriesToDelete
				)
			);
		}

		// dispatch product saved event
		$this->getEventManager()->dispatch(
			new CakeEvent('EvShop.Model.Product.saved', $this, array(
				'newItem' => $created
			))
		);
	}

	/**
	 * readForEdit - bring out the variants and calculate the options
	 *
	 */
	public function readForEdit($id, $query = array()) {
		$query['contain']['Variant'] = array('Option');
		$query['contain']['CategoriesProduct'] = array('Category');

		$data = parent::readForEdit($id, $query);

		$data['Options'] = Hash::combine($data['Variant'], '{n}.Option.{n}.id', '{n}.Option.{n}.id', '{n}.Option.{n}.option_group_id');
		$data['CategoriesProduct'] = Hash::combine($data['CategoriesProduct'], '{n}.category_id', '{n}');

		return $data;
	}

	/**
	 * readForView - Do the Variants separately so the callbacks will be run
	 *
	 * @param integer $id ID of row to edit
	 * @param array $params The db query array - can be used to pass in additional parameters such as contain
	 * @return array
	 */
	public function readForView($id, $query = array()) {
		$query['contain']['CategoriesProduct'] = array('Category');

		if (Configure::read('EvShop.showBrands')) {
			$query['contain'][] = 'Brand';
		}

		$data = parent::readForView($id, $query);

		if (empty($data)) {
			return false;
		}

		// perform this on it's own rather then contain so the inventory behaviours / cakllbacks can be triggered
		$data['Variant'] = $this->Variant->readForProductView($id);

		if (! empty($data['Variant'])) {
			$data['Options'] = Hash::combine($data['Variant'], '{n}.Option.{n}.id', '{n}.Option.{n}.id', '{n}.Option.{n}.option_group_id');
		}

		if (! empty($data['CategoriesProduct'])) {
			$data['CategoriesProduct'] = Hash::combine($data['CategoriesProduct'], '{n}.category_id', '{n}');
		}

		$data = $this->defaultMetaDescription($data);

		return $data;
	}

	/**
	 * unless the meta description has been overridden
	 * set seo product meta description
	 *
	 * @param 	array 	$data 	Array of product data
	 * @return 	array 	$data 	Updated product data array with updated meta description
	 */
	public function defaultMetaDescription($data) {
		if (! isset($data['MetaData']['description']) || empty($data['MetaData']['description'])) {
			$variantCount = count($data['Variant']);
			$siteTitle = Configure::read('SiteSetting.general.site_title');

			if ($variantCount > 1) {
				$prices = Hash::extract($data, 'Variant.{n}.price');
				foreach ($prices as $key => $price) {
					if (empty($price) || $price == '0.000000') {
						unset($prices[$key]);
					}
				}
				sort($prices);

				if (! empty($prices['0']) && (float)$prices['0'] > 0) {
					$price = CakeNumber::currency($prices['0']);

					$description = "Order " . $data['Product']['name'] . " from only " . $price . " from " . $siteTitle . " now!";
				}
			} elseif ($variantCount === 1) {
				$price = CakeNumber::currency($data['Variant']['0']['price']);

				$description = "Order " . $data['Product']['name'] . " for only " . $price . " from " . $siteTitle . " now!";
			}

			if (! empty($description)) {
				$data['MetaData']['description'] = $description;
			}
		}

		return $data;
	}
}
