<?php
App::uses('EvShopAppModel', 'EvShop.Model');
/**
 * Category Model
 *
 * @property Product $Product
 */
class Category extends EvShopAppModel {

	public $actsAs = array(
		'Tree',
		'Routable.Routable' => array(
			'config' => 'EvShop',
			'alias' => 'category/:displayField/*'
		),
		'MetaData.Meta',
		'EvTemplates.Template' => array(
			'formInject' => true,
			'restrictTo' => 'Category'
		)
	);

	/**
	 * Display field
	 *
	 * @var string
	 */
	public $displayField = 'name';

	public $order = array('lft' => 'ASC');

	/**
	 * Validation rules
	 *
	 * @var array
	 */
	public $validate = array(
		'name' => array(
			'notEmpty' => array(
				'rule' => array('notBlank'),
				'message' => 'A category name must be entered.'
			),
		),
	);

	//The Associations below have been created with all possible keys, those that are not needed can be remove

	/**
	 * hasMany relationships
	 */
	public $hasMany = array(
		'ChildCategory' => array(
			'className' => 'EvShop.Category',
			'foreignKey' => 'parent_id'
		),
		'CategoriesProduct' => array(
			'className' => 'EvShop.CategoriesProduct',
			'order' => 'CategoriesProduct.sequence ASC'
		)
	);

	/**
	 * belongsTo relationships
	 */
	public $belongsTo = array(
		'ParentCategory' => array(
			'className' => 'EvShop.Category',
			'foreignKey' => 'parent_id'
		)
	);

	/**
	 * image slots
	 */
	public $imageSlots = array(
		'listing' => array(
			'slots' => 1,
			'fields' => false
		)
	);

	/**
	 * beforeSave / afterFind callbacks to transform parent_id
	 * into top level category option iuf selected
	 */
	/**
	 * Called after each find operation. Can be used to modify any results returned by find().
	 * Return value should be the (modified) results.
	 *
	 * @param mixed $results The results of the find operation
	 * @param bool $primary Whether this model is being queried directly (vs. being queried as an association)
	 * @return mixed Result of the find operation
	 * @link http://book.cakephp.org/2.0/en/models/callback-methods.html#afterfind
	 */
	public function afterFind($results, $primary = false) {
		foreach ($results as $key => $item) {
			if (empty($item[$this->alias]['parent_id'])) {
				$results[$key][$this->alias]['parent_id'] = 0;
			}
		}

		return $results;
	}

	/**
	 * Called before each save operation, after validation. Return a non-true result
	 * to halt the save.
	 *
	 * @param array $options Options passed from Model::save().
	 * @return bool True if the operation should continue, false if it should abort
	 * @link http://book.cakephp.org/2.0/en/models/callback-methods.html#beforesave
	 * @see Model::save()
	 */
	public function beforeSave($options = array()) {
		$return = parent::beforeSave($options);

		if (empty($this->data[$this->alias]['parent_id'])) {
			$this->data[$this->alias]['parent_id'] = null;
		}

		return $return;
	}

	/**
	 * check the selected categories to see which ones we need to unlink
	 *
	 * @param 	array 	The array of selected categories passed by reference
	 * @return 	array 	Array fo linker table rows to deleted
	 */
	public function processDeletedCategories(&$data) {
		$toDelete = array();

		foreach ($data as $categoryId => $category) {
			if (isset($category['id']) && ! isset($category['category_id'])) {

				$toDelete[] = $category['id']; // id of the linker table row
				unset($data[$categoryId]);
			}
		}

		return $toDelete;
	}

	/**
	 * get the category list for use in calculating breadcrumb
	 *
	 * @return array
	 */
	public function getBreadcrumbList() {
		$categories = $this->find(
			'all',
			array(
				'conditions' => array(
					'Category.is_active' => 1
				),
				'callbacks' => false
			)
		);

		return Hash::combine(
			$categories,
			'{n}.Category.id',
			'{n}'
		);
	}
}
