<?php

App::uses('AppComponent', 'Controller/Component');

class ProductsComponent extends AppComponent {

	/**
	 * get a product listing
	 *
	 * @param 	array 	query / paginate params
	 * @param 	bool 	Paginate results or not?
	 * @return 	array 	Array of results
	 */
	public function listing($params = array(), $paginate = true) {
		$params = Hash::merge(
			array(
				'joins' => array(
					array(
						'table' => 'ev_shop_categories_products',
						'alias' => 'CategoriesProduct',
						'conditions' => array(
							'Product.id = CategoriesProduct.product_id'
						)
					)
				),
				'conditions' => array(
					'Product.is_active' => 1
				),
				'order' => 'CategoriesProduct.sequence ASC, Product.name ASC',
				'contain' => array(
					'ListingImage',
					'Variant' => array(
						'order' => 'Variant.sequence ASC'
					)
				),
				'group' => 'Product.id'
			),
			$params
		);

		$Product = EvClassRegistry::init('EvShop.Product');

		if ($paginate && ! empty($this->_controller)) {
			$this->_controller->Paginator->settings = $params;
			return $this->_controller->Paginator->paginate(
				$Product
			);
		} else {
			return $Product->find('all', $params);
		}
	}

	/**
	 * get a featured product listing
	 *
	 * @param 	array 	query / paginate params
	 * @param 	bool 	Paginate results or not?
	 * @return 	array 	Array of results
	 */
	public function featured($params = array(), $paginate = false) {
		$params = Hash::merge(
			array(
				'conditions' => array(
					'Product.is_featured' => 1
				)
			),
			$params
		);

		return $this->listing($params, $paginate);
	}

	/**
	 * get a listing by category
	 *
	 * @param 	int|array 	Category ID or array of category IDs
	 * @param 	array 		query / paginate params
	 * @param 	bool 	Paginate results or not?
	 * @return 	array 		Array of results
	 */
	public function listingByCategory($categoryId, $params = array(), $paginate = true) {
		$defaults = array(
			'conditions' => array(
				'CategoriesProduct.category_id' => $categoryId
			)
		);

		// if it's many categories, and the group by and9
		// also contain the categories so we know which is which
		if (is_array($categoryId)) {
			$defaults['contain']['CategoriesProduct'] = array(
				'limit' => 1,
				'Category'
			);
		}

		$params = Hash::merge(
			$defaults,
			$params
		);

		return $this->listing($params, $paginate);
	}

	/**
	 * get a listing by a brand
	 *
	 * @param 	int|array 	brand ID or array of brand IDs
	 * @param 	array 		query / paginate params
	 * @param 	bool 	Paginate results or not?
	 * @return 	array 		Array of results
	 */
	public function listingByBrand($brandId, $params = array(), $paginate = true) {
		$defaults = array(
			'conditions' => array(
				'Product.brand_id' => $brandId
			)
		);

		// if it's many categories, and the group by and
		// also contain the categories so we know which is which
		if (is_array($brandId)) {
			$defaults['contain'][] = 'Brand';
		}

		$params = Hash::merge(
			$defaults,
			$params
		);

		return $this->listing($params, $paginate);
	}
}
