<?php

App::uses('EvShopAppModel', 'EvShop.Model');
App::uses('ArrayUtil', 'EvCore.Lib');

class ProductAttribute extends EvShopAppModel {

	// public $tablePrefix = false;
	public $useTable = "product_attributes";

	public $hasMany = array(
		"ProductAttributeOption" => array(
			'className' => 'EvShop.ProductAttributeOption',
			'cascade' => true,
			'dependent' => true,
			// 'foreignKey' => 'product_attribute_id'
		),
		"Category" => array(
			'className' => 'EvCategory.Category'
		)
	);

	public $validate = array(
		'name' => array(
			'required' => array(
				'rule' => 'notEmpty',
				'message' => 'Attribute name is mandatory'
			),
			'unique' => array(
				'rule' => 'isUnique',
				'message' => 'Attribute name must be unique'
			)
		)
	);

	public function readForEdit($id, $query = array()) {

		if(! isset($query['contain']['ProductAttributeOption'])) {

			$query['contain']["ProductAttributeOption"] = array(
				'order' => 'sequence ASC'
			);

		}

		return parent::readForEdit($id, $query);

	}

/**
 * After Save we need to rebuild routes for categories which are associated with product attributes
 * @see Model::afterSave
 */
	public function afterSave($created, $options = array()) {

		$reliantCategories = $this->Category->find('all', array(
				'conditions' => array(
					'Category.product_attribute_id' => $this->id
				)
			)
		);

		if($reliantCategories) {

			foreach($reliantCategories as $category) {

				$this->Category->rebuildCategoryProductAttributeRoutes($category);

			}

		}

	}


/**
 * Gets the primary attribute and its options
 * The primary attribute is indicated by the "is_primary" flag
 * Only one product attribute should have this flag
 * If none have the flag set then it will default back to the first attribute
 * @param integer $productId The product id of the product, the reason we need this is becuase not all
 * 							 Products are going to have the primary attribute on them
 */
	public function getPrimaryAttribute($productId) {

		$primaryAttribute = $this->find('first', array(
			'joins' => array(
				array(
					'table' => 'ev_shop_product_attribute_options',
					'alias' => 'ProductAttributeOption',
					'type' => 'RIGHT',
					'conditions' => array(
						'ProductAttributeOption.product_attribute_id = ProductAttribute.id',
						// 'ProductProductAttribute.product_id' => $productId
					)
				),
				array(
					'table' => 'ev_shop_product_attribute_options_products',
					'alias' => 'ProductProductAttributeOption',
					'type' => 'RIGHT',
					'conditions' => array(
						'ProductProductAttributeOption.product_attribute_option_id = ProductAttributeOption.id',
						'ProductProductAttributeOption.product_id' => $productId
					)
				)
			),
			'conditions' => array(
				'ProductAttribute.affects_price' => true
			),
			'order' => 'is_primary DESC, id ASC'
		));

		if(!empty($primaryAttribute)) {


			//Get attribute options but only those applicable to the product, cannot be done in the contain due to the join
			$attributeOptions = $this->ProductAttributeOption->find('all' , array(
					'conditions' => array(
						'ProductAttributeOption.product_attribute_id' => $primaryAttribute['ProductAttribute']['id']
					),
					'joins' => array(
						array(
							'table' => 'ev_shop_product_attribute_options_products',
							'alias' => 'ProductProductAttributeOption',
							'type' => 'RIGHT',
							'conditions' => array(
								'ProductProductAttributeOption.product_attribute_option_id = ProductAttributeOption.id',
								'ProductProductAttributeOption.product_id' => $productId
							)
						)
					),
					'order' => 'sequence ASC'
				)
			);

			//Change back into the contain format and add back to return array
			$primaryAttribute['ProductAttributeOption'] = Hash::combine($attributeOptions, '{n}.ProductAttributeOption.id', '{n}.ProductAttributeOption');

		}

		return $primaryAttribute;

	}


/**
 * Sets the primary attribute
 * This is the main attribute for all products such as size
 * @param $id integer 	id of the product attribute
 */
	public function setPrimaryAttribute($id) {

		$this->id = $id;
		$this->saveField('is_primary', $id);

		$this->updateAll(
			array(
				'is_primary' => false
			),
			array(
				'ProductAttribute.id <>' => $id
			)
		);

	}

/**
 * Brings back a list of product attribute option combinations
 * Example:
 * 		Attribute 1
 *			|-> Option 1a
 *			|-> Option 1b
 * 		Attribute 2
 *			|-> Option 2a
 *			|-> Option 2b
 * 	Would produce:
 *		Option 1a, Option2a
 *		Option 1a, Option2b
 *		Option 1b, Option2a
 *		Option 1b, Option2b
 *
 * @param int $productId
 * @return array
 */
	public function getProductOptionCombinations($productId) {

		$primaryAttribute = $this->getPrimaryAttribute($productId);

		$queryParams = array(
			'conditions' => array(
				'ProductAttribute.id IS NOT NULL'
			),
			'joins' => array(
				//Only want attributes we have enabled for this product
				array(
					'table' => 'ev_shop_product_attribute_options',
					'alias' => 'ProductAttributeOption',
					'type' => 'RIGHT',
					'conditions' => array(
						'ProductAttributeOption.product_attribute_id = ProductAttribute.id',
					)
				),
				array(
					'table' => 'ev_shop_product_attribute_options_products',
					'alias' => 'ProductProductAttributeOption',
					'type' => 'RIGHT',
					'conditions' => array(
						'ProductProductAttributeOption.product_attribute_option_id = ProductAttributeOption.id',
						'ProductProductAttributeOption.product_id' => $productId
					)
				)
			),
			'group' => 'ProductAttribute.id'
		);

		if($primaryAttribute) {

			//This attribute goes across the top of the table so we don't want to include it
			$queryParams['conditions']['ProductAttribute.id <>'] = $primaryAttribute['ProductAttribute']['id'];

		}

		$productAttributes = $this->find('all', $queryParams);

		//Build array for cartensian product and also add in the product options for each one
		//@review is there a more efficient way? I couldn't do it in the contain because of the required join
		$allOptions = array();

		foreach($productAttributes as $productAttribute) {

			//Get attribute options but only those applicable to the product, cannot be done in the contain due to the join
			$attributeOptions = $this->ProductAttributeOption->find('all' , array(
					'conditions' => array(
						'ProductAttributeOption.product_attribute_id' => $productAttribute['ProductAttribute']['id']
					),
					'joins' => array(
						array(
							'table' => 'ev_shop_product_attribute_options_products',
							'alias' => 'ProductProductAttributeOption',
							'type' => 'RIGHT',
							'conditions' => array(
								'ProductProductAttributeOption.product_attribute_option_id = ProductAttributeOption.id',
								'ProductProductAttributeOption.product_id' => $productId
							)
						)
					),
					'order' => 'sequence ASC'
				)
			);

			//Change back into the contain format and add back to return array
			$productAttribute['ProductAttributeOption'] = Hash::combine($attributeOptions, '{n}.ProductAttributeOption.id', '{n}.ProductAttributeOption');


			$allOptions[] = $productAttribute['ProductAttributeOption'];

		}

		$combinations = ArrayUtil::arrayCartesianProduct($allOptions);

		return $combinations;
	}

/**
 * Gets the options for a given attribute
 * @param  (int) $productId               	Product ID
 * @param  (int) $attributeId             	Attribute ID
 * @param  array  $selectedAttributeOptions	An array of currently selected product attribute options, this refines the options for the given attribute
 * @return array  							find(all) array of product attribute options
 */
	public function getAttributeOptionsForProduct($productId, $attributeId, $selectedAttributeOptions = array(), $attributeOptionsContains = array()) {

		$subqueryParams = array(
			'fields' => array('ProductVariant.id'),
			'joins' => array(
				array(
					'table' => '`ev_shop_product_attribute_options_product_variants`',
					'alias' => 'ProductAttributeOptionProductVariant',
					'type' => 'RIGHT',
					'conditions' => array(
						'ProductVariant.id = ProductAttributeOptionProductVariant.product_variant_id'
					)
				)
			),
			'conditions' => array(
				'ProductVariant.product_id' => $productId,
				'ProductVariant.price <>' => ""

			),
			'group' => 'ProductVariant.id'
		);

		if(!empty($selectedAttributeOptions)) {
			$subqueryParams['group'] = 'ProductVariant.id HAVING COUNT(DISTINCT ProductAttributeOptionProductVariant.product_attribute_option_id) = "'.count($selectedAttributeOptions).'"';
			$subqueryParams['conditions']['ProductAttributeOptionProductVariant.product_attribute_option_id'] = $selectedAttributeOptions;
		}

		$ProductVariant = ClassRegistry::init('EvShop.ProductVariant');

		$this->ProductAttributeOption->virtualFields['product_variant_id'] = "ProductVariant.id";
		$this->ProductAttributeOption->virtualFields['min_price'] = "ProductVariant.price";
		$this->ProductAttributeOption->virtualFields['min_rrp'] = "ProductVariant.rrp";
		$this->ProductAttributeOption->virtualFields['stock'] = "ProductVariant.stock";

		$params = array(
			'joins' => array(
				array(
					'table' => 'ev_shop_product_attribute_options_product_variants',
					'alias' => 'ProductAttributeOptionProductVariant',
					'conditions' => array(
						'ProductAttributeOptionProductVariant.product_attribute_option_id = '.$this->ProductAttributeOption->escapeField('id'),
						'ProductAttributeOptionProductVariant.product_variant_id IN ('.$ProductVariant->subquery($subqueryParams).')'
					)
				),
				array(
					'table' => 'ev_shop_product_variants',
					'alias' => 'ProductVariant',
					'type' => 'LEFT',
					'conditions' => array(
						'ProductVariant.id = ProductAttributeOptionProductVariant.product_variant_id'
					)
				)
			),
			'conditions' => array(
				'product_attribute_id' => $attributeId
			),
			'group' => $this->ProductAttributeOption->escapeField('id'),
			'order' => 'MIN(ProductVariant.price)',
			'contain' => $attributeOptionsContains
		);

		$options = $this->ProductAttributeOption->find('all', $params);

		unset($this->ProductAttributeOption->virtualFields['product_variant_id']);
		unset($this->ProductAttributeOption->virtualFields['min_price']);
		unset($this->ProductAttributeOption->virtualFields['min_rrp']);
		unset($this->ProductAttributeOption->virtualFields['stock']);

		return $options;

	}

/**
 * Gets the product attributes which are available on a given product and their options
 * @param  int 		$productId Product Id
 * @return array    find(all) formatted array
 */
	public function getProductAttributes($productId, $query = array()) {

		$query['joins'][] = array(
			'table' => 'ev_shop_product_attribute_options',
			'alias' => 'ProductAttributeOption',
			'joinType' => 'RIGHT',
			'conditions' => array(
				'ProductAttributeOption.product_attribute_id = ProductAttribute.id'
			)
		);
		$query['joins'][] = array(
			'table' => 'ev_shop_product_attribute_options_product_variants',
			'alias' => 'ProductAttributeOptionProductVariant',
			'type' => 'RIGHT',
			'conditions' => array(
				'ProductAttributeOptionProductVariant.product_attribute_option_id = ProductAttributeOption.id'
			)
		);
		$query['joins'][] = array(
			'table' => 'ev_shop_product_variants',
			'alias' => 'ProductVariant',
			'type' => 'RIGHT',
			'conditions' => array(
				'ProductAttributeOptionProductVariant.product_variant_id = ProductVariant.id',
				'ProductVariant.product_id' => $productId,
				'ProductVariant.price <>' => ""
			)
		);

		$query['group'] = 'ProductAttribute.id HAVING COUNT(DISTINCT ProductAttributeOption.id) > 1';
		$query['order'] = 'ProductAttribute.is_primary DESC';


		//Get attributes and options and reformat into containable form
		//Is inefficient could do with been switched out for an array map
		$productAttributes = $this->find('all', $query);

		foreach($productAttributes as $key=>$productAttribute) {

			$attributeOptions = $this->getAttributeOptionsForProduct($productId, $productAttribute['ProductAttribute']['id']);

			foreach($attributeOptions as $attributeOptionKey => $attributeOption) {

				$formattedProductAttributeOption = $attributeOption['ProductAttributeOption'];
				unset($attributeOption['ProductAttributeOption']);

				$formattedProductAttributeOption = array_merge($formattedProductAttributeOption, $attributeOption);

				$productAttributes[$key]['ProductAttributeOption'][$formattedProductAttributeOption['id']] = $formattedProductAttributeOption;

			}

		}

		return $productAttributes;
	}

/**
 * Gets an array of product attributes and available product options
 * which are marked for filtering on the frontend
 * @return array find('all') formatted array
 */
	public function getProductFilters($filters = array()) {

		$queryParams = array(
			'conditions' => array(
				'ProductAttribute.id IS NOT NULL'
			),
			'joins' => array(
				//Only want attributes we have enabled for this product
				array(
					'table' => 'ev_shop_product_attribute_options',
					'alias' => 'ProductAttributeOption',
					'type' => 'RIGHT',
					'conditions' => array(
						'ProductAttributeOption.product_attribute_id = ProductAttribute.id',
					)
				),
				array(
					'table' => 'ev_shop_product_attribute_options_product_variants',
					'alias' => 'ProductVariantProductAttributeOption',
					'type' => 'RIGHT',
					'conditions' => array(
						'ProductAttributeOption.id = ProductVariantProductAttributeOption.product_attribute_option_id',
					)
				),
				array(
					'table' => 'ev_shop_product_variants',
					'alias' => 'ProductVariant',
					'type' => 'RIGHT',
					'conditions' => array(
						'ProductVariant.id = ProductVariantProductAttributeOption.product_variant_id',
						'ProductVariant.price <> ""'
					)
				)
			),
			'group' => 'ProductAttribute.id HAVING COUNT(DISTINCT ProductAttributeOption.id) > 1'
		);

		if(isset($filters['category'])) {

			$queryParams['joins'][] = array(
				'table' => 'ev_related_items_related_items',
				'alias' => 'RelatedItemCategory',
				'type' => 'RIGHT',
				'conditions' => array(
					'RelatedItemCategory.model' => 'Product',
					'RelatedItemCategory.related_model' => 'Category',
					'RelatedItemCategory.model_id = ProductVariant.product_id',
					'RelatedItemCategory.related_model_id' => $filters['category']
				)
			);


		}


		if(isset($filters['brand'])) {

			$queryParams['joins'][] = array(
				'table' => 'ev_related_items_related_items',
				'alias' => 'RelatedItemBrand',
				'type' => 'RIGHT',
				'conditions' => array(
					'RelatedItemBrand.model' => 'Product',
					'RelatedItemBrand.related_model' => 'Brand',
					'RelatedItemBrand.model_id = ProductVariant.product_id',
					'RelatedItemBrand.related_model_id' => $filters['brand']
				)
			);

		}

		$productAttributes = $this->find('all', $queryParams);

		foreach($productAttributes as $key=>$productAttribute) {

			//Get attribute options but only those applicable to the product, cannot be done in the contain due to the join

			$attributeOptions = $this->getAttributeOptions(
				$productAttribute['ProductAttribute']['id'],
				isset($filters['category']) ? $filters['category'] : false
			);

			//Change back into the contain format and add back to return array
			foreach($attributeOptions as $attributeOptionKey=>$attributeOption) {

				$additionalInformation = $attributeOption;
				unset($additionalInformation['ProductAttributeOption']);

				$productAttributes[$key]['ProductAttributeOption'][$attributeOption['ProductAttributeOption']['id']] = array_merge($attributeOption['ProductAttributeOption'], $additionalInformation);
			}

		}

		return $productAttributes;

	}

/**
 * Returns a find('all') of all product attribute options
 * @param int	$attributeId	Product Attribute ID
 * @param int	$categoryId	You can specify a category which will filter product attribute options to those applicable to that category
 */
	public function getAttributeOptions($attributeId, $categoryId = false, $query = array()) {

		//Get attribute options but only those applicable to the product, cannot be done in the contain due to the join

		$query['conditions']['ProductAttributeOption.product_attribute_id'] = $attributeId;

		$query['joins'][] = array(
			'table' => 'ev_shop_product_attribute_options_product_variants',
			'alias' => 'ProductVariantProductAttributeOption',
			'type' => 'RIGHT',
			'conditions' => array(
				'ProductAttributeOption.id = ProductVariantProductAttributeOption.product_attribute_option_id',
			)
		);

		$query['joins'][] = array(
			'table' => 'ev_shop_product_variants',
			'alias' => 'ProductVariant',
			'type' => 'RIGHT',
			'conditions' => array(
				'ProductVariant.id = ProductVariantProductAttributeOption.product_variant_id',
				'ProductVariant.price <> ""'
			)
		);

		$query['order'] = 'sequence ASC';
		$query['group'] = 'ProductAttributeOption.id';

		if($categoryId) {

			$query['joins'][] = array(
				'table' => 'ev_related_items_related_items',
				'alias' => 'RelatedItem',
				'type' => 'RIGHT',
				'conditions' => array(
					'RelatedItem.model_id = ProductVariant.product_id',
					'RelatedItem.model' => 'Product',
					'RelatedItem.related_model' => 'Category',
					'RelatedItem.related_model_id' => $categoryId
				)
			);

		}

		$attributeOptions = $this->ProductAttributeOption->find('all' , $query);

		return $attributeOptions;
	}
}
