<?php

App::uses('EvShopAppModel', 'EvShop.Model');

class Product extends EvShopAppModel {

	public $actsAs = array(
		'EvRelatedItems.Relatable' => array(
			'models' => array(
				'EvCategory.Category' => array(),
				'EvShop.Brand' => array()
			)
		),
		'MetaData.Meta',
		'Searchable.Searchable' => array(),
		'Routable.Routable' => array(
			'actual' => 'products/view/:primaryKey'
		)
	);

	public $imageSlots = array(
		'main' => array(
			'slots' => 5,
			'fields' => array(
				'alt'
			)
		),
		'listing' => array(
			'label' => 'List',
			'slots' => 1,
			'fields' => array(
				'alt'
			)
		)
	);

	public $hasMany = array(
		'ProductVariant' => array(
			'className' => 'EvShop.ProductVariant',
			'dependent' => true,
			'cascade' => true
		)
	);

	public $belongsTo = array(
		'VatRate' => array(
			'className' => 'EvCheckout.VatRate'
		)
	);

	public $hasAndBelongsToMany = array(
		'ProductAttributeOption' => array(
			'className' => 'EvShop.ProductAttributeOption',
			'joinTable' => 'ev_shop_product_attribute_options_products'
		)
	);

	public $order = array('Product.modified' => 'DESC');

	public $validate = array(
		'name' => array(
			'required' => array(
				'rule' => 'notEmpty',
				'message' => 'Product name is mandatory'
			),
			'maxLength' => array(
				'rule' => array('maxLength', 250),
				'message' => 'No more tham 250 characters long'
			)
		),
		'SKU' => array(
			'maxLength' => array(
				'rule' => array('maxLength', 45),
				'message' => 'No more tham 45 characters long',
				'allowEmpty' => true
			)
		),
		'lead_time' => array(
			'maxLength' => array(
				'rule' => array('maxLength', 45),
				'message' => 'No more tham 45 characters long',
				'allowEmpty' => true
			)
		)
	);

/**
 * Gets product out ready for editing in the Admin
 * @param	int 	$id 	Product ID
 * @param	array	$query	Any additional query parameters you wish to be passed through
 * @return 	mixed	False if no product or a find('first') array
 */
	public function readForEdit($id, $query = array()) {
		if($this->hasBehaviour('EvRelatedItems.Relatable')) {

			$query = $this->containRelatedItems($query);

		}

		$query['contain'][] = 'ProductAttributeOption';

		return parent::readForEdit($id, $query);
	}

/**
 * Gets product out read for displaying on the frontend
 *
 * @param int $id Product ID
 * @param array $query Any additional query parameters you wish to be passed through
 * @return bool|array False if no product or a find('first') array
 */
	public function readForView($id, $query = array()) {
		$query['contain']['ProductVariant'] = array(
			'conditions' => array(
				'ProductVariant.is_removed <>' => true
			)
		);
		$query['conditions'][] = array('Product.is_active' => true);

		return $this->readForEdit($id, $query);
	}

/**
 * Helper function to build the query params needed to filter products by category.
 *
 * @param  mixed $filterParams 	This is the parameters as passed into readForListingQuery
 * @param  array  $query  		The find params so far to append to
 * @return array                The initail find params with parts added for category filtering
 */
	protected function _filterByCategoryQuery($filterParams, $query = array()) {
		$query['joins'][] = array(
			'table' => 'ev_related_items_related_items',
			'alias' => 'RelatedItem',
			'type' => 'RIGHT',
			'conditions' => array(
				'RelatedItem.model_id = Product.id',
				'RelatedItem.model' => 'Product',
				'RelatedItem.related_model' => 'Category'
			)
		);

		// Build up an array of categories to search within (we want to include all child
		// categories in our condition).
		if (is_array($filterParams)) {

			$categories = $filterParams;

			foreach ($filterParams as $category) {
				$childCategories = $this->RelatedCategory->children($category, false, 'RelatedCategory.id');
				$categories = array_merge(
					$categories,
					array_keys(Hash::combine($childCategories, '{n}.RelatedCategory.id'))
				);
			}

		} else {

			$childCategories = $this->RelatedCategory->children($filterParams, false, 'RelatedCategory.id');
			$categories = array_keys(Hash::combine($childCategories, '{n}.RelatedCategory.id'));
			$categories[] = (int)$filterParams;

		}

		$query['conditions']['RelatedItem.related_model_id'] = $categories;

		return $query;
	}

/**
 * Helper function to build the query params needed to filter products by search term
 * @param  mixed $filterParams 	This is the parameters as passed into readForListingQuery
 * @param  array  $query  		The find params so far to append to
 * @return array                The initail find params with parts added for category filtering
 */
	protected function _filterBySearchQuery($filterParams, $query = array()) {
		$query['joins'][] = array(
			'table' => 'search_indices',
			'alias' => 'SearchIndex',
			'conditions' => array(
				'SearchIndex.model' => 'Product',
				'SearchIndex.association_key = Product.id'
			)
		);

		// We want to perform a MATCH AGAINST for the search index and do a LIKE on the
		// product name to overcome issues with MySQL's stop words like 'never' which
		// may important when searching for a product.
		$query['conditions']['OR'] = array(
			'MATCH(SearchIndex.data) AGAINST(? IN BOOLEAN MODE)' => $filterParams,
			'Product.name LIKE' => '%' . $filterParams . '%'
		);

		return $query;
	}

/**
 * Function that builds up search query array for products
 * @param  array  $filters An array of filter params (order, product_attribute_options, product_attributes, category)
 * @return array 		   Find params array
 */
	public function readForListingQuery($filters = array()) {
		$query = array();

		// These are the filter params that this method handles.
		// We will try to handle any others by looking for filter methods
		// category -> filterByCategoryQuery($filters['category'])
		// search -> filterBySearchQuery($filters['category'])
		$knownFilterOptions = array('product_attribute_options', 'product_attributes', 'order');
		$unknownFilterOptions = array_diff(array_keys($filters), $knownFilterOptions);

		foreach ($unknownFilterOptions as $filterOption) {

			$methodName = "_filterBy" . Inflector::camelize($filterOption) . "Query";
			if (method_exists($this, $methodName)) {
				$query = $this->$methodName($filters[$filterOption], $query);
			}

		}

		// Process filters
		if (!empty($filters['product_attribute_options'])) {

			// Get all possible matching variants

			$subqueryParams = array(
				'fields' => array('ProductVariant.id'),
				'joins' => array(
					array(
						'table' => '`ev_shop_product_attribute_options_product_variants`',
						'alias' => 'ProductAttributeOptionProductVariant',
						'type' => 'RIGHT',
						'conditions' => array(
							'ProductVariant.id = ProductAttributeOptionProductVariant.product_variant_id'
						)
					)
				),
				'conditions' => array(
					'ProductVariant.price <>' => ""

				),
				'group' => 'ProductVariant.id',
				'order' => 'ProductVariant.price ASC'
			);

			// If there is only one product_attribute_options then there can only be one attribute so can skip this count
			// (saves us always having to pass it in)
			if (count($filters['product_attribute_options']) > 1) {

				// The commented out line below relies on their always been a product_attriubtes variable passed in.
				// $attributeCount = count($filters['product_attributes']);

				// Slightly more inefficient but reduces the amount of posted and passed data significantly
				$countQueryResult = $this->ProductAttributeOption->find('first', array(
						'fields' => array(
							'COUNT( DISTINCT product_attribute_id) AS count'
						),
						'conditions' => array(
							'id' => $filters['product_attribute_options']
						)
					)
				);

				$attributeCount = $countQueryResult[0]['count'];

			} else {

				$attributeCount = 1;

			}

			$subqueryParams['order'] =  'ProductVariant.price ASC';

			$subqueryParams['group'] = 'ProductVariant.id HAVING COUNT(DISTINCT ProductAttributeOptionProductVariant.product_attribute_option_id) = "' . $attributeCount . '"';
			$subqueryParams['conditions']['ProductAttributeOptionProductVariant.product_attribute_option_id'] = $filters['product_attribute_options'];

		}

		if (isset($subqueryParams)) {

			$allVariantIds = $this->ProductVariant->find('list', $subqueryParams);

			$subqueryParams2 = array(
				'fields' => array('ProductVariant.id'),
				'conditions' => array(
					'ProductVariant.product_id = Product.id',
					'ProductVariant.id' => $allVariantIds
				),
				'group' => 'ProductVariant.id',
				'order' => 'ProductVariant.price ASC',
				'limit' => 1 // Only want the cheapest variant
			);

		}

		$defaultOrder = array(
			'field' => 'price',
			'direction' => 'ASC'
		);

		if (!empty($filters['order'])) {

			if (!is_array($filters['order'])) {
				$filters['order']['field'] = $filters['order'];
			}

		} else {

			$filters['order'] = array();

		}

		$filters['order'] = array_merge($defaultOrder, $filters['order']);

		switch ($filters['order']['field']) {
			case 'price':
				$query['order'] = "Product__min_price ".$filters['order']['direction'];
				break;

			default:
				$query['order'] = "Product__min_price ASC";
				break;
		}

		$this->virtualFields['min_price'] = 'ProductVariant.price';
		$this->virtualFields['min_rrp'] = 'ProductVariant.rrp';

		$query['contain'][] = 'ListingImage';
		$query['contain']['Image'] = array(
			'limit' => 1
		);

		$query['joins'][] = array(
			'table' => 'ev_shop_product_variants',
			'alias' => 'ProductVariant',
			'type' => 'LEFT',
			'conditions' => array(
				'ProductVariant.product_id = Product.id',
				isset($subqueryParams2) ? 'ProductVariant.id = (' . $this->ProductVariant->subquery($subqueryParams2) . ')' : ""
			)
		);

		// Has a product variant
		$query['conditions'][] = 'ProductVariant.id IS NOT NULL';

		$query['conditions'][] = array('Product.is_active' => true);
		$query['conditions'][] = array('Product.is_removed !=' => true);
		$query['conditions'][] = array('ProductVariant.is_removed !=' => true);

		$query['group'] = 'Product.id';

		return $query;
	}

	public function readForListing($query = array()) {
		$query = $this->readForListingQuery($query);
		$products = $this->find('all', $query);

		return $products;
	}

/**
 * Gets the related products for a given product
 * @param int 	$id	Product ID
 */
	public function getRelatedProducts($id, $query = array()) {
		$subqueryParams = array(
			'fields' => array('ProductVariant.id'),
			'conditions' => array(
				'ProductVariant.product_id = Product.id',
				'ProductVariant.price <>' => ""

			),
			'group' => 'ProductVariant.id',
			'order' => 'ProductVariant.price ASC',
			'limit' => 1 //Only want the cheapest variant
		);

		$this->virtualFields['min_price'] = 'ProductVariant.price';
		$this->virtualFields['min_rrp'] = 'ProductVariant.rrp';

		$query['contain']['Image'] = array(
			'limit' => 1
		);
		$query['contain']['ListingImage'] = array(
			'limit' => 1
		);

		$query['joins'][] = array(
			'table' => 'ev_shop_product_variants',
			'alias' => 'ProductVariant',
			'conditions' => array(
				'ProductVariant.product_id = Product.id',
				'ProductVariant.price <>' => '',
				isset($subqueryParams) ? 'ProductVariant.id = (' . $this->ProductVariant->subquery($subqueryParams) . ')' : ""
			)
		);

		$query['joins'][] = array(
			'table' => 'ev_related_items_related_items',
			'alias' => 'RelatedItem',
			'conditions' => array(
				'RelatedItem.model' => 'Product',
				'RelatedItem.related_model' => 'Product',
				'RelatedItem.model_id' => $id,
				'RelatedItem.related_model_id = Product.id'
			)
		);

		$products = $this->find('all', $query);

		unset($this->virtualFields['min_price']);
		unset($this->virtualFields['min_rrp']);

		return $products;
	}

/**
 * We override the default saveAll to also clean up any product variants that dont
 * match the current attributes
 * @todo  Find a better way to clean up redundantVariants
 * @param  array  $data
 * @param  array  $options
 * @return boolean Indicates whether the save was successful
 */
	public function saveAll($data = array(), $options = array()) {
		$success = parent::saveAll($data, $options);

		if ($success) {
			$this->ProductVariant->cleanUpRedundantVariants($this->id);
		}

		return $success;
	}

}
