<?php

App::uses('EvShopAppController', 'EvShop.Controller');

/**
 * Product App Controller
 *
 * @package EvShop/Product
 */
class ProductsController extends EvShopAppController {

	public $components = array(
		'EvRelatedItems.RelatedItems'
	);

	public $paginate = array(
		'paramType' => 'querystring'
	);

	public function __construct($request = null, $response = null) {
		parent::__construct($request, $response);

		// Add the Product helper.
		$this->helpers[] = 'EvShop.Product';
		$this->adminActions[] = 'admin_attributes';

		return;
	}

	public function beforeFilter() {
		parent::beforeFilter();

		$this->Auth->allow(array(
			'ajax_ProductAttributeOptions',
			'index',
			'view',
			'search',
			'add_to_basket'
		));
	}

/**
 * Product admin forms toolbars.
 *
 * @param int $id
 * @return array
 */
	protected function _adminFormToolbar($id = null) {
		$actions = parent::_adminFormToolbar($id);

		if ($id !== null) {
			$actions['Attributes'] = array(
				'url' => array('action' => 'attributes', $id),
			);
		}

		return $actions;
	}

/**
 * Product add/edit form.
 *
 * @param int $id
 * @return void
 */
	public function admin_edit($id = null) {
		$Product = $this->{$this->modelClass};

		if ($Product->hasBehaviour('EvRelatedItems.Relatable')) {

			// Process the related items.
			if (!empty($this->request->data)) {

				// We need to check that our data validates before processing the
				// related items as this method will remove existing data before
				// saving the product.
				$Product->set($this->request->data);
				if ($Product->validates()) {
					$this->request->data = $this->RelatedItems->processInputs($Product, $this->request->data, $id);
				}

			}
		}

		parent::admin_edit($id);

		// Get the product attributes for the inventory tables
		$this->ProductAttribute = ClassRegistry::init('EvShop.ProductAttribute');
		$this->set('productAttributeCombinations', $this->ProductAttribute->getProductOptionCombinations($id));
		$this->set('productAttributes', $this->ProductAttribute->find('all', array(
					'contain' => array(
						'ProductAttributeOption' => array(
							'order' => 'sequence ASC'
						)
					)
				)
			)
		);
		$this->set('primaryAttribute', $this->ProductAttribute->getPrimaryAttribute($id));

		// Get the product variants
		$this->request->data['ProductVariant'] = $Product->ProductVariant->getProductVariantsIndexedByAttributeOptions($id);

		$this->view = 'EvShop.Products/admin_form';

		return;
	}

/**
 * Manage a product's attributes.
 *
 * @param int $id Product ID
 * @return void
 */
	public function admin_attributes($id) {
		$Product = $this->{$this->modelClass};
		$data = $Product->readForEdit($id);

		if (empty($data)) {
			throw new NotFoundException();
		}

		if (!empty($this->request->data)) {

			$this->request->data['Product']['id'] = $id;

			if ($Product->saveAll($this->request->data, array('deep' => true))) {

				$this->Session->setFlash(
					array(
						'title' => 'Product Attributes Updated',
						'description' => __('%s has been successfully updated!', $data['Product']['name']) . ' ' . __('Check stock and prices.')
					),
					'flash_success'
				);

				return $this->redirect(array('action' => 'edit', $id));

			} else {

				$this->Session->setFlash(
					array(
						'title' => __('Save failed'),
						'description' => __('Failed to save %s, see below for details', 'product attributes')
					),
					'flash_fail'
				);

			}

		} else {
			$this->request->data = $data;
		}

		// Get the product attributes for the inventory tables
		$this->ProductAttribute = ClassRegistry::init('EvShop.ProductAttribute');
		$this->set('productAttributeCombinations', $this->ProductAttribute->getProductOptionCombinations($id));
		$this->set('productAttributes', $this->ProductAttribute->find('all', array(
					'contain' => array(
						'ProductAttributeOption' => array(
							'order' => 'sequence ASC'
						)
					)
				)
			)
		);
		$this->set('primaryAttribute', $this->ProductAttribute->getPrimaryAttribute($id));

		$this->set('toolbar', $this->_adminFormToolbar($id));

		$this->view = 'EvShop.Products/admin_attributes';

		return;
	}

/**
 * Used to populate form drop down selects
 *
 * Override in your controller to customise
 */
	protected function _adminPopulateLookups() {
		$Product = $this->{$this->modelClass};
		$this->set('relatedBrands', $Product->RelatedBrand->find('list'));
		// $this->set('relatedProducts', $Product->find('list'));
		$this->set('relatedCategories', $Product->RelatedCategory->find('list'));
		$this->set('vatRates', $Product->VatRate->find('list'));
	}

/**
 * Defines the fields displayed in the admin_form.
 *
 * @return array
 */
	protected function _adminFormFields() {
		$fields = parent::_adminFormFields();

		$Product = $this->{$this->modelClass};

		if ($Product->hasBehaviour('EvRelatedItems.Relatable')) {

			// Insert related fields
			if (isset($fields['is_featured'])) {
				$insertFieldsAfter = 'is_featured';
			} elseif (isset($fields['Product.is_featured'])) {
				$insertFieldsAfter = 'Product.is_featured';
			} elseif (isset($fields['Product.description'])) {
				$insertFieldsAfter = 'Product.description';
			} else {
				$insertFieldsAfter = 'description';
			}
			$fields = ArrayUtil::addAfter($fields, $insertFieldsAfter, $this->RelatedItems->generateFormFields($Product));
		}

		$fields['Product.SKU']['label'] = 'SKU';

		$fields['Route.alias']['url_prefix'] = 'products';

		return $fields;
	}

/**
 * Defines the fields displayed in a filter form.
 *
 * Defaults to id, display name, is_active, created and modified fields.
 *
 * Override in your controller to customise.
 *
 * Format:
 *
 * 'id' => array(
 *         'label' => label text,
 *         'type' => data type of control
 *         'compare' => SQL query. e.g. Model.field='%s' (where %s is replace by the value of the field)
 *         'default' => value you want the data filtering by by default
 * )
 */
	protected function _adminFilterFields() {
		$Model = $this->{$this->modelClass};
		$fields = parent::_adminFilterFields();

		$fields['RelatedCategory.id'] = array(
			'label' => 'Category',
			'type' => 'select',
			'options' => $Model->RelatedCategory->find('list'),
			'compare' => array(
				'RelatedCategory.id' => '%s'
			)
		);

		$fields['RelatedBrand.id'] = array(
			'label' => 'Brand',
			'type' => 'select',
			'options' => $Model->RelatedBrand->find('list'),
			'compare' => array(
				'RelatedBrand.id' => '%s'
			)
		);

		unset($fields['Product.id']);
		unset($fields['Product.created']);
		unset($fields['Product.modified']);

		return $fields;
	}

/**
 * Used to populate filter form drop down selects
 *
 * Override in your controller to customise
 */
	protected function _adminPopulateFilterLookups() {
		$Model = $this->{$this->modelClass};

		parent::_adminPopulateFilterLookups();
	}

/**
 * Pagination settings for admin_index
 *
 * Override in your own controller to customise
 *
 * @return array
 */
	protected function _adminIndexPaginate() {
		$Model = $this->{$this->modelClass};
		$params = parent::_adminIndexPaginate();

		$params['joins'][] = array(
			'table' => 'ev_related_items_related_items',
			'alias' => 'RelatedItemCategory',
			'type' => 'LEFT',
			'conditions' => array(
				'RelatedItemCategory.model' => $Model->alias,
				'RelatedItemCategory.related_model' => 'Category',
				'Product.id = RelatedItemCategory.model_id'
			)
		);

		$params['joins'][] = array(
			'table' => 'ev_category_categories',
			'alias' => 'RelatedCategory',
			'type' => 'LEFT',
			'conditions' => array(
				'RelatedItemCategory.related_model_id = RelatedCategory.id'
			)
		);

		$params['joins'][] = array(
			'table' => 'ev_related_items_related_items',
			'alias' => 'RelatedItemBrand',
			'type' => 'LEFT',
			'conditions' => array(
				'RelatedItemBrand.model' => $Model->alias,
				'RelatedItemBrand.related_model' => 'Brand',
				'Product.id = RelatedItemBrand.model_id'
			)
		);

		$params['joins'][] = array(
			'table' => 'ev_shop_brands',
			'alias' => 'RelatedBrand',
			'type' => 'LEFT',
			'conditions' => array(
				'RelatedItemBrand.related_model_id = RelatedBrand.id'
			)
		);

		$params['group'] = $Model->alias . '.id';

		return $params;
	}

/**
 * The listing page for products
 *
 * @param integer $categoryId ID of the category to filter by
 * @return void
 */
	public function index($categoryId = null) {
		$Product = $this->{$this->modelClass};

		$params = $this->_processProductListingFilter();

		if (!empty($categoryId)) {
			$params['category'] = $categoryId;
		}

		$this->_fetchAdditionalIndexData($params);

		// Setup $this->Request->data so the filter form shows the correct selections
		if (!empty($this->request->query['data'])) {

			$this->request->data = $this->request->query['data'];

		} elseif (!empty($params['product_attribute_options'])) {

			foreach ($params['product_attribute_options'] as $productAttributeOption) {

				$this->request->data['ProductSearch']['ProductAttributeOption'][$productAttributeOption] = $productAttributeOption;

			}
		}

		$this->paginate = array_merge($this->paginate, $Product->readForListingQuery($params));

		$this->set('products', $this->paginate());


		// Store the listing page into the EvChckout listing session to allow
		// us to come back here on "Continue Shopping" links.
		$this->Session->write('EvCheckout.lastListingPage', Router::reverse($this->request));

		$this->_getProductFilters($params);

		$this->view = 'EvShop.Products/index';

		return;
	}

/**
 * Product page.
 *
 * @param integer $id product Id
 * @return void
 */
	public function view($id) {
		$this->_addToBasket();

		$Product = $this->{$this->modelClass};

		$product = $Product->readForView($id);
		$this->set('data', $product);
		$this->Meta->set($product);

		if (empty($product)) {
			throw new NotFoundException('Product not found.');
		}

		// Set the product ID for the form data.
		$this->request->data['Product']['id'] = $id;

		$relatedProducts = $Product->getRelatedProducts(
			$id,
			array(
				'limit' => 5
			)
		);
		$this->set('relatedProducts', $relatedProducts);

		$ProductAttribute = &$Product->ProductAttributeOption->ProductAttribute;
		$this->set('productAttributes', $ProductAttribute->getProductAttributes($id));

		$this->view = 'EvShop.Products/view';

		return;
	}

/**
 * Adds a product to the basket.
 *
 * @param integer $productVariantId
 * @return boolean returns false if there is a problem adding to the basket
 */
	protected function _addToBasket($productVariantId = null) {
		$Model = $this->{$this->modelClass};

		if ($this->checkProcessForm('EvShop.add_to_basket') === true) {

			if ($productVariantId === null) {

				$productVariant = $Model->ProductVariant->findByProductAttributeOptions(
					$this->request->data['Product']['id'],
					!empty($this->request->data['ProductVariant']['ProductAttributeOption']) ? $this->request->data['ProductVariant']['ProductAttributeOption'] : array()
				);

				if (!empty($productVariant)) {

					$productVariantId = $productVariant['ProductVariant']['id'];

				} else {

					throw new InternalErrorException('Unable to find product variant to add to basket.');
				}

			}

			$this->loadModel('EvCheckout.Order');

			// Get cart ID
			$orderId = $this->Order->getCartId();

			// Get Item ready for basket
			$addToBasketParams = $Model->ProductVariant->prepareForBasket($productVariantId);

			// Check the item is available before proceeding.
			if ($addToBasketParams === false) {

				$this->Session->setFlash(__('Sorry, this item is no longer available'), 'flash_fail');

			} else {

				$addToBasketParams['order_id'] = $orderId;
				$addToBasketParams['quantity'] = !empty($this->request->data['OrderItem']['quantity']) && is_numeric($this->request->data['OrderItem']['quantity']) ? $this->request->data['OrderItem']['quantity'] : 1;

				$result = $this->Order->addToBasket($addToBasketParams);

				if ($result === 1 || $result === -1) {

					if ($result === -1) {
						// Quantity was modified due to stock levels. Warn the customer.
						$this->Session->setFlash(__('Sorry, we have insufficient stock for that item so have reduced the quantity in your basket'), 'flash_info');
					}

					$this->redirect(array(
						'plugin' => 'ev_checkout',
						'controller' => 'orders',
						'action' => 'basket'
					));
					return;

				} else {

					return false;

				}

			}

		}

		return true;
	}

/**
 * Searches and displayings listing page for products that match search
 * @param  string 	$searchTerm 	The product search term
 */
	public function search($searchTerm) {
		$Product = $this->{$this->modelClass};
		$this->set('products', $Product->search($searchTerm, array('contain' => 'Product')));

		//Get filter options
		$ProductAttribute = ClassRegistry::init('EvShop.ProductAttribute');
		$this->set('filters', $ProductAttribute->getProductFilters($params));

		$this->view = 'EvShop.Products/index';
	}

/**
 * Gets the available product attribute options for a product given the already selected product attribute options
 * @param  int    $productId               Product Id
 * @param  int    $attributeId             Attribute Id
 * @param  string $currentAttributeOptions In the format of "9,4,20"
 * @return json string                     Includes all the product attribute options available and the min_price of the product variants associated
 */
	public function ajax_ProductAttributeOptions($productId, $attributeId, $currentAttributeOptions = "") {
		$Product = $this->{$this->modelClass};
		$ProductAttribute = $Product->ProductAttributeOption->ProductAttribute;

		//Convert current attribute options into array
		if (!empty($currentAttributeOptions)) {
			$currentAttributeOptionsArray = explode(",", $currentAttributeOptions);
		} else {
			$currentAttributeOptionsArray = array();
		}

		echo json_encode($ProductAttribute->getAttributeOptionsForProduct($productId, $attributeId, $currentAttributeOptionsArray));

		die();
	}

/**
 * Gets additional information needed for the index view
 * This includes getting information for the category if set
 * @param  array $params search filter params
 */
	protected function _fetchAdditionalIndexData($params) {
		$Product = $this->{$this->modelClass};

		if (isset($params['search'])) {

			$this->set('searchTerm', $params['search']);

		}

		if (isset($params['category'])) {

			//Get the category's information
			$category = $Product->RelatedCategory->readForView($params['category']);
			$this->set('category', $category);

		}

		if (isset($category) && $category) {

			// Pull out meta data, this is not done by the behaviour because the alias is different
			$this->loadModel('MetaData.MetaData');
			$metaData = $this->MetaData->find('first', array(
					'conditions' => array(
						'model' => 'Category',
						'model_id' => $category['RelatedCategory']['id']
					)
				)
			);

			$category['MetaData'] = $metaData['MetaData'];

			$this->Meta->set($category, 'RelatedCategory');

		}

		return;
	}

/**
 * Processes the incoming product filters which are set in the request->data or request->query
 * @return array An array of search params
 */
	protected function _processProductListingFilter() {
		// If we're performing a search convert the search request into a query string.
		if (isset($this->request->data['ProductSearch']['Search'])) {

			$this->redirect(array(
					'action' => 'index',
					'?' => array(
						'data' => array(
							'ProductSearch' => array(
								'Search' => $this->request->data['ProductSearch']['Search']
							)
						)
					)
				)
			);

			return;

		}

		// Redirect to convert filters into a query string.
		if (isset($this->request->data['ProductSearch'])) {

			$query = array(
				'data' => array(
					'ProductSearch' => $this->request->data['ProductSearch']
				)
			);
			if (!empty($this->request->query['data']['ProductSearch']['Search'])) {
				$query['data']['ProductSearch']['Search'] = $this->request->query['data']['ProductSearch']['Search'];
			}
			unset($this->request->query['data']);
			// Reset the page when filtering/sorting the current page is no longer relevant and may no longer exist!
			unset($this->request->query['page']);
			$query = array_merge_recursive($this->request->query, $query);

			$this->redirect($this->here . '?' . http_build_query($query));

			return;

		}

		$params = array();

		$this->request->data = array_merge_recursive($this->request->query, array('data' => $this->request->data));

		// Process params and get into one array for parsing
		$namedParamsInput = array();

		// Process named and request variables
		if (isset($this->request->category)) {

			$namedParamsInput['category'] = $this->request->category;

			$this->request->params['named']['category'] = $this->request->category;

		} elseif (isset($this->request->named['category'])) {

			$namedParamsInput['category'] = $this->request->named['category'];

		}

		if (isset($this->request->ProductAttributeOption)) {

			$namedParamsInput['product_attribute_options'] = array($this->request->ProductAttributeOption);

		}

		if (!empty($this->request->named['ProductAttributeOption'])) {

			$namedParamsInput['product_attribute_options'] = array($this->request->named['ProductAttributeOption']);

		}

		if (!empty($this->request->named['ProductAttribute'])) {

			$namedParamsInput['product_attributes'] = array($this->request->named['ProductAttribute']);

		}

		// Process data handed by request data
		$requestParamsInput = array();

		if (!empty($this->request->data['data']['ProductSearch']['ProductAttributeOption'])) {

			$productAttributeOptions = Hash::extract($this->request->data['data']['ProductSearch']['ProductAttributeOption'], '{n}');
			$requestParamsInput['product_attribute_options'] = $productAttributeOptions;

		}

		if (isset($this->request->data['data']['ProductSearch']['Search'])) {

			$requestParamsInput['search'] = $this->request->data['data']['ProductSearch']['Search'];

		}

		if (!empty($this->request->data['data']['ProductSearch']['Sort'])) {

			switch ($this->request->data['data']['ProductSearch']['Sort']) {

				case 'price-desc':
					$requestParamsInput['order'] = array(
							'field' => 'price',
							'direction' => 'DESC'
					);
					break;
				case 'price-asc':
				default:
					$requestParamsInput['order'] = array(
							'field' => 'price',
							'direction' => 'ASC'
					);
					break;

			}

		}

		$params = array_merge($namedParamsInput, $requestParamsInput);

		return $params;
	}

/**
 * Sets up the product filters for the frontend
 * @param	array	Any parameters to filter the list of categories by
 */
	protected function _getProductFilters($params = array()) {
		$Product = $this->{$this->modelClass};
		$this->set('filters', $Product->ProductAttributeOption->ProductAttribute->getProductFilters($params));
	}

}
