<?php
App::uses('AppHelper', 'View/Helper');

class ProductsHelper extends AppHelper {

	public $helpers = ['Number'];

	/**
	 * for a product with single variant, extract the requests data
	 *
	 * @param 	string 		The column to extract
	 * @param 	array 		The Data array
	 * @return 	mixed 		The extracted data or bool false if multiple variants
	 */
	public function singleVariantData($item, $data) {
		if (count($data['Variant']) > 1) {
			return false;
		}

		return Hash::get($data, 'Variant.0.' . $item);
	}

	/**
	 * get the option groups for the variants
	 *
	 * @param  	array 	Product Data array
	 * @return 	array 	Array of option groups
	 */
	public function getOptionGroups($data) {
		if (! empty($data['Variant']['0']['Option'])) {
			return Hash::combine($data['Variant'], '{n}.Option.{n}.OptionGroup.id', '{n}.Option.{n}.OptionGroup.name');
		}

		return array();
	}

	/**
	 * given the variant array and the option group
	 * return the option name
	 *
	 * @param 	array 	Variant Array
	 * @param 	int 	Option group Id
	 * @return 	string 	Option value
	 */
	public function getOption($Variant, $groupId) {
		return Hash::get($Variant, 'Option.' . $groupId . '.name', '-');
	}

/**
 * Given an array of VariantPricings, the pricing that matches the current currency ID will be found
 * and an array will be returned with the price. If the variant pricing contains a sale_price then
 * that will be returned as well as the lowest_price and the original price returned as the price.
 * @param  array $pricings An array of variant pricings for a specific variant
 * @return array           An array containing the price and the sale_price if it exists.
 */
	public function getVariantPrice($pricings) {
		$possiblePricings = Hash::combine($pricings, '{n}.currency_id', '{n}');
		$currencyId = CakeSession::read('EvCurrency.currencyId');

		$pricing = $possiblePricings[$currencyId];

		if (isset($pricing['sale_price']) && !empty($pricing['sale_price'])) {
			$originalPrice = $pricing['price'];
			if (Configure::read('EvShop.displayTaxAsVat')) {
				if (isset($pricing['price_incTax']) && !empty($pricing['price_incTax'])) {
					$originalPrice = $pricing['price_incTax'];
				}
			}
			return [
				'price' => $originalPrice,
				'sale_price' => $pricing['lowest_price']
			];
		} else {
			return [
				'price' => $pricing['lowest_price'],
			];
		}
	}

/**
 * Given a set of variant images, remove the restricted images so that only the generic images are left.
 * @param  array $allVariantImages An array of variant images
 * @return array                  An array without an restricted images in
 */
	public function getGenericVariantImages($allVariantImages) {
		foreach ($allVariantImages as $imageKey => $image) {
			if (isset($image['is_restricted']) && $image['is_restricted']) {
				unset($allVariantImages[$imageKey]);
			}
		}

		return $allVariantImages;
	}

/**
 * Given a set of variant images, remove the generic images so that only the restricted images are left.
 * Also remove any restricted images that don't have any options associated with them.
 * @param  array $allVariantImages An array of variant image
 * @return array                   An array without any generic images in,
 */
	public function getRestrictedVariantImages($allVariantImages) {
		foreach ($allVariantImages as $imageKey => $image) {
			if (!isset($image['is_restricted']) || !$image['is_restricted']) {
				unset($allVariantImages[$imageKey]);
			}

			if (!isset($image['VariantImageOption']) || empty($image['VariantImageOption'])) {
				unset($allVariantImages[$imageKey]);
			}
		}

		return $allVariantImages;
	}

/**
 * Given a product variant and a set of variant images, find all the images that are applicable to be displayed
 * for the variant. The variant images need to have options associated with them otherwise they will not be
 * returned.
 * @param  array $variant       An array containing the information about a specific variant
 * @param  array $variantImages An array of variant images.
 * @return array                An array of variant images that are applicable to the specified variant.
 */
	public function getRestrictedImagesForVariant($variant, $variantImages) {
		$imageArray = [];

		if (isset($variant['Option']) && !empty($variant['Option'])) {
			$allVariantOptions = Hash::extract($variant['Option'], '{n}.id');

			$restrictedImages = $this->getRestrictedVariantImages($variantImages);

			foreach ($restrictedImages as $image) {
				$currentGroupsChecked = [];

				foreach ($image['VariantImageOption'] as $imageOption) {
					if (!isset($currentGroupsChecked[$imageOption['option_group_id']])) {
						$currentGroupsChecked[$imageOption['option_group_id']] = false;
					}

					if (in_array($imageOption['option_id'], $allVariantOptions)) {
						$currentGroupsChecked[$imageOption['option_group_id']] = true;
					}
				}

				if (!in_array(false, $currentGroupsChecked)) {
					$imageArray[] = $image;
				}
			}
		}

		return $imageArray;
	}

/**
 * Basic GETTER for getting sale price of the product
 * It is better than accessing them directly as this way it can be changed without affecting the template
 * @param $product  Product array
 * @param $incVat Defaults to false, if set to true INC vat price is returned
 * @return  formatted currency Sale price
 */
	public function getSalePrice($product, $incVat = false) {
		$currencies = $this->_View->viewVars['currencies'];
		if (!empty($product['was_price_ex_vat'])) {
			if ($incVat && !empty($product['was_price'])) {
				return $this->Number->currency($product['was_price'], $currencies[CakeSession::read('EvCurrency.currencyId')]);
			} else {
				return $this->Number->currency($product['was_price_ex_vat'], $currencies[CakeSession::read('EvCurrency.currencyId')]);
			}
		} else {
			return false;
		}
	}

/**
 * Basic GETTER for getting RRP price of the product
 * It is better than accessing them directly as this way it can be changed without affecting the template
 * @param $product  Product array
 * @param $incVat Defaults to false, if set to true INC vat price is returned
 * @return  formatted currency RRP price
 */
	public function getRrpPrice($product, $incVat = false) {
		$currencies = $this->_View->viewVars['currencies'];
		if ($incVat) {
			return $this->Number->currency($product['rrp_price'], $currencies[CakeSession::read('EvCurrency.currencyId')]);
		} else {
			return $this->Number->currency($product['rrp_price_ex_vat'], $currencies[CakeSession::read('EvCurrency.currencyId')]);
		}
	}

/**
 * Basic GETTER for getting price of the product
 * It is better than accessing them directly as this way it can be changed without affecting the template
 * @param $product  Product array
 * @return  formatted currency price
 */
	public function getPrice($product) {
		$currencies = $this->_View->viewVars['currencies'];
		return $this->Number->currency($product['price'], $currencies[CakeSession::read('EvCurrency.currencyId')]);
	}

/**
 * Basic GETTER for getting max price of the product
 * It is better than accessing them directly as this way it can be changed without affecting the template
 * @param $product  Product array
 * @return  formatted currency price
 */
	public function getMaxPrice($product) {
		$currencies = $this->_View->viewVars['currencies'];
		return $this->Number->currency($product['max_price'], $currencies[CakeSession::read('EvCurrency.currencyId')]);
	}

/**
 * Returns true if there is a WAS price to the product
 * @param  Product  $product Product array
 * @return boolean          Returns true if the item is on sale
 */
	public function isSalePrice($product) {
		return ($product['was_price_ex_vat'] > 0);
	}
}
