<?php

App::uses('AppComponent', 'Controller/Component');

class ProductsComponent extends AppComponent {

	/**
	 * get a product listing
	 *
	 * @param 	array 	query / paginate params
	 * @param 	bool 	Paginate results or not?
	 * @param   array   An array of order data (direction and currencyId)
	 * @return 	array 	Array of results
	 */
	public function listing($params = array(), $paginate = true, $orderData = null) {
		//Initialise the product model to find/paginate on
		$Product = EvClassRegistry::init('EvShop.Product');

		//Get the database source so we can use it to build sub-queries
		$database = $Product->getDataSource();

		//get the original virtual fields so they can be set back to normal after listing
		$productVirtualFields = $Product->virtualFields;

		//Add virtual fields to the product listing. By default, price and max_price are added
		$this->_addVirtualFieldsToProductListing($Product, $orderData);

		//Get the variant sub-query. By Default it returns all variants that match filter options ordered by price
		$variantSubQuery = $this->_getVariantSubQueryForProductListing($database, $orderData);

		$variantImagesSubQuery = $this->_getVariantImageSubQueryForProductListing($database, $orderData);

		$defaults = [
			'fields' => [
				'Product.*',
				'Variant.*',
				'Image.*'
			],
			'joins' => [
				[
					'table' => '(' . $variantSubQuery . ')',
					'alias' => 'Variant',
					'conditions' => [
						'Variant.product_id = Product.id'
					],
				],
				[
					'table' => '(' . $variantImagesSubQuery . ')',
					'alias' => 'Image',
					'type' => 'left',
					'conditions' => [
						'Image.model' => $Product->alias,
						'Image.model_id = Product.id'
					]
				]
			],
			'group' => 'Product.id',
			'conditions' => [
				'Product.is_active' => true
			]
		];

		$defaults = $this->_addAdditionalParamsToListing($Product, $defaults, $orderData);

		$params = array_merge_recursive($defaults, $params);

		if ($paginate && ! empty($this->_controller)) {
			$this->_controller->Paginator->settings = $params;
			$result = $this->_controller->Paginator->paginate(
				$Product
			);
		} else {
			$result = $Product->find('all', $params);
		}

		//Setting virtual fields back to default
		$Product->virtualFields = $productVirtualFields;

		return $result;
	}

	/**
	 * get a featured product listing
	 *
	 * @param 	array 	query / paginate params
	 * @param 	bool 	Paginate results or not?
	 * @return 	array 	Array of results
	 */
	public function featured($params = array(), $paginate = false) {
		$params = Hash::merge(
			array(
				'conditions' => array(
					'Product.is_featured' => 1
				)
			),
			$params
		);

		return $this->listing($params, $paginate);
	}

	/**
	 * get a listing by category
	 *
	 * @param 	int|array 	Category ID or array of category IDs
	 * @param 	array 		query / paginate params
	 * @param 	bool 	Paginate results or not?
	 * @param   array       An array to pass through to listing that if set will contain the direction of ordering and currency_id
	 * @return 	array 		Array of results
	 */
	public function listingByCategory($categoryId, $params = array(), $paginate = true, $orderData = null) {
		$params['joins'][] = [
			'table' => 'ev_shop_categories_products',
			'alias' => 'CategoriesProduct',
			'conditions' => [
				'CategoriesProduct.product_id = Product.id'
			]
		];

		$params['conditions']['CategoriesProduct.category_id'] = $categoryId;

		return $this->listing($params, $paginate, $orderData);
	}

	/**
	 * get a listing by a brand
	 *
	 * @param 	int|array 	brand ID or array of brand IDs
	 * @param 	array 		query / paginate params
	 * @param 	bool 	Paginate results or not?
	 * @return 	array 		Array of results
	 */
	public function listingByBrand($brandId, $params = array(), $paginate = true) {
		$defaults = array(
			'conditions' => array(
				'Product.brand_id' => $brandId
			)
		);

		// if it's many categories, and the group by and
		// also contain the categories so we know which is which
		if (is_array($brandId)) {
			$defaults['contain'][] = 'Brand';
		}

		$params = Hash::merge(
			$defaults,
			$params
		);

		return $this->listing($params, $paginate);
	}

	/**
	 * Add virtual fields to the query model to attach data that isn't included in the Model.
	 * Used by default to assign the price of the cheapest variant to it's product.
	 * @param model &$Model     The model that is being queried on that the virtual fields will be assigned to.
	 * @param array $orderData  An array containing any filters or sorts that are to be applied to the query.
	 */
	protected function _addVirtualFieldsToProductListing(&$Model, $orderData) {
		$Model->virtualFields['price'] = 'Variant.lowest_price';
		$Model->virtualFields['max_price'] = 'MAX(Variant.lowest_price)';
		$Model->virtualFields['was_price_ex_vat'] = 'Variant.was_price';
		$Model->virtualFields['rrp_price_ex_vat'] = 'Variant.rrp_price';

		return;
	}

	/**
	 * A default sub-query included in the product listing that returns all the variants ordered by their price,
	 * sale price is used if available. Uses the "_addAdditionalParamsToVariantSubQuery" to filter variants by
	 * their options.
	 * @param  object $DBSource  The database source of the project, used to create the sub-query.
	 * @param  array $orderData  An array containing the relevant data to filter and sort the variants by.
	 * @return string            The sql statement that generates the table for the sub-query.
	 */
	protected function _getVariantSubQueryForProductListing($DBSource, $orderData) {
		//Get the variant model to build a query on it
		$Variant = EvClassRegistry::init('EvShop.Variant');

		//Set up the basic parms needed for the variant sub-query
		//Basic params join variant pricing and order by price
		$variantSubQueryParams = [
			'fields' => [
				'Variant.*',
				'VariantPricing.lowest_price AS lowest_price',
				'VariantPricing.rrp AS rrp_price',
				'IF(VariantPricing.sale_price < VariantPricing.price, VariantPricing.price, VariantPricing.sale_price) AS was_price'
			],
			'table' => $DBSource->fullTableName($Variant),
			'alias' => 'Variant',
			'joins' => [
				[
					'table' => 'ev_shop_variant_pricings',
					'alias' => 'VariantPricing',
					'conditions' => [
						'VariantPricing.variant_id = Variant.id'
					],
				],
			],
			'conditions' => [
				'Variant.is_active' => true
			],
			'group' => 'Variant.id',
			'order' => 'VariantPricing.lowest_price ASC'
		];

		//Add any extra filtering or ordering to variants
		$variantSubQueryParams = $this->_addAdditionalParamsToVariantSubQuery($variantSubQueryParams, $orderData);

		//Build the variant sub-query
		return $DBSource->buildStatement($variantSubQueryParams, $Variant);
	}

	/**
	 * Add extra parameters that aren't part of the default variant sub-query. Override this if you need to make
	 * changes/additions to the default variant sub query.
	 * @param array $currentParams An array containing the current parameters used for the variant sub-query.
	 * @param array $orderData     An array containing data passed to be used for adding relevant parameters.
	 */
	protected function _addAdditionalParamsToVariantSubQuery($currentParams, $orderData) {
		if (isset($orderData['option']) && !empty($orderData['option'])) {
			$currentParams = $this->_addOptionFiltersToVariantSubQuery($currentParams, $orderData['option']);
		}

		if (CakePlugin::loaded('EvCurrency')) {
			$currentParams['conditions']['VariantPricing.currency_id'] = CakeSession::read('EvCurrency.currencyId');
		}

		return $currentParams;
	}

	/**
	 * Adds a specific set of parameters to the variant sub-query to filter out variants by there options.
	 * The options to filter by are provided in the orderOptions variable listed under their optionGroup.
	 * Multiple joins are created for each option group to enforce that each variant meets the criteria.
	 * Joins are named with the option group included to prevent clashes.
	 * @param array $currentParams An array containing the current parameters used for the variant sub-query.
	 * @param array $orderData     An array containing data passed to be used for adding relevant parameters.
	 */
	protected function _addOptionFiltersToVariantSubQuery($currentParams, $orderOptions) {
		//Foreach option group we need to join on a separate set of option_variants and options, aliased with the
		//name of the option group to prevent clashes
		foreach ($orderOptions as $optionGroup => $options) {

			$groupName = str_replace(" ", "", $optionGroup);

			//Add the option_variant join
			$currentParams['joins'][] = [
				'table' => 'ev_shop_options_variants',
				'alias' => 'OptionVariantFor' . $groupName,
				'conditions' => [
					'OptionVariantFor' . $groupName . '.variant_id = Variant.id'
				]
			];

			//Add the option join
			$currentParams['joins'][] = [
				'table' => 'ev_shop_options',
				'alias' => 'OptionFor' . $groupName,
				'conditions' => [
					'OptionFor' . $groupName . '.id = OptionVariantFor' . $groupName . '.option_id',
					'OptionFor' . $groupName . '.name' => $options
				]
			];
		}

		return $currentParams;
	}

	/**
	 * A default sub-query that attempts to find a variant image that fulfills the most criteria
	 * provided in the orderData.
	 * @param  object $DBSource   The database source of the project, used to create the sub-query.
	 * @param  array  $orderData  An array containing the relevant data to filter and sort the variants by.
	 * @return string             The sql statement that generates the table for the sub-query.
	 */
	protected function _getVariantImageSubQueryForProductListing($DBSource, $orderData) {
		//Start building the Image sub-query by initialising the Image Model
		$Image = EvClassRegistry::init('EvCore.Image');

		$variantImageSubQueryParams = [
			'fields' => [
				'Image.*'
			],
			'table' => $DBSource->fullTableName($Image),
			'alias' => 'Image',
			'group' => 'Image.id',
			'order' => 'Image.model_id ASC'
		];

		$variantImageSubQueryParams = $this->_addAdditionalParamsToVariantImageSubQuery($variantImageSubQueryParams, $orderData);

		//Build the variant image sub-query
		return $DBSource->buildStatement($variantImageSubQueryParams, $Image);
	}

	/**
	 * Add extra parameters that aren't part of the default variant image sub-query. Override this if you need to make
	 * changes/additions to the default variant image sub query.
	 * @param array $currentParams An array containing the current parameters used for the variant image sub-query.
	 * @param array $orderData     An array containing data passed to be used for adding relevant parameters.
	 */
	protected function _addAdditionalParamsToVariantImageSubQuery($currentParams, $orderData) {
		if (isset($orderData['option']) && !empty($orderData['option'])) {
			$currentParams = $this->_addOptionFiltersToVariantImageSubQuery($currentParams, $orderData['option']);
		}

		return $currentParams;
	}

	/**
	 * Add the filtering joins to the variant image sub-query so that only images that meet the criteria provided
	 * in the orderOptions are chosen. If multiple images are found that meet the criteria, the image that contains
	 * the most selected options is selected.
	 * @param array $currentParams An array containing the current parameters used for the variant image sub-query.
	 * @param array $orderData     An array containing data passed to be used for adding relevant parameters.
	 */
	protected function _addOptionFiltersToVariantImageSubQuery($currentParams, $orderOptions) {
		//Add the joins to check by option
		$currentParams['joins'][] = [
			'table' => 'ev_shop_variant_image_options',
			'alias' => 'VariantImageOption',
			'type' => 'left',
			'conditions' => [
				'VariantImageOption.image_id = Image.id'
			]
		];

		$imageOptionJoin = [
			'table' => 'ev_shop_options',
			'alias' => 'Option',
			'type' => 'left',
			'conditions' => [
				'Option.id = VariantImageOption.option_id'
			]
		];

		$variantImageFilters = [];
		foreach ($orderOptions as $optionGroup => $options) {
			//Options don't need to be separated by group so chuck them all in
			$variantImageFilters = Hash::merge($variantImageFilters, $options);
		}

		$imageOptionJoin['conditions']['Option.name'] = $variantImageFilters;

		$currentParams['joins'][] = $imageOptionJoin;

		$currentParams['order'] = 'Image.model_id ASC, COUNT(Option.option_group_id) DESC';

		return $currentParams;
	}

	/**
	 * Check if specific ordering data is provided and if so call the function that adds the correct parameters to the
	 * query. Extend to add add your own parameters to the listing query.
	 * @param array $currentParams The current parameters of the query to change. Normally the defaults set in listing()
	 * @param array $orderData     The ordering data (direction and currencyId if it is set)
	 * @param Model $queryModel    The model that is being queried on, listing uses Product
	 */
	protected function _addAdditionalParamsToListing(&$queryModel, $currentParams, $orderData) {
		//Go through the passed ordering parameters and add it to the pagination query
		if (isset($orderData['price']) && !empty($orderData['price'])) {
			$currentParams = $this->_addPriceParamsToListing($currentParams, $orderData['price'], $queryModel);
		}

		if (isset($orderData['brand']) && !empty($orderData['brand'])) {
			$currentParams = $this->_addBrandParamsToListing($currentParams, $orderData['brand'], $queryModel);
		}

		return $currentParams;
	}

	/**
	 * Adds joins and order to the pagination query parameters. Defaults to ordering
	 * in ascending order. The ordering will chose the lowest price between the price and sale_price
	 * to order by.
	 * @param array $currentParams The current parameters of the query to change. Normally the defaults set in listing()
	 * @param array $orderPrice     The ordering data (direction and currencyId if it is set)
	 * @param Model $queryModel    The model that is being queried on, listing uses Product
	 */
	protected function _addPriceParamsToListing($currentParams, $orderPrice, &$queryModel) {
		$orderDirection = 'ASC';
		if (isset($orderPrice['direction'])) {
			$orderDirection = $orderPrice['direction'];
		}

		$currentParams['order'] = "Product.price $orderDirection";

		return $currentParams;
	}

	/**
	 * Adds joins and conditions to the listing params so that products are selected only if they belong to a single
	 * brand. Brands are provided throught the order data as a list of the ids that the products need to match to.
	 * @param array $currentParams The current parameters of the query to change. Normally the defaults set in listing()
	 * @param array $orderBrands     The ordering data (list of brand ids)
	 * @param Model $queryModel    The model that is being queried on, listing uses Product
	 */
	protected function _addBrandParamsToListing($currentParams, $orderBrands, &$queryModel) {
		$currentParams['joins'][] = [
			'table' => 'ev_shop_brands',
			'alias' => 'FilteredBrand',
			'conditions' => [
				'FilteredBrand.id = Product.brand_id'
			]
		];

		$currentParams['conditions']['FilteredBrand.name'] = $orderBrands;

		return $currentParams;
	}

	/**
	 * Adds the _incTax fields to the query results. Checks that the EvTax plugin is loaded if not then
	 * it is assumed that the tax will be added manually from whatever peeps be using.
	 * @param array $products The results of the product query containing each product to add tax to.
	 */
	protected function _addTaxToProducts($products) {
		if (CakePlugin::loaded('EvTax')) {
			$TaxLevelModel = EvClassRegistry::init('EvTax.TaxLevel');
			$VariantPricingModel = EvClassRegistry::init('EvShop.VariantPricing');

			if (! empty($products)) {
				foreach ($products as $key => $item) {
					$taxLevel = array();
					if (isset($item['Product']['tax_level_id']) && $item['Product']['tax_level_id'] > 0) {

						$taxLevel = $TaxLevelModel->find('first', array(
							'conditions' => array(
								'id' => $item['Product']['tax_level_id']
							)
						));

						if (!empty($taxLevel)) {
							$taxRate = $taxLevel['TaxLevel']['rate'];
							if (isset($item['Product']['price']) && !empty($item['Product']['price'])) {
								$priceIncTax = $VariantPricingModel->addTaxToPrice($item['Product']['price'], $taxRate);
								$products[$key]['Product']['price_incTax'] = $priceIncTax;
							}

							if (isset($item['Product']['max_price']) && !empty($item['Product']['max_price'])) {
								$priceIncTax = $VariantPricingModel->addTaxToPrice($item['Product']['max_price'], $taxRate);
								$products[$key]['Product']['max_price_incTax'] = $priceIncTax;
							}
						}
					}
				}
			}
		}

		return $products;
	}

	/**
	 * Get a set of options to filter listing pages by. By default all the options will be retrieved, but if you want to limit
	 * selection by category or featured or something else then you can pass extra find parameters in.
	 * @param  array  $extraParams   Parameters that will be passed into the main find that gets the options. Variants, options
	 *                               and option groups are joined by default so any of their feilds can be accessed.
	 * @param  bool   $includeBrands Set to true if you want to include a list of brands to filter on. Retrieved brands are based
	 *                               on the products that are found, so if you are limiting filters to categories or featured
	 *                               then only brands that match those products will be returned.
	 * @param  array  $brandParams   Parameters that will be passed into the brand find. Used to limit brands further than the
	 *                               matched products or if other data is required.
	 * @return array                 An array containing each filter group and the options for each group and any further data
	 *                               added to options.
	 */
	public function getOptionFiltersForListing($extraParams = [], $includeBrands = false, $brandParams = []) {
		$Product = EvClassRegistry::init('EvShop.Product');

		$defaultParams = [
			'joins' => [
				[
					'table' => 'ev_shop_variants',
					'alias' => 'Variant',
					'conditions' => [
						'Variant.product_id = Product.id'
					]
				],
				[
					'table' => 'ev_shop_options_variants',
					'alias' => 'OptionVariant',
					'conditions' => [
						'OptionVariant.variant_id = Variant.id'
					]
				],
				[
					'table' => 'ev_shop_options',
					'alias' => 'Option',
					'conditions' => [
						'Option.id = OptionVariant.option_id'
					]
				],
				[
					'table' => 'ev_shop_option_groups',
					'alias' => 'OptionGroup',
					'conditions' => [
						'OptionGroup.id = Option.option_group_id'
					]
				]
			],
			'conditions' => [
				'Product.is_active' => true,
				'OptionGroup.on_front_end' => true
			],
			'fields' => [
				'Option.*',
				'OptionGroup.*',
				'Product.id'
			],
			'order' => 'Option.sequence'
		];

		$params = array_merge_recursive($defaultParams, $extraParams);

		$productOptions = $Product->find('all', $params);

		// Get unique option groups
		$optionGroups = Hash::combine($productOptions, '{n}.OptionGroup.id', '{n}.OptionGroup');

		// Get unique options
		$options = Hash::combine($productOptions, '{n}.Option.id', '{n}.Option');

		//Loop through each option and add it to it's corresponding option group.
		foreach ($options as $option) {
			$currentOptionGroupId = $option['option_group_id'];
			$currentOption = $option;
			$currentOption = $this->_addExtraDataToOptionsForListing($currentOption);

			$optionGroups[$currentOptionGroupId]['Option'][] = $currentOption;
		}

		// Assign extra information to Option Groups
		foreach ($optionGroups as $optionIndex => $optionGroup) {
			$optionGroups[$optionIndex] = $this->_addExtraDataToOptionGroupsForListing($optionGroups[$optionIndex]);
		}

		$optionGroups = $this->_addExtraOptionGroupsToOptionsForListing($optionGroups, $extraParams);

		if ($includeBrands) {
			$productIds = Hash::extract($productOptions, '{n}.Product.id');
			$optionGroups[] = $this->_addBrandsToOptionsForListing($productIds, $brandParams);
		}

		//Tidy up the array and sort
		$optionGroups = array_values($optionGroups);
		$optionGroups = Hash::sort($optionGroups, '{n}.name');

		return $optionGroups;
	}

	/**
	 * Add any additional data to each option found from the main find. This method is called from inside a loop
	 * so every option will obtain any data added here.
	 * @param array $option The returned data from the database of the current option.
	 * @return array        The option array with any added or manipulated data.
	 */
	protected function _addExtraDataToOptionsForListing($option) {
		return $option;
	}

	/**
	 * Add any additional data to each option group found from the main find. This method is called from inside a loop
	 * so every option group will obtain any data added here.
	 * @param array $optionGroup The returned data from the database of the current option group. Will also contain
	 *                           the options that belong to this option group.
	 * @return array             The option group array with any added or manipulated data.
	 */
	protected function _addExtraDataToOptionGroupsForListing($optionGroup) {
		return $optionGroup;
	}

	/**
	 * Add any additional option groups to the filters. Any options that may not have a database object could be added
	 * here or if you need options from a non ev shop table. This method could also be used to structure the data
	 * differently if you need to.
	 * @param array $optionGroups The current set of option groups and their options.
	 * @return 					  The modified set of option groups and their options.
	 */
	protected function _addExtraOptionGroupsToOptionsForListing($optionGroups, $extraParams) {
		return $optionGroups;
	}

	/**
	 * Get the brands that match the current products and create an option group for them so they can be added
	 * to the option group filters.
	 * @param array  $productIds    A list of products ids that will be used to find matching brands
	 * @param array  $params        Additional params passed through from the initial getOptionFiltersForListing
	 *                              call.
	 * @return                      A filter group containing the brands as options to be added to the option groups
	 */
	protected function _addBrandsToOptionsForListing($productIds, $params = []) {
		$Product = EvClassRegistry::init('EvShop.Product');

		$brands = [
			'name' => 'Brands'
		];

		$defaultParams = [
			'joins' => [
				[
					'table' => 'ev_shop_brands',
					'alias' => 'Brand',
					'conditions' => [
						'Brand.id = Product.brand_id'
					]
				]
			],
			'conditions' => [
				'Product.id' => $productIds
			],
			'fields' => [
				'Brand.*'
			],
			'group' => 'Brand.id'
		];

		$params = array_merge_recursive($defaultParams, $params);

		$optionBrands = $Product->find('all', $params);
		$optionBrands = Hash::extract($optionBrands, '{n}.Brand');

		foreach ($optionBrands as $optionBrand) {
			$this->_addExtraDataToBrandForListing($optionBrand);

			$brands['Option'][] = $optionBrand;
		}

		$this->_addExtraDataToBrandGroupForListing($brands);

		return $brands;
	}

	/**
	 * Add any extra data to each brand option. This method is called in a loop so any data added here will be added
	 * to every brand option.
	 * @param array $currentBrand The current brand array containing data from the database.
	 * @return array              The current brand with any additional of modified data.
	 */
	protected function _addExtraDataToBrandForListing($currentBrand) {
		return $currentBrand;
	}

	/**
	 * Add any extra data to the brand option group.
	 * @param array $currentBrandGroup The current brand filter group. Also contains the brand options.
	 * @return array                   The current brand filter group with any additional or modified data.
	 */
	protected function _addExtraDataToBrandGroupForListing($currentBrandGroup) {
		return $currentBrandGroup;
	}
}
