<?php
App::uses('EvShopAppController', 'EvShop.Controller');
App::uses('ArrayUtil', 'EvCore.Lib');

class CategoriesController extends EvShopAppController {

	/**
	 * redefine to allow the index / view page through the auth component!
	 * and also add the reorder product method
	 */
	public function beforeFilter() {
		$this->adminActions[] = 'admin_reorderProducts';

		parent::beforeFilter();

		$this->Auth->allow(array('index', 'view'));
	}

	/**
	 * process the param array for (sub) categories
	 *
	 * @param 	array 	$Category The parent we are finding children for (if null, it's index method)
	 * @return 	array 		   	  Standard parameter array to pass to find
	 */
	protected function _processCategoryParams($Category = null) {
		return array();
	}

	/**
	 * process the param array for products
	 *
	 * @param 	array 	$Category The parent category
	 * @return 	array 		   	  Standard parameter array to pass to find
	 */
	protected function _processProductParams($Category) {
		return array();
	}

	/**
	 * categories index page
	 * list all categories out
	 *
	 */
	public function index() {
		$this->view = 'EvShop./Fallbacks/Categories/index';
		$pageId = Configure::read('EvShop.pageIds.category');

		// check if we have a specific content page set
		// if so - set the template
		if (! empty($pageId)) {
			$pageData = $this->assignPage($pageId);

			if (! empty($pageData)) {
				$this->view = $this->Tpl->getTemplate($pageData, 'Page');
			}
		}

		$this->set(
			'listing',
			$this->Categories->listing(
				$this->_processCategoryParams()
			)
		);

		$this->Breadcrumb->addCrumb(__('All Categories'));
	}

	/**
	 * brand view details page
	 *
	 * @param 	int 	Brand ID
	 */
	public function view($id) {
		$Model = $this->{$this->modelClass};
		$this->view = 'EvShop./Fallbacks/Categories/view';

		$data = $Model->readForView($id);
		if (empty($data)) {
			throw new NotFoundException();
		}

		$this->set('data', $data);
		$this->Meta->set($data);

		if (! empty($data[$Model->alias]['template_id'])) {
			$this->view = $this->Tpl->getTemplate($data);
		}

		// get the sub categories for product retrieval
		$subCategories = $Model->children($id);
		$this->categoryIds = Hash::extract(
			$subCategories,
			'{n}.' . $Model->alias . '.id'
		);

		$this->categoryIds[] = $id;

		// get any sub categories and products
		$this->set(
			array(
				'categoryListing' => $this->Categories->listingByParent(
					$id,
					$this->_processCategoryParams($data),
					false
				),
				'productListing' => $this->Products->listingByCategory(
					$this->categoryIds,
					$this->_processProductParams($data)
				)
			)
		);

		$this->_getCategoryParentsBreadcrumb($data);
	}

	protected function _adminFilterFields() {
		$fields = parent::_adminFilterFields();

		unset(
			$fields['Category.id'],
			$fields['Category.created'],
			$fields['Category.modified']
		);

		return $fields;
	}

	protected function _adminIndexColumns() {
		$fields = parent::_adminIndexColumns();

		$fields = ArrayUtil::addAfter(
			$fields,
			'Category.name',
			array(
				'ParentCategory.name' => array(
					'type' => 'string',
					'label' => 'Parent',
					'default' => '-'
				)
			)
		);

		return $fields;
	}

	protected function _adminIndexPaginate() {
		$paginate = parent::_adminIndexPaginate();

		$paginate['contain'][] = 'ParentCategory';
		$paginate['order'] = 'Category.lft ASC';

		return $paginate;
	}

	/**
	 * add the product reorder link
	 */
	protected function _adminIndexActions() {
		$actions = array(
			'Reorder' => array(
				'cell' => array(
					'class' => 'action pricing'
				),
				'link' => array(
					'url' => array(
						'action' => 'reorderProducts'
					),
					'text' => '<i class="fa fa-bars"></i>',
					'options' => array(
						'escape' => false
					)
				)
			)
		) + parent::_adminIndexActions();

		return $actions;
	}

	protected function _adminPopulateLookups() {
		parent::_adminPopulateLookups();

		$this->set(
			'parents',
			( // merge arrays with the + operator .... preserves numeric keys in arrays
				array('Top Level') +
				$this->{$this->modelClass}->generateTreeList(
					null,
					null,
					null,
					'--'
				)
			)
		);
	}

	/**
	 * form fields - check for EvTax and auto remove
	 * check for brands setting and auto remove
	 */
	protected function _adminFormFields() {
		$fields = parent::_adminFormFields();

		$fields['Category.body']['type'] = 'html';

		return $fields;
	}

	/**
	 * reorder products within the category
	 */
	public function admin_reorderProducts($id) {
		$Model = $this->{$this->modelClass};

		if ($this->request->is('post') || $this->request->is('put')) {
			// format to allow us to save

			$save = array();
			foreach ($this->request->data['reorder'] as $item) {

				$save[] = array(
					'CategoriesProduct' => array(
						'id' => $item['id'],
						'sequence' => $item['sequence']
					)
				);
			}

			if ($Model->CategoriesProduct->saveMany($save)) {
				$this->Flash->success(
					array(
						'title' => 'Order Saved',
						'description' => 'Reordering has been saved successfully!'
					)
				);

				$this->redirect(
					array(
						'action' => 'index'
					)
				);
			} else {
				$this->Flash->fail(
					array(
						'title' => 'Reordering Failed',
						'description' => 'Unable to find an ordering column on the table'
					)
				);
			}
		}

		$data = $Model->CategoriesProduct->find('all', array(
			'conditions' => array(
				'CategoriesProduct.category_id' => $id
			),
			'order' => 'CategoriesProduct.sequence ASC',
			'contain' => array(
				'Product'
			)
		));

		$this->set(array(
			'order_key' => 'sequence',
			'max_depth' => 1,
			'data' => $data,
			'columns' => array(
				'CategoriesProduct.id' => true,
				'Product.name' => true
			)
		));

		$this->view = '/Scaffolds/admin_reorder';
	}
}
