<?php
App::uses('EvShippingAppModel', 'EvShipping.Model');
/**
 * Zone Model
 *
 * @property Country $Country
 * @property ZoneExtra $ZoneExtra
 * @property ShippingRule $ShippingRule
 */
class Zone extends EvShippingAppModel {

	/**
	 * Display field
	 *
	 * @var string
	 */
	public $displayField = 'name';

	/**
	 * Validation rules
	 *
	 * @var array
	 */
	public $validate = array(
		'name' => array(
			'notBlank' => array(
				'rule' => array('notBlank'),
				'message' => 'Please enter a name for this Zone'
			),
		),
	);

	//The Associations below have been created with all possible keys, those that are not needed can be removed

	/**
	 * hasMany associations
	 *
	 * @var array
	 */
	public $hasMany = array(
		'ZoneExtra' => array(
			'className' => 'EvShipping.ZoneExtra',
			'dependent' => true,
			'cascade' => true
		)
	);

	/**
	 * hasAndBelongsToMany associations
	 *
	 * @var array
	 */
	public $hasAndBelongsToMany = array(
		'ShippingRule' => array(
			'className' => 'EvShipping.ShippingRule',
			'joinTable' => 'ev_shipping_shipping_rules_zones'
		),
		'Country' => array(
			'className' => 'EvCountry.Country',
			'joinTable' => 'ev_shipping_zones_countries',
			'foreignKey' => 'zone_id',
			'associationForeignKey' => 'country_id',
		),
	);

	/**
	 * readForEdit - retrieve the zone extra data
	 *
	 * @param integer $id ID of row to edit
	 * @param array $query The db query array - can be used to pass in additional parameters such as contain
	 * @return array
	 */
	public function readForEdit($id, $query = array()) {
		$query['contain'][] = 'ZoneExtra';
		$query['contain'][] = 'Country';

		return parent::readForEdit($id, $query);
	}

	/**
	 * Called after each successful save operation.
	 *
	 * @param bool $created True if this save created a new record
	 * @param array $options Options passed from Model::save().
	 * @return void
	 * @link http://book.cakephp.org/2.0/en/models/callback-methods.html#aftersave
	 * @see Model::save()
	 */
	public function afterSave($created, $options = array()) {
		parent::afterSave($created, $options);

		// process any custom field deletions
		if (! empty($this->data[$this->alias]['ZoneExtraDeletes']['toDelete'])) {

			$ids = explode(',', $this->data[$this->alias]['ZoneExtraDeletes']['toDelete']);
			$ids = ArrayUtil::clearArray($ids);

			$this->ZoneExtra->deleteAll(
				array(
					'ZoneExtra.id' => $ids
				),
				true,
				true
			);
		}
	}

	/**
	 * get the zones and the zone extra data given the country Id
	 *
	 * @param 	int 	$countryId 	Country Id
	 * @return  array 	$zones
	 */
	public function getByCountry($countryId) {
		$zones = $this->find('all', array(
			'conditions' => array(
				'Zone.is_active' => 1,
				'CountryTable.country_id' => $countryId
			),
			'contain' => array(
				'ZoneExtra',
			),
			'joins' => array(
				array(
					'table' => 'ev_shipping_zones_countries',
					'alias' => 'CountryTable',
					'conditions' => array(
						'Zone.id = CountryTable.zone_id'
					),
				),
			),
		));

		return $zones;
	}

	/**
	 * given the list of applicable zones and the address
	 * try to filter on an extras
	 *
	 * @param 	array 		$zones  		Zones as returned from getByCountry()
	 * @param 	array 		$address 		The address to search
	 * @return 	array|bool 	$filteredZones	The filtered array of applicable zones
	 */
	public function filterOnExtras($zones, $address) {
		$primary = array(); // for zones that have extras and get matched
		$secondary = array(); // for zones withj no extras (i.e. a bkup)

		foreach ($zones as $zone) {
			if (empty($zone['ZoneExtra'])) {
				$secondary[] = $zone;
				continue;
			}

			foreach ($zone['ZoneExtra'] as $zoneExtra) {
				if ($this->zoneExtraMatch($zoneExtra, $address)) {
					$primary[] = $zone;
				}
			}
		}

		if (! empty($primary)) {
			return $primary;
		}

		if (empty($primary) && ! empty($secondary)) {
			return $secondary;
		}

		return false;
	}

/**
 * Given the zone extra row, check if it matches given the address.
 *
 * @param array $zoneExtra Zone Extra row from the db.
 * @param array $address   The address to try and match.
 * @return bool True if matched, false otherwise.
 */
	public function zoneExtraMatch($zoneExtra, $address) {
		// first check the field exists
		if (! isset($address[$zoneExtra['field']])) {
			return false;
		}

		$searchField = $address[$zoneExtra['field']];
		$operator = $zoneExtra['operator'];
		$value = $zoneExtra['field_value'];

		switch ($operator) {
			case 'LIKE':
				return $this->_zoneExtraMatchLike($searchField, $value);
				break;
			case 'REGEX':
				return $this->_zoneExtraMatchRegex($searchField, $value);
				break;
			default:
				return strtolower($searchField) == strtolower($value);
				break;
		}
	}

/**
 * Check if a zone extra matches the address using the like operator.
 *
 * @param string $field The field to match against.
 * @param string $value The value of the zone extra.
 * @return bool True if field matches, false otherwise.
 */
	protected function _zoneExtraMatchLike($field, $value) {
		// find where / if it has any wildcards
		if (strpos($value, '%') === 0) {
			// wildcard at the beginng - check endsWith
			$checkEndsWith = true;
		}

		if (strpos($value, '%', 1) === (strlen($value) - 1)) {
			// wildcard at the end - check startsWith
			$checkStartsWith = true;
		}

		$SearchString = \Stringy\Stringy::create($field, 'UTF-8');

		$searchValue = str_replace('%', '', $value);
		if (isset($checkEndsWith) && isset($checkStartsWith)) {
			// has both - search contains
			return $SearchString->contains($searchValue, false);
		} elseif (isset($checkEndsWith)) {
			// wildcard at the beginning
			return $SearchString->endsWith($searchValue, false);
		} elseif (isset($checkStartsWith)) {
			// wildcard at the end
			return $SearchString->startsWith($searchValue, false);
		}

		return false;
	}

/**
 * Check if a zone extra matches the address using the regex operator.
 *
 * @param string $field The field to match against.
 * @param string $value The value of the zone extra.
 * @return bool True if field matches, false otherwise.
 */
	protected function _zoneExtraMatchRegex($field, $value) {
		$searchValue = str_replace(' ', '', $value);
		$result = preg_match($searchValue, $field);

		return is_numeric($result) && $result > 0;
	}
}
