<?php

App::uses('EvShippingAppModel', 'EvShipping.Model');

class DeliverySlot extends EvShippingAppModel {

/**
 * Display field
 *
 * @var string
 */
	public $displayField = 'weekday';

/**
 * hasMany associations
 *
 * @var array
 */
	public $hasMany = array(
		'OverrideDeliverySlot' => array(
			'className' => 'EvShipping.OverrideDeliverySlot'
		)
	);

/**
 * Read for function for the admin edit slots page which allows all delivery slots to be editted
 * at once.
 * @return [type] [description]
 */
	public function readForEditSlots() {
		return $this->find('all');
	}

/**
 * Return all the delivery slots formatted so they are ordered by delivery slot.
 * @return array The array of formatted delivery slots
 */
	public function getDeliverySlots() {
		//Get all the delivery slots
		$deliverySlots = $this->find(
			'all',
			[
				'conditions' => [
					'DeliverySlot.no_delivery' => false
				]
			]
		);

		//Format them by weekday
		$results = [];
		foreach ($deliverySlots as $deliverySlot) {
			$results[$deliverySlot['DeliverySlot']['weekday']] = [
				'max_cutoff' => $deliverySlot['DeliverySlot']['max_cutoff']
			];
		}

		//Add any overrides

		return $results;
	}

/**
 * The validation required for validating a delivery slot date
 * @return array Validation rules
 */
	public function deliverySlotDateValidation() {
		return [
			'date' => array(
				'notBlank' => [
					'rule' => ['notBlank'],
					'message' => 'Please provide an order date to receive your order',
				],
				'futureDate' => [
					'rule' => ['deliverySlotDateInFuture'],
					'message' => 'Your selected date must be in the future',
				],
				'sameDay' => [
					'rule' => ['deliverySlotDateSameDay'],
					'message' => 'Same day orders must be made before 3pm',
				],
				'notSunday' => [
					'rule' => ['deliverySlotDateNoDelivery'],
					'message' => 'Orders can\'t be received on a Sunday',
				]
			),
		];
	}

/**
 * A validation method that checks if the date provided is in the future
 * @return boolean        Whether the validation has passed or not
 */
	public function deliverySlotDateInFuture($check) {
		$data = $this->data['DeliverySlot'];
		$providedDate = DateTime::createFromFormat('d/m/Y', $data['date']);

		$today = new DateTime();
		$todayDate = new DateTime($today->format('Y-m-d H:i:s'));
		$todayDate->setTime(0, 0, 0);

		if ($providedDate < $todayDate) {
			return false;
		}

		return true;
	}

/**
 * A validation method that checks if the date provided is the same day and if it is checks to see if there is a cutoff time and the
 * current order is being made before the cutoff
 * @return boolean        Whether the validation has passed or not
 */
	public function deliverySlotDateSameDay($check) {
		$field = key($check);

		$data = $this->data['DeliverySlot'];
		$providedDate = DateTime::createFromFormat('d/m/Y', $data['date'])->setTime(0, 0, 0);

		$today = new DateTime();
		$todayDate = new DateTime($today->format('Y-m-d H:i:s'));
		$todayDate->setTime(0, 0, 0);

		if ($providedDate == $todayDate) {
			$deliverySlots = $this->getDeliverySlots();

			//Check there are not any overridden cutoffs
			$overrideCutoff = $this->_checkOverrideCutoffTime($todayDate, $today);
			if (isset($overrideCutoff['cutoff']) && $overrideCutoff['cutoff'] === false) {
				$this->invalidate($field, __d('ev_shipping', 'Same day orders must be made before ' . $overrideCutoff['cutoffDate']->format('g:iA')));
			} elseif (empty($overrideCutoff)) {
				if (array_key_exists($todayDate->format('l'), $deliverySlots)) {
					$todayDeliverySlot = $deliverySlots[$todayDate->format('l')];

					if (!empty($todayDeliverySlot['max_cutoff'])) {
						list($hour, $minute, $second) = explode(':', $todayDeliverySlot['max_cutoff']);

						//Check if it is later than cutoff
						$todayCutoff = new DateTime($today->format('Y-m-d H:i:s'));
						$todayCutoff->setTime($hour, $minute, $second);

						if ($today > $todayCutoff) {
							$this->invalidate($field, __d('ev_shipping', 'Same day orders must be made before ' . $todayCutoff->format('g:iA')));
						}
					}
				}
			}
		}

		return true;
	}

/**
 * A validation method that checks if the date provided isn't on a day that has no delivery
 * @return boolean        Whether the validation has passed or not
 */
	public function deliverySlotDateNoDelivery($check) {
		$field = key($check);

		$data = $this->data['DeliverySlot'];
		$providedDate = DateTime::createFromFormat('d/m/Y', $data['date'])->setTime(0, 0, 0);

		$deliverySlots = $this->getDeliverySlots();

		$overrideNoDelivery = $this->_checkOverrideNoDelivery($providedDate);
		if ($overrideNoDelivery === null) {
			if (!array_key_exists($providedDate->format('l'), $deliverySlots)) {
				$this->invalidate($field, __d('ev_shipping', 'Orders can\'t be ordered on a ' . $providedDate->format('l')));
			}
		} elseif ($overrideNoDelivery === true) {
			$this->invalidate($field, __d('ev_shipping', 'Orders can\'t be ordered on ' . $providedDate->format('l jS F')));
		}

		return true;
	}

/**
 * Check if there is an overridden cut off time and if there is have we reached it. If the overidden cutoff time has
 * been reached then the cutoff time is returned as a date so it can be used to invalidate the same day validation.
 * @param  DateTime $todayDate    Todays date at midnight
 * @param  DateTime $today        Todays date at the current time
 * @return array                  An empty array is returned if no override is found. If an override is found then
 *                                   an array is returned that conains if the cutoff has been reached and the date
 *                                   of the cutoff.
 */
	protected function _checkOverrideCutoffTime($todayDate, $today) {
		$overrideCutoff = $this->_getOverride($todayDate->format('l'), $todayDate->format('Y-m-d H:i:s'));
		if ($overrideCutoff !== null) {
			if (!empty($overrideCutoff['max_cutoff'])) {
				list($hour, $minute, $second) = explode(':', $overrideCutoff['max_cutoff']);

				$overrideCutoffTime = new DateTime($today->format('Y-m-d H:i:s'));
				$overrideCutoffTime->setTime($hour, $minute, $second);

				if ($today < $overrideCutoffTime) {
					return ['cutoff' => true, 'cutoffDate' => $overrideCutoffTime];
				} else {
					return ['cutoff' => false, 'cutoffDate' => $overrideCutoffTime];
				}
			}
		}

		return [];
	}

/**
 * Check that the provided date doesn't have an overridden delivery slot.
 * @param  DateTime $providedDate The provided date to check as a DateTime object with a time of midnight
 * @return bool                   True if override was found and no delivery has been selected on that date.
 *                                     False if override was found and no delivery wasn't selected.
 *                                     Null if no override was found.
 */
	protected function _checkOverrideNoDelivery($providedDate) {
		$overrideNoDelivery = $this->_getOverride($providedDate->format('l'), $providedDate->format('Y-m-d H:i:s'));

		if ($overrideNoDelivery !== null && $overrideNoDelivery['no_delivery']) {
			return true;
		} elseif ($overrideNoDelivery !== null) {
			return false;
		} else {
			return null;
		}
	}

/**
 * Attempt to get an override for a specific delivery slot on a specific date. If no override is found
 * then return null otherwise return the override data (max_cutoff and no_delivery).
 * @param  String $weekDay      The name of the delivery slot to find an override for
 * @param  String $overrideDate The dateTime string of the date to search for.
 * @return Array                Null if no override is found. Otherwise return the override data.
 */
	protected function _getOverride($weekDay, $overrideDate) {
		$overrides = $this->OverrideDeliverySlot->getOverrideDeliverySlots();

		if (!empty($overrides[$weekDay][$overrideDate])) {
			return $overrides[$weekDay][$overrideDate];
		}

		return null;
	}

/**
 * Gets an estimated delivery date if an order is placed today based on a lead time in days
 * Takes into account all non-delivery dates
 */
	public function estimateDeliveryDate($leadTime, $format = 'd/m/Y', $limitToSlotIds = null) {
		$daysToDeliver = $leadTime;
		$currentDate = date('Y-m-d');
		$currentTime = date('H:i:s');

		// Key by weekday so it's easier to search
		$deliverySlots = $this->find('all');
		$deliverySlots = Hash::combine( $deliverySlots, '{n}.DeliverySlot.weekday', '{n}.DeliverySlot');

		// Check if we have any overrides for this day
		$overrideToday = $this->OverrideDeliverySlot->find('first', [
			'conditions' => [
				'date' => $currentDate,
				'is_active' => true
			]
		]);

		// Add an extra day lead time if we've missed todays delivery cutoff
		if (!empty($override)) {
			if ($currentTime >= $override['OverrideDeliverySlot']['max_cutoff']) {
				$daysToDeliver++;
			}
		} else {
			if ($currentTime >= $deliverySlots[date('l', strtotime($currentDate))]['max_cutoff']) {
				$daysToDeliver++;
			}
		}

		for ($i = $daysToDeliver - 1; $i < $daysToDeliver; $i++) {

			$checkDate = date('Y-m-d', strtotime($currentDate . ' + ' . ($i + 1) . ' days'));

			if (empty($limitToSlotIds) || in_array($deliverySlots[date('l', strtotime($checkDate))]['id'], $limitToSlotIds)) {

				// Check if we have any overrides for this day
				$override = $this->OverrideDeliverySlot->find('first', [
					'conditions' => [
						'date' => $checkDate,
						'is_active' => true
					]
				]);

				// If we can't deliver on this day increase the days to deliver
				if (!empty($override)) {
					if ($override['OverrideDeliverySlot']['no_delivery']) {
						$daysToDeliver++;
					}
				} else {
					if ($deliverySlots[date('l', strtotime($checkDate))]['no_delivery']) {
						$daysToDeliver++;
					}
				}
			} else {
				$daysToDeliver++;
			}
		}

		return date($format, strtotime($currentDate . ' + ' . $daysToDeliver . ' days'));
	}
}
