<?php

App::uses('AppComponent', 'Controller/Component');

class ShippingManagerComponent extends AppComponent {

	/**
	 * surcharges - any surcharges to add to the rate
	 */
	public $surcharges = array();

	/**
	 * data needed for process the rules
	 * basket / product data etc...
	 */
	public $data = array();

	/**
	 * calculate shipping given an address
	 *
	 *
	 * @param 	array 		$address 	Address array (EvAddressBook)
	 * @return 	float|bool 	$shipping 	Cost of shipping
	 */
	public function calculateShippingFromAddress($address) {
		// find the country zones the address sits in
		$zones = $this->getZoneIds(
			EvClassRegistry::init('EvShipping.Zone'),
			$address
		);

		if (empty($zones)) {
			return false;
		}

		// find the shipping rules
		$ShippingRule = EvClassRegistry::init('EvShipping.ShippingRule');
		$ShippingRules = $this->getShippingRules(
			$ShippingRule,
			$zones
		);

		if (empty($ShippingRules)) {
			return false;
		}

		$rate = $this->getRate(
			$ShippingRules
		);

		if ($rate === false) {
			return false;
		}

		$rate = $this->processSurcharges(
			$rate,
			$this->surcharges
		);

		return $rate;
	}

	/**
	 * given the address, find which zones we fit in
	 *
	 * @param 	object 		$Zone 		Zone model object
	 * @param 	array 		$address 	Address array (EvAddressBook)
	 * @return 	array|bool 	$zoneIds	Array of Zone IDs
	 */
	public function getZoneIds($Zone, $address) {
		if (empty($address['country_id'])) {
			return false;
		}

		$zones = $Zone->getByCountry($address['country_id']);

		$filteredZones = $Zone->filterOnExtras($zones, $address);

		if (empty($filteredZones)) {
			return array();
		}

		return Hash::extract(
			$filteredZones,
			'{n}.Zone.id'
		);
	}

	/**
	 * get all the shipping rules given the array of zone ids
	 *
	 * @param 	object 		$ShippingRule 	ShippingRule model object
	 * @param 	array 		$zoneIds 		Zone ids to search on
	 * @return  array|bool 	$ShippingRules  Array of Shipping Rules
	 */
	public function getShippingRules($ShippingRule, $zoneIds) {
		if (empty($zoneIds)) {
			return false;
		}

		$ShippingRules = $ShippingRule->getByZones($zoneIds);

		if (empty($ShippingRules)) {
			return array();
		}

		return $ShippingRules;
	}

	/**
	 * given an array of ShippingRules, get a shipping rate
	 *
	 * @param 	array 		$ShippingRules 	Array of shipping rules
	 * @return 	float|bool 	$rate 			bool false on fail or a shipping rate
	 */
	public function getRate($ShippingRules) {
		if (empty($ShippingRules)) {
			return false;
		}

		$rate = false;

		foreach ($ShippingRules as $Rule) {
			$shippingRate = $this->processShippingRule($Rule);

			if ($shippingRate !== false && ! is_null($shippingRate)) {
				$rate = $shippingRate;
				break;
			}
		}

		return $rate;
	}

	/**
	 * given an individual rule - load it's handle and try to process
	 *
	 * @param 	array 		$Rule 	Shipping Rule data
	 * @return 	float|bool 	$rate
	 */
	public function processShippingRule($Rule) {
		if (empty($Rule['ShippingRule']['handler'])) {
			return false;
		}

		$ShippingHandler = $this->loadHandler($Rule['ShippingRule']['handler']);

		if (is_object($ShippingHandler)) {
			return $ShippingHandler->processRule($Rule);
		}

		return false;
	}

	/**
	 * load a handler
	 *
	 * @param 	string 		$handler 			The shipping handler to use
	 * @return 	object 		$ShippingHandler 	The loaded handler
	 */
	public function loadHandler($handler) {
		list($plugin, $handler) = pluginSplit($handler, true, '');
		App::uses($handler, $plugin . 'ShippingHandlers');

		if (class_exists($handler)) {
			return new $handler($this);
		}

		return false;
	}

	/**
	 * add a surcharge
	 *
	 * @param 	string 	$name 		surcharge name
	 * @param 	float 	$amount 	surcharge amount
	 * @param 	bool 	$override 	Override a surcharge amount
	 * @return 	bool
	 */
	public function addSurcharge($name, $amount, $override = true) {
		if ($override === false && isset($this->surcharges[$name])) {
			return false;
		}

		if ($amount > 0) {
			$this->surcharges[$name] = $amount;
			return true;
		}

		return false;
	}

	/**
	 * get a surcharge
	 *
	 * @param 	string 	$name 		surcharge name
	 * @return 	float|bool
	 */
	public function getSurcharge($name) {
		if (isset($this->surcharges[$name])) {
			return $this->surcharges[$name];
		}

		return false;
	}

	/**
	 * remove a surcharge
	 *
	 * @param 	string 	$name 		surcharge name
	 * @return 	float|bool
	 */
	public function removeSurcharge($name) {
		if (isset($this->surcharges[$name])) {
			unset($this->surcharges[$name]);

			return true;
		}

		return false;
	}

	/**
	 * process all the surcharges
	 *
	 * @param 	float 	$rate 		The current selected rate
	 * @param 	array 	$surcharges Array of surcharges to apply
	 * @return 	float 	$newRate
	 */
	public function processSurcharges($rate, $surcharges) {
		foreach ($surcharges as $name => $amount) {
			$rate += $amount;
		}

		return $rate;
	}

	/**
	 * add data
	 *
	 * @param 	string 	$key 		data key
	 * @param 	mixed 	$data 		data
	 * @param 	bool 	$override 	Override a surcharge amount
	 * @return 	bool
	 */
	public function addData($key, $data, $override = true) {
		if ($override === false && isset($this->data[$key])) {
			return false;
		}

		$this->data[$key] = $data;
		return true;
	}

	/**
	 * get a surcharge
	 *
	 * @param 	string 	$key 		data key
	 * @return 	mixed|bool
	 */
	public function getData($key) {
		if (isset($this->data[$key])) {
			return $this->data[$key];
		}

		return false;
	}

/**
 * Get all the delivery slots using the model method.
 * @return array The delivery slots
 */
	public function getDeliverySlots() {
		$DeliverySlot = EvClassRegistry::init('EvShipping.DeliverySlot');
		return $DeliverySlot->getDeliverySlots();
	}

/**
 * Get all the overridden delivery slots using the model method.
 * @return array The overridden delivery slots
 */
	public function getOverrideDeliverySlots() {
		$OverrideDeliverySlot = EvClassRegistry::init('EvShipping.OverrideDeliverySlot');
		return $OverrideDeliverySlot->getOverrideDeliverySlots();
	}

}
