<?php
App::uses('AppController', 'Controller');
App::uses('CakeRequest', 'Network');
App::uses('CakeResponse', 'Network');
App::uses('ComponentCollection', 'Controller');
App::uses('ShippingManagerComponent', 'EvShipping.Controller/Component');
App::uses('AbstractHandler', 'EvShipping.Lib');

// A fake controller to test against
class ShippingManagersControllerTest extends AppController {

}

class WorkingHandler extends AbstractHandler {

	public function getRuleData() {
		return array();
	}

	public function processRule($ShippingRule) {
		return 10.00;
	}

}

class NonWorkingHandler extends AbstractHandler {

	public function getRuleData() {
		return array();
	}

	public function processRule($ShippingRule) {
		return false;
	}

}

class ShippingManagerComponentTest extends CakeTestCase {

	public $ShippingManager = null;

	public $Controller = null;

	/**
	 * Fixtures
	 *
	 * @var array
	 */
	public $fixtures = array(
		'plugin.ev_shipping.zone',
		'plugin.ev_country.country',
		'plugin.ev_shipping.zone_extra',
		'plugin.ev_shipping.shipping_rule',
		'plugin.ev_shipping.shipping_rule_data',
		'plugin.ev_shipping.shipping_rules_zone'
	);

	/**
	 * setUp method
	 *
	 * @return void
	 */
	public function setUp() {
		parent::setUp();

		// Setup our component and fake test controller
		$Collection = new ComponentCollection();

		// mock the component to override the generateHash method so we can control it for testing
		$this->ShippingManager = new ShippingManagerComponent($Collection);
		$CakeRequest = new CakeRequest();
		$CakeResponse = new CakeResponse();
		$workingHandler = new WorkingHandler($this->ShippingManager);
		$nonWorkingHandler = new NonWorkingHandler($this->ShippingManager);

		$this->Controller = new ShippingManagersControllerTest($CakeRequest, $CakeResponse);
		$this->Controller->constructClasses();

		$this->ShippingManager->initialize($this->Controller);

		// mock the component to override the loadHandler method so we can control it for testing
		$this->ShippingManagerWorkingRule = $this->getMockBuilder('ShippingManagerComponent')
			->setConstructorArgs(
				array(
					$Collection
				)
			)
			->setMethods(array('loadHandler'))
			->getMock();

		$this->ShippingManagerWorkingRule->expects($this->any())
			->method('loadHandler')
			->will($this->returnValue($workingHandler));

		// mock the component to override the loadHandler method so we can control it for testing
		$this->ShippingManagerNonWorkingRule = $this->getMockBuilder('ShippingManagerComponent')
			->setConstructorArgs(
				array(
					$Collection
				)
			)
			->setMethods(array('loadHandler'))
			->getMock();

		$this->ShippingManagerWorkingRule->expects($this->any())
			->method('loadHandler')
			->will($this->returnValue($nonWorkingHandler));
	}

	/**
	 * tearDown method
	 *
	 * @return void
	 */
	public function tearDown() {
		parent::tearDown();
		// Clean up after we're done
		unset($this->ShippingManager);
		unset($this->Controller);
	}

	public function testGetZoneIdsReturnsTheCorrectZoneIds() {
		$this->assertEquals(
			array(3),
			$this->ShippingManager->getZoneIds(
				ClassRegistry::init('EvShipping.Zone'),
				array(
					'address1' => 'Evoluted New Media',
					'address2' => '35 Lambert Street',
					'city' => 'Sheffield',
					'post_code' => 'S3 7BH',
					'country_id' => 1
				)
			)
		);

		$this->assertFalse(
			$this->ShippingManager->getZoneIds(
				ClassRegistry::init('EvShipping.Zone'),
				array(
					'address1' => 'Evoluted New Media',
					'address2' => '35 Lambert Street',
					'city' => 'Sheffield',
					'post_code' => 'S3 7BH'
				)
			)
		);

		$this->assertEquals(
			array(),
			$this->ShippingManager->getZoneIds(
				ClassRegistry::init('EvShipping.Zone'),
				array(
					'address1' => 'Evoluted New Media',
					'address2' => '35 Lambert Street',
					'city' => 'Sheffield',
					'post_code' => 'S3 7BH',
					'country_id' => 9999
				)
			)
		);
	}

	public function testGetShippingRulesReturnsTheCorrectRules() {
		$this->assertEquals(
			array(
				array(
					'ShippingRule' => array(
						'id' => 2,
						'name' => 'Weight Based UK None Mainland',
						'handler' => 'NonWorkingHandler',
						'sequence' => 1,
						'is_selectable' => 0,
						'is_active' => 1,
						'created' => '2015-09-03 09:45:31',
						'modified' => '2015-09-03 09:45:31'
					),
					'ShippingRuleData' => array(
						array(
							'id' => 2,
							'shipping_rule_id' => 2,
							'name' => 'weight_bands',
							'rule_data' => '1-10;15.00:11-20;20.00:21-99999999;30',
							'created' => '2015-09-03 09:45:58',
							'modified' => '2015-09-03 09:45:58'
						)
					)
				)
			),
			$this->ShippingManager->getShippingRules(
				ClassRegistry::init('EvShipping.ShippingRule'),
				array(3)
			)
		);

		$this->assertFalse(
			$this->ShippingManager->getShippingRules(
				ClassRegistry::init('EvShipping.ShippingRule'),
				array()
			)
		);

		$this->assertEquals(
			array(),
			$this->ShippingManager->getShippingRules(
				ClassRegistry::init('EvShipping.ShippingRule'),
				array(9999)
			)
		);
	}

	public function testAddSurchargeAddsCorrectly() {
		$this->assertTrue(
			$this->ShippingManager->addSurcharge('large-item', 25.00)
		);

		$this->assertEquals(
			array(
				'large-item' => 25.00
			),
			$this->ShippingManager->surcharges
		);
	}

	public function testAddSurchargeFailsWhenOverrideIsOffAndItemExists() {
		$this->ShippingManager->addSurcharge('large-item', 25.00);

		$this->assertFalse(
			$this->ShippingManager->addSurcharge('large-item', 20.00, false)
		);

		$this->assertFalse(
			$this->ShippingManager->addSurcharge('free-surcharge', 0)
		);
	}

	public function testGetSurchargeReturnsSurchargeCorrectlyOrFalseIfDoesntExist() {
		$this->ShippingManager->addSurcharge('large-item', 25.00);

		$this->assertEquals(
			25.00,
			$this->ShippingManager->getSurcharge('large-item')
		);

		$this->assertFalse(
			$this->ShippingManager->getSurcharge('none-existant-item')
		);
	}

	public function testRemoveSurchargeRemovesItemCorrectlyAndErrorsWhenItemDoesntExist() {
		$this->ShippingManager->addSurcharge('large-item', 25.00);

		$this->assertTrue(
			$this->ShippingManager->removeSurcharge('large-item')
		);

		$this->assertFalse(
			$this->ShippingManager->removeSurcharge('large-item')
		);
	}

	public function testProcessSurchargeAddsUpCorrectly() {
		$this->ShippingManager->addSurcharge('large-item', 25.00);
		$this->ShippingManager->addSurcharge('other', 5.50);

		$this->assertEquals(
			35.50,
			$this->ShippingManager->processSurcharges(
				5.00,
				$this->ShippingManager->surcharges
			)
		);
	}

	public function testAddDataAddsCorrectly() {
		$this->assertTrue(
			$this->ShippingManager->addData('basket', array('BasketItem' => array(), 'BasketTotals' => array()))
		);

		$this->assertEquals(
			array(
				'basket' => array('BasketItem' => array(), 'BasketTotals' => array())
			),
			$this->ShippingManager->data
		);
	}

	public function testAddDataFailsWhenOverrideIsOffAndItemExists() {
		$this->ShippingManager->addData('basket', array('BasketItem' => array(), 'BasketTotals' => array()));

		$this->assertFalse(
			$this->ShippingManager->addData('basket', array(), false)
		);
	}

	public function testGetDataReturnsDataCorrectlyOrFalseIfDoesntExist() {
		$this->ShippingManager->addData('basket', array('BasketItem' => array(), 'BasketTotals' => array()));

		$this->assertEquals(
			array('BasketItem' => array(), 'BasketTotals' => array()),
			$this->ShippingManager->getData('basket')
		);

		$this->assertFalse(
			$this->ShippingManager->getData('none-existant-item')
		);
	}

	public function testLoadHandlerLoadsTheCorrectClassOrReturnsFalseIfNotCreated() {
		$this->assertInstanceOf(
			'AmountTotalsHandler',
			$this->ShippingManager->loadHandler('EvShipping.AmountTotalsHandler')
		);

		$this->assertFalse(
			$this->ShippingManager->loadHandler('NonExistantHandler')
		);
	}

	public function testProcessShippingRuleCanAccessHandlerAndReturnValidRate() {
		$this->assertEquals(
			10.00,
			$this->ShippingManagerWorkingRule->processShippingRule(
				array(
					'ShippingRule' => array(
						'id' => 1,
						'name' => 'Weight Based UK Mainland',
						'handler' => 'EvShipping.AmountTotalsHandler',
						'sequence' => 2,
						'is_selectable' => 0,
						'is_active' => 1,
						'created' => '2015-09-03 09:45:31',
						'modified' => '2015-09-03 09:45:31'
					),
					'ShippingRuleData' => array()
				)
			)
		);
	}

	public function testProcessShippingRuleReturnsFalseOnInvalidDataAndRates() {
		$ShippingRule = array(
			'ShippingRule' => array(
				'id' => 1,
				'name' => 'Weight Based UK Mainland',
				'handler' => 'EvShipping.AmountTotalsHandler',
				'sequence' => 2,
				'is_selectable' => 0,
				'is_active' => 1,
				'created' => '2015-09-03 09:45:31',
				'modified' => '2015-09-03 09:45:31'
			),
			'ShippingRuleData' => array()
		);

		$this->assertFalse(
			$this->ShippingManagerNonWorkingRule->processShippingRule(
				$ShippingRule
			)
		);

		unset($ShippingRule['ShippingRule']['handler']);

		$this->assertFalse(
			$this->ShippingManager->processShippingRule(
				$ShippingRule
			)
		);
	}

	public function testGetRateCanReturnAValidShippingRateAndReturnsFalseWhenNoRulesPassedOrNoValidRateFound() {
		$rules = array(
			array(
				'ShippingRule' => array(
					'id' => 1,
					'name' => 'Weight Based UK Non-Mainland',
					'handler' => 'NonWorkingHandler',
					'sequence' => 1,
					'is_selectable' => 0,
					'is_active' => 1,
					'created' => '2015-09-03 09:45:31',
					'modified' => '2015-09-03 09:45:31'
				),
				'ShippingRuleData' => array()
			),
			array(
				'ShippingRule' => array(
					'id' => 1,
					'name' => 'Weight Based UK Mainland',
					'handler' => 'WorkingHandler',
					'sequence' => 2,
					'is_selectable' => 0,
					'is_active' => 1,
					'created' => '2015-09-03 09:45:31',
					'modified' => '2015-09-03 09:45:31'
				),
				'ShippingRuleData' => array()
			)
		);

		$this->assertEquals(
			10.00,
			$this->ShippingManager->getRate($rules)
		);

		$rules['1']['ShippingRule']['handler'] = 'NonWorkingHandler';
		$this->assertFalse(
			$this->ShippingManager->getRate($rules)
		);

		$this->assertFalse(
			$this->ShippingManager->getRate(array())
		);
	}

	public function testCalculateShippingFromAddressReturnsCorrectRate() {
		$this->assertEquals(
			10.00,
			$this->ShippingManager->calculateShippingFromAddress(
				array(
					'address1' => 'Evoluted New Media',
					'address2' => '35 Lambert Street',
					'city' => 'Sheffield',
					'post_code' => 'AA1 1AA',
					'country_id' => 1
				)
			)
		);
	}

	public function testCalculateShippingFromAddressReturnsFalseUponFails() {
		// no zones found that matches country id
		$this->assertFalse(
			$this->ShippingManager->calculateShippingFromAddress(
				array(
					'address1' => 'Evoluted New Media',
					'address2' => '35 Lambert Street',
					'city' => 'Sheffield',
					'post_code' => 'AA1 1AA',
					'country_id' => 9999
				)
			)
		);

		// zone found - matches country but no rules assigned to that zone
		$this->assertFalse(
			$this->ShippingManager->calculateShippingFromAddress(
				array(
					'address1' => 'Evoluted New Media',
					'address2' => '35 Lambert Street',
					'city' => 'Sheffield',
					'post_code' => 'AA1 1AA',
					'country_id' => 2
				)
			)
		);

		// Everything matches and loads handler but handler returns false
		// no rate found
		$this->assertFalse(
			$this->ShippingManager->calculateShippingFromAddress(
				array(
					'address1' => 'Evoluted New Media',
					'address2' => '35 Lambert Street',
					'city' => 'Sheffield',
					'post_code' => 'S3 7BH',
					'country_id' => 1
				)
			)
		);
	}

}

