<?php
App::uses('EvShippingAppModel', 'EvShipping.Model');
/**
 * ShippingRule Model
 *
 * @property ShippingRuleData $ShippingRuleData
 * @property Zone $Zone
 */
class ShippingRule extends EvShippingAppModel {

	/**
	 * Display field
	 *
	 * @var string
	 */
	public $displayField = 'name';

	/**
	 * Validation rules
	 *
	 * @var array
	 */
	public $validate = array(
		'name' => array(
			'notBlank' => array(
				'rule' => array('notBlank'),
				'message' => 'Please enter a name for this shipping rule'
			),
		),
		'handler' => array(
			'notBlank' => array(
				'rule' => array('notBlank'),
				'message' => 'A handler must be selected'
			),
		),
	);

	//The Associations below have been created with all possible keys, those that are not needed can be removed

	/**
	 * hasMany associations
	 *
	 * @var array
	 */
	public $hasMany = array(
		'ShippingRuleData' => array(
			'className' => 'EvShipping.ShippingRuleData',
			'dependent' => true,
			'cascade' => true
		)
	);

	/**
	 * hasAndBelongsToMany associations
	 *
	 * @var array
	 */
	public $hasAndBelongsToMany = array(
		'Zone' => array(
			'className' => 'EvShipping.Zone',
			'joinTable' => 'ev_shipping_shipping_rules_zones'
		),
		'DeliverySlot' => [
			'className' => 'EvShipping.DeliverySlot',
			'joinTable' => 'ev_shipping_delivery_slots_shipping_rules'
		]
	);

	/**
	 * beforeBeforeSave - Transform the rule data so cake can handle it all
	 *
	 * @return 	array|bool
	 */
	public function beforeBeforeSave($data, $options) {
		if (! empty($data['RuleData'])) {

			// Reformat to include currency
			foreach ($data['RuleData'] as $currencyId => $fields) {
				foreach ($fields as $fieldKey => $field) {
					$data['RuleData'][$currencyId][$fieldKey]['currency_id'] = $currencyId;
				}
			}

			$data['ShippingRuleData'] = $this->ShippingRuleData->formatForSave($data['RuleData']);
		}

		return $data;
	}

	/**
	 * redefine to bring in the zone
	 *
	 */
	public function readForEdit($id, $params = array()) {
		$params['contain'][] = 'Zone';

		if (Configure::read('EvShipping.enableDeliveryEstimates')) {
			$params['contain'][] = 'DeliverySlot';
		}

		return parent::readForEdit($id, $params);
	}

	/**
	 * get all the shipping rules in order
	 * by the zone id
	 *
	 * @param 	array 	$zoneIds 		Array of ZoneIds
	 * @return 	array 	$ShippingRules  Array of shipping rules
	 */
	public function getByZones($zoneIds) {
		$query = array(
			'joins' => array(
				array(
					'table' => 'ev_shipping_shipping_rules_zones',
					'alias' => 'ShippingRulesZone1',
					'conditions' => array(
						'ShippingRulesZone1.shipping_rule_id = ShippingRule.id'
					)
				)
			),
			'conditions' => array(
				'ShippingRulesZone1.zone_id' => $zoneIds,
				'ShippingRule.is_active' => 1,
			),
			'order' => 'ShippingRule.sequence ASC',
			'contain' => array(
				'ShippingRuleData' => array(
					'conditions' => array(
						'ShippingRuleData.currency_id' => CakeSession::read('EvCurrency.currencyId')
					)
				),
			)
		);

		if (Configure::read('EvShipping.enableDeliveryEstimates')) {
			$query['contain'][] = 'DeliverySlot';
		}

		return $this->find('all', $query);
	}
}
