<?php

App::uses('SearchableAppController', 'Searchable.Controller');
App::uses('Sanitize', 'Utility');

class SearchController extends SearchableAppController {

	public $uses = array('Searchable.SearchIndex');

	public $helpers = array('Searchable.Searchable');

/**
 * Allow searching without a login
 *
 * @return void
 */
	public function beforeFilter() {
		parent::beforeFilter();

		$this->Auth->allow(
			array(
				'index'
			)
		);
	}

/**
 * Search results page
 *
 * @return void
 */
	public function index() {
		$Model = $this->{$this->modelClass};

		$this->assignPage('SEARCH');

		if (isset($this->request->query['searchTerm']) && ! empty($this->request->query['searchTerm'])) {

			// NOTE: Remember to escape any sql using Sanitize if this function is overridden
			$searchTerm = $this->request->query['searchTerm'];

			$modelList = $this->_getModelList();
			$Model->searchModels($modelList); // set the models we are search
			$contains = $this->_getContains($Model->relationize($modelList)); // get the relationshipped version of the names so we can contain
			$pluginRef = $Model->pluginReference($modelList); // get a reference array for any that are plugin models

			$searchQuery = $this->_getSearchQuery($searchTerm, $contains);
			$paginationLimit = $this->_getPaginationLimit();

			if (!empty($paginationLimit)) {
				$searchQuery['limit'] = $paginationLimit;
				$this->paginate = $searchQuery;
				$results = $this->paginate($Model, [], ['relevance']);
				$this->set('paginated', true);
			} else {
				// Just get all the results
				$results = $Model->find('all', $searchQuery);
			}
			$this->set('searchTerm', $searchTerm);

			$showResultsCount = Configure::read('Searchable.showResultsCount');
			if (!empty($showResultsCount)) {
				$this->set('resultsCount', $Model->find('count', $searchQuery));
			}

			if (! empty($results)) {

				// format into sub array grouped by model
				$splitResults = Configure::read('Searchable.splitResults');
				if (!empty($splitResults)) {
					foreach ($results as $row) {
						$formattedResults[$row['SearchIndex']['model']][] = $row;
					}
					$this->set('sectionCount', count($formattedResults));
				} else {
					$formattedResults[$Model->alias] = $results;
				}

				$this->set(
					compact(
						'formattedResults',
						'pluginRef'
					)
				);
			} else {

				$this->set('noResults', true);
			}
		} else {

			$this->set('noResults', true);
		}

		$this->view = 'Searchable.Search/index';
	}

/**
 * Get the list of models, in app and plugins
 * then filter out all the ones that don't have the searchable behaviour
 *
 * @return array A list of models
 */
	protected function _getModelList() {
		// Get all the models in the main app model directory
		$modelList = App::objects('Model');

		// get all the plugins so we can get all their models
		$pluginList = App::objects('plugins');

		$ignore = Configure::read('Searchable.ignore');

		// loop and get the models
		foreach ($pluginList as $plugin) {
			if (!empty($ignore['plugins']) && in_array($plugin, $ignore['plugins'])) {
				continue;
			}

			// get the plugin models and prefix with the plugin name
			$pluginModel = App::objects($plugin . '.Model');

			foreach ($pluginModel as $key => $model) {
				$className = $plugin . '.' . $model;

				if (!empty($ignore['models']) && in_array($className, $ignore['models'])) {
					continue;
				}

				$pluginModel[$key] = $className;

				$overridenModelKey = array_search($plugin . $model, $modelList);
				if ($overridenModelKey !== false) {
					unset($modelList[$overridenModelKey]);
				}
			}

			$modelList = array_merge($modelList, $pluginModel);
		}

		foreach ($modelList as $key => $model) {

			if (strtolower($model) != 'searchindex') {

				try {
					$tmpModel = EvClassRegistry::init($model);
				} catch (CakeException $e) {
				}

				if (
					! is_object($tmpModel) ||
					! method_exists($tmpModel, 'hasBehaviour') ||
					! $tmpModel->hasBehaviour('Searchable.Searchable')
				) {

					unset($modelList[$key]);
				}
			}
		}
		return $modelList;
	}

/**
 * Override this to adjust the contains list and add some in for other models
 *
 * @param array $contains The current list of contains
 * @return array An array of contains
 */
	protected function _getContains($contains) {
		return $contains;
	}

/**
 * Get the search query. Override this to adjust the search
 * Note: searchTerm is not escaped. Make sure to escape it if using in raw SQL
 *
 * @param string $searchTerm The unescaped search term
 * @param array $contains An array of models to contain
 * @return array Query params
 */
	protected function _getSearchQuery($searchTerm, $contains) {
		$escapedSearchTerm = Sanitize::clean($searchTerm, ['remove_html' => true]);

		// do the search
		$searchQuery = [
			'fields' => [
				'*',
				"((MATCH(SearchIndex.data) AGAINST ('\"$escapedSearchTerm\"' IN BOOLEAN MODE) * 10)
					+ (MATCH(SearchIndex.data) AGAINST ('*$escapedSearchTerm*' IN BOOLEAN MODE) * 1.5)) AS relevance"
			],
			'conditions' => [
				'OR' => [
					"MATCH(SearchIndex.data) AGAINST('\"$escapedSearchTerm\"' IN BOOLEAN MODE) > 0",
					"MATCH(SearchIndex.data) AGAINST('*$escapedSearchTerm*' IN BOOLEAN MODE) > 0"
				],
				'SearchIndex.is_active' => 1
			],
			'order' => [
				'relevance' => 'DESC'
			],
			'contain' => $contains
		];

		return $searchQuery;
	}

/**
 * Determines how many results to fetch
 *
 * @return int The limit or 0 if disabled
 */
	protected function _getPaginationLimit() {
		return intval(Configure::read('Searchable.paginationLimit'));
	}

}
