<?php

App::uses('SearchableAppController', 'Searchable.Controller');

App::uses('Sanitize', 'Utility');

class SearchController extends SearchableAppController {

	public $uses = array('Searchable.SearchIndex');

	public $helpers = array('Searchable.Searchable');

	public function beforeFilter() {
		parent::beforeFilter();

		$this->Auth->allow(
			array(
				'index'
			)
		);
	}

	public function index() {
		$Model = $this->{$this->modelClass};

		if (defined('Page::SEARCH')) {
			$this->assignPage('SEARCH');
		}

		if (isset($this->request->query['searchTerm']) && ! empty($this->request->query['searchTerm'])) {
			$searchTerm = $this->request->query['searchTerm'];
			$escapedSearchTerm = Sanitize::escape($searchTerm);

			$modelList = $this->_getModelList();

			$Model->searchModels($modelList); // set the models we are search
			$contains = $Model->relationize($modelList); // get the relationshipped version of the names so we can contain
			$pluginRef = $Model->pluginReference($modelList); // get a reference array for any that are plugin models

			// do the search
			$results = $Model->find(
				'all',
				array(
					'fields' => array(
						'*',
						"((MATCH(SearchIndex.data) AGAINST ('\"$escapedSearchTerm\"' IN BOOLEAN MODE) * 10)
							+ (MATCH(SearchIndex.data) AGAINST ('$escapedSearchTerm' IN BOOLEAN MODE) * 1.5)) AS relevance"
					),
					'conditions' => array(
						'OR' => array(
							"MATCH(SearchIndex.data) AGAINST('\"$escapedSearchTerm\"' IN BOOLEAN MODE) > 0",
							"MATCH(SearchIndex.data) AGAINST('$escapedSearchTerm' IN BOOLEAN MODE) > 0"
						)
					),
					'order' => array(
						'relevance' => 'DESC'
					),
					'contain' => $contains
				)
			);

			if (! empty($results)) {
				// format into sub array grouped by model
				$formattedResults = array();
				foreach ($results as $row) {
					$formattedResults[$row['SearchIndex']['model']][] = $row;
				}

				$sectionCount = count($formattedResults);

				$this->set(
					compact(
						'formattedResults',
						'pluginRef',
						'sectionCount',
						'searchTerm'
					)
				);
			} else {
				$this->set('noResults', true);
			}
		} else {
			$this->set('noResults', true);
		}

		$this->view = 'Searchable.Search/index';
	}

	/**
	 * get the list of models, in app and plugins
	 * then filter out all the ones that don't have the searchable behaviour
	 *
	 */
	protected function _getModelList() {
		// Get all the models in the main app model directory
		$modelList = App::objects('Model');

		// get all the plugins so we can get all their models
		$pluginList = App::objects('plugins');

		// loop and get the models
		foreach ($pluginList as $plugin) {
			// get the plugin models and prefix with the plugin name
			$pluginModel = App::objects($plugin . '.Model');

			foreach ($pluginModel as $key => $model) {
				$pluginModel[$key] = $plugin . '.' . $model;
			}

			$modelList = array_merge($modelList, $pluginModel);
		}

		foreach ($modelList as $key => $model) {
			if (strtolower($model) != 'searchindex') {
				try {
					$tmpModel = ClassRegistry::init($model);
				} catch (CakeException $e) {
				}

				if (! is_object($tmpModel) ||
					! method_exists($tmpModel, 'hasBehaviour') ||
					! $tmpModel->hasBehaviour('Searchable.Searchable')
				) {
					unset($modelList[$key]);
				}
			}
		}

		return $modelList;
	}
}
