<?php
class SearchableBehavior extends ModelBehavior {

	public $defaultSettings = array(
		'foreignKey' => false,
		'_index' => false,
		'rebuildOnUpdate' => true,
		'fields' => '*',
		'stopwords_lang' => 'english'
	);

	public $settings = array();

	public $stopwords = array();

	public $SearchIndex;

	public $model;

	public function setup(Model $Model, $config = array()) {
		if (! isset($this->settings[$Model->alias])) {
			$this->settings[$Model->alias] = array_merge($this->defaultSettings, $config);
		}
		$this->model =& $Model;

		Configure::load('Searchable.stopwords');
		$stopwords = Configure::read('Searchable.stopwords');
		$stopwordsLang = $this->settings[$Model->alias]['stopwords_lang'];
		if (isset($stopwords[$stopwordsLang]) && is_array($stopwords[$stopwordsLang])) {
			$this->stopwords = $stopwords[$stopwordsLang];
			$this->__prepareStopwords();
		}
	}

	private function __prepareStopwords() {
		$stopwords = array();
		foreach ($this->stopwords as $word) {
			$stopwords[md5($word)] = $word;
		}
		$this->stopwords = $stopwords;
	}

	private function __processData(Model $Model) {
		if (method_exists($Model, 'indexData')) {
			return $Model->indexData();
		} else {
			return $this->index($Model);
		}
	}

	public function beforeSave(Model $Model, $options = array()) {
		if ($Model->id) {
			$this->settings[$Model->alias]['foreignKey'] = $Model->id;
		} else {
			$this->settings[$Model->alias]['foreignKey'] = 0;
		}

		if ($this->settings[$Model->alias]['foreignKey'] == 0 || $this->settings[$Model->alias]['rebuildOnUpdate']) {
			$this->settings[$Model->alias]['_index'] = $this->__processData($Model);
		}

		return true;
	}

	public function afterSave(Model $Model, $created, $options = array()) {
		$ModelAlias = EvClassRegistry::getNameFromModel($Model);
		if (!$this->SearchIndex) {
			$this->SearchIndex = EvClassRegistry::init('Searchable.SearchIndex', true);
		}

		if ($this->settings[$Model->alias]['_index'] !== false) {
			if ($this->settings[$Model->alias]['foreignKey'] == 0) {
				$this->settings[$Model->alias]['foreignKey'] = $Model->getLastInsertID();
				$this->SearchIndex->create();
				$this->SearchIndex->save(
					array(
						'SearchIndex' => array(
							'model' => $ModelAlias,
							'association_key' => $this->settings[$Model->alias]['foreignKey'],
							'data' => $this->settings[$Model->alias]['_index'],
							'is_active' => isset($Model->data[$Model->alias]['is_active']) ? $Model->data[$Model->alias]['is_active'] : 1
						)
					)
				);
			} else {
				$searchEntry = $this->SearchIndex->find('first', array(
					'conditions' => array(
						'model' => $ModelAlias,
						'association_key' => $this->settings[$Model->alias]['foreignKey']
					)
				));

				$this->SearchIndex->clear();
				$this->SearchIndex->save(
					array(
						'SearchIndex' => array(
							'id' => empty($searchEntry) ? 0 : $searchEntry['SearchIndex']['id'],
							'model' => $ModelAlias,
							'association_key' => $this->settings[$Model->alias]['foreignKey'],
							'data' => $this->settings[$Model->alias]['_index'],
							'is_active' => isset($Model->data[$Model->alias]['is_active']) ? $Model->data[$Model->alias]['is_active'] : 1
						)
					)
				);
			}
			$this->settings[$Model->alias]['_index'] = false;
			$this->settings[$Model->alias]['foreignKey'] = false;

		} elseif ($this->settings[$Model->alias]['foreignKey'] !== 0) {
			// This should not be indexed - remove the entry if one exists
			$this->SearchIndex->deleteAll([
				'association_key' => $this->settings[$Model->alias]['foreignKey'],
				'model' => $ModelAlias,
			]);
		}
		return true;
	}

/**
 * Generate a string to add to the search indices table
 *
 * @param Model $Model Model
 * @return string
 */
	public function index(Model $Model) {
		$index = array();

		$data = $Model->data[$Model->alias];

		$columns = $Model->getColumnTypes();

		if ($this->settings[$Model->name]['fields'] === '*') {
			$this->settings[$Model->name]['fields'] = array_keys($columns);
		} elseif (is_string($this->settings[$Model->name]['fields'])) {
			$this->settings[$Model->name]['fields'] = [$this->settings[$Model->name]['fields']];
		}

		foreach ($data as $field => $value) {
			if (
				is_string($value)
				&& in_array($field, $this->settings[$Model->name]['fields'])
				&& $this->_isIndexableField($Model, $field, $columns)
			) {
				$index[] = strip_tags(html_entity_decode($value, ENT_COMPAT, 'UTF-8'));
			}
		}

		$index = implode('. ', $index);
		// Use mb_convert_encoding over iconv if available as it will handle illegal characters
		// better.
		if (function_exists('mb_convert_encoding')) {
			$index = mb_convert_encoding($index, 'ASCII', 'UTF-8');
		} else {
			$index = iconv('UTF-8', 'ASCII//TRANSLIT', $index);
		}

		$index = preg_replace('/[\ ]+/', ' ', $index);

		return $this->__removeStopwords($index);
	}

/**
 * Determine if a field is indexable
 *
 * @param Model $Model Model
 * @param string $field Field name
 * @param array $columns Model columns
 * @return bool True if the field can added to the index
 */
	protected function _isIndexableField(Model $Model, $field, array $columns = []) {
		if ($field === $Model->primaryKey) {
			// The primary key is never indexed
			return false;
		}

		if (empty($columns[$field])) {
			// The field is unknown
			return false;
		}

		return in_array($columns[$field], ['text', 'varchar', 'char', 'string']);
	}

	private function __removeStopwords($index) {
		$words = explode(' ', $index);
		foreach ($words as $word) {
			if (isset($this->stopwords[md5($word)])) {
				$search = ' ' . $word . ' ';
				$index = str_replace($search, ' ', $index);
			}
		}

		return $index;
	}

	public function afterDelete(Model $Model) {
		if (!$this->SearchIndex) {
			$this->SearchIndex = EvClassRegistry::init('Searchable.SearchIndex', true);
		}
		$conditions = array('model' => EvClassRegistry::getNameFromModel($Model), 'association_key' => $Model->id);
		$this->SearchIndex->deleteAll($conditions);
	}

/**
 * Search model for query term
 *
 * @param Model $Model Model to search
 * @param string $q Query term
 * @param array $findOptions Find options
 * @return array Search results
 */
	public function search(Model $Model, $q, $findOptions = array()) {
		return $this->_search($Model, $q, $findOptions);
	}

/**
 * Return count of found search results for a query term
 *
 * @param Model $Model Model to search
 * @param string $q Query term
 * @param array $findOptions Find options
 * @return array Search results
 */
	public function searchCount(Model $Model, $q, $findOptions = array()) {
		return $this->_search($Model, $q, $findOptions, 'count');
	}

/**
 * Search model for query term
 *
 * @param Model $Model Model to search
 * @param string $q Query term
 * @param array $findOptions Find options
 * @param string $findMethod Find method, either 'all' or 'count'
 * @return array|int Search results or count
 */
	protected function _search(Model $Model, $q, $findOptions = array(), $findMethod = 'all') {
		App::uses('Sanitize', 'Utility');
		$q = Sanitize::escape($q);

		if (! $this->SearchIndex) {
			$this->SearchIndex = EvClassRegistry::init('Searchable.SearchIndex', true);
		}

		// Bind the model to the SearchIndex and contain with the query as we use the model's
		// primary key in the conditions to ensure that the index is still for an existing record.
		$this->SearchIndex->searchModels(EvClassRegistry::getNameFromModel($Model));
		$findOptions['contain'] = array_merge(
			! empty($findOptions['contain']) ? (array)$findOptions['contain'] : array(),
			array($Model->alias)
		);

		if (! isset($findOptions['fields'])) {
			$findOptions['fields'] = array();
		}

		$findOptions['fields'] = array_merge(
			$findOptions['fields'],
			array(
				'*',
				"((MATCH(SearchIndex.data) AGAINST ('\"$q\"' IN BOOLEAN MODE) * 10)
					+ (MATCH(SearchIndex.data) AGAINST ('*$q*' IN BOOLEAN MODE) * 1.5)) AS relevance"
			)
		);

		if (! isset($findOptions['conditions'])) {
			$findOptions['conditions'] = array();
		}

		$findOptions['conditions'] = array_merge(
			$findOptions['conditions'],
			array(
				'OR' => array(
					"MATCH(SearchIndex.data) AGAINST('\"$q\"' IN BOOLEAN MODE) > 0",
					"MATCH(SearchIndex.data) AGAINST('*$q*' IN BOOLEAN MODE) > 0"
				)
			)
		);

		$findOptions['order'] = array('relevance ASC');

		return $this->SearchIndex->find($findMethod, $findOptions);
	}

}
