<?php
class IndexTask extends Shell {

	public $uses = array('SearchIndex');

	public $tasks = array('DbConfig', 'Model', 'Searchable.ProgressBar');

	public $modelCache = array();

	public function execute() {
		ini_set('memory_limit', '5000M');
		set_time_limit(0);

		if (isset($this->args[0])) {
			$this->connection = $this->args[0];
			$this->out(__('Using database config: %s', $this->connection));
		}

		if (empty($this->connection)) {
			$this->connection = $this->DbConfig->getConfig();
		}

		$this->tables = $this->Model->getAllTables($this->connection);
		$this->modelNames = array();

		if (isset($this->args[1])) {
			$models = str_replace(' ', '', $this->args[1]);

			if (strtolower($models) == 'all') {
				$count = count($this->tables);
				$models = array();
				for ($i = 0; $i < $count; $i++) {
					$models[] = $this->_modelName($this->tables[$i]);
				}
			} else {
				$models = explode(',', $models);
			}

			foreach ($models as $k => $model) {
				if (strtolower($model) != 'searchindex') {
					$tmpModel = EvClassRegistry::init($model);
					if ($tmpModel) {
						if (
							isset($tmpModel->actsAs) &&
							is_array($tmpModel->actsAs) &&
							$tmpModel->hasBehaviour('Searchable.Searchable')
						) {
							$this->modelCache[$model] = $tmpModel;
							$this->modelNames[$k] = $model;
						} else {
							$this->out(__('<warning>Notice: Searchable is no behaviour for %s.</warning>', $model));
						}
					} else {
						$this->error(__('Class %s does not exists.', $model));
					}
				}
			}

			if (count($this->modelCache) > 0) {
				$this->out(__('Rebuilding indices for: %s', implode(', ', array_keys($this->modelCache))));
				$this->hr();
				$this->indexModels();
				$this->finish();
			}

			return;
		}

		$this->hr();
		$this->out(__('Rebuild index for the following models:'));
		$this->hr();
		$count = count($this->tables);
		$this->out(sprintf("%2d. %s", 0, __('All models')));

		$modelList = $this->_getModelList();

		$tableIndex = 0;
		foreach ($modelList as $model) {

			if (strtolower($model) != 'searchindex') {

				try {
					$tmpModel = EvClassRegistry::init($model);
				} catch (CakeException $e) {
				}

				if ($tmpModel) {
					if (
						isset($tmpModel->actsAs) &&
						is_array($tmpModel->actsAs) &&
						$tmpModel->hasBehaviour('Searchable.Searchable')
					) {
						$this->modelCache[$model] = $tmpModel;
						$tableIndex++;
						$this->modelNames[$tableIndex] = $model;
						$this->out(sprintf("%2d. %s", $tableIndex, $this->modelNames[$tableIndex]));
						unset($tmpModel);
					}
				}
			}
		}

		$this->out(sprintf(" Q. %s", __('Quit')));
		$modelChoice = strtolower($this->in(null, null, 'Q'));

		if ($modelChoice == 'q') {
			$this->out('');
			$this->hr();
			return;
		}

		if ($modelChoice == '0') {
			$this->indexModels();
		} else {
			$this->indexModel($this->modelNames[(int)$modelChoice]);
		}

		$this->finish();
		return;
	}

	/**
	 * get the list of models, in app and plugins
	 *
	 */
	protected function _getModelList() {
		// Get all the models in the main app model directory
		$modelList = App::objects('Model');

		// get all the plugins so we can get all their models
		$pluginList = App::objects('plugins');

		// loop and get the models
		foreach ($pluginList as $plugin) {

			// get the plugin models and prefix with the plugin name
			$pluginModel = App::objects($plugin . '.Model');
			foreach ($pluginModel as $key => $model) {

				$pluginModel[$key] = $plugin . '.' . $model;

				$overridenModelKey = array_search($plugin . $model, $modelList);
				if ($overridenModelKey !== false) {
					unset($modelList[$overridenModelKey]);
				}
			}

			$modelList = array_merge($modelList, $pluginModel);
		}

		return $modelList;
	}

	public function indexModel($model) {
		$Model = $this->modelCache[$model];
		$results = $Model->find('all', array(
			'recursive' => 0
		));

		$this->out(__('Creating new indices for %s...', $model));
		$this->ProgressBar->start(count($results));

		// try to unload all behaviours except the searchable behaviour
		foreach ($Model->actsAs as $key => $value) {

			if (is_numeric($key)) {

				$behaviour = $value;
			} else {

				$behaviour = $key;
			}

			if (in_array($behaviour, array('Searchable.Searchable', 'Searchable'))) {
				continue;
			}

			$Model->Behaviors->unload($behaviour);
		}

		foreach ($results as $result) {
			$Model->id = $result[$Model->alias]['id'];
			$data = $result[$Model->alias];
			$data['modified'] = false;
			unset($data['id']);

			$Model->save($data, array('validate' => false));
			$this->ProgressBar->next();
		}

		$this->out('');
		$this->hr();
	}

	public function indexModels() {
		foreach ($this->modelNames as $model) {
			$this->indexModel($model);
		}
	}

	public function finish() {
		$this->out('', 1);
		$this->out(__('All indices have been updated :)'), 2);
		$this->hr();
	}
}