<?php

App::uses('Component', 'Controller');
use Omnipay\Omnipay as Omnipay;

class SagePayComponent extends Component {

	protected $_controller = null;

	protected $_api = null;

	protected $_data = null;

	protected $_errorUrl = null;

	/**
	 * Sets a reference to the calling controller from within the component.
	 *
	 * @see Component::initialize()
	 */
	public function initialize(Controller $controller) {
		parent::initialize($controller);

		$this->_controller = $controller;

		$this->_api = Omnipay::create('SagePay_Direct');
	}

	/**
	 * init
	 *
	 * Use the function setup and connection / settings that need setting up in the gateway
	 */
	public function setup() {
		if (strtolower(Configure::read('app.environment')) === 'production') {

			$this->_config = Configure::read('EvSagePay.SagePayDirect.live');

		} else {

			$this->_config = Configure::read('EvSagePay.SagePayDirect.dev');

			$this->_api->setTestMode(true);

		}
		$this->_api->setVendor($this->_config['vendor']);
	}

	/**
	 * setupPayment
	 *
	 * Use this function setup the actual payment, i.e. setup the basket, the amount to take etc...
	 *
	 * @param int - transaction id we have created
	 * @param array $return - array with 2 keys of 'return', 'cancel'. Containing either a link or router array for redirecting user
	 * @param array $model - array with two keys of 'model' and 'model_id', used to link transactions polymorphically to other model items
	 * @param float|array $amount - amount of monies to take (I GOT YOUR MONIESSSSSS), or array of 'amount' and 'currency' to change currencies (will take default from config)
	 * @param array $items - multidimenisional array break down of the transaction items, required elements are 'description' and 'amount'
	 * @param mixed $extra - variable allowing you to pass ay data needed to the gateway, could be things like addresses that are not tracked by the transactions model
	 */
	public function setupPayment($transactionId, $return, $model, $amount, $items, $extra = array()) {
		$this->Transaction = EvClassRegistry::init('Transactions.Transaction');
		$this->Transaction->id = $transactionId;

		$return['return'] = $return['return'] . '?transaction=' . $transactionId;

		if (strpos($extra['CreditCard']['card_holder_name'], ' ') !== false) {
			$name = explode(' ', $extra['CreditCard']['card_holder_name'], 2);
			$firstName = $name[0];
			$lastName = $name[1];
		} else {
			$firstName = $extra['CreditCard']['card_holder_name'];
			$lastName = ' . ';
		}

		$cardData = array(
			'firstName' => $firstName,
			'lastName' => $lastName,
			'number' => $extra['CreditCard']['card_number'],
			'startMonth' => (!empty($extra['CreditCard']['start_month'])) ? $extra['CreditCard']['start_month'] : null,
			'startYear' => (!empty($extra['CreditCard']['start_year'])) ? $extra['CreditCard']['start_year'] : null,
			'expiryMonth' => $extra['CreditCard']['expiry_month'],
			'expiryYear' => $extra['CreditCard']['expiry_year'],
			'cvv' => $extra['CreditCard']['security_number'],
			'address1' => $extra['delivery']['Address']['address1'],
			'address2' => $extra['delivery']['Address']['address2'],
			'city' => $extra['delivery']['Address']['city'],
			'postcode' => $extra['delivery']['Address']['post_code'],
			'state' => '',
			'country' => 'GB',
			'email' => $extra['user']['User']['email'],
			'phone' => ''
		);

		$params = Hash::merge(['card' => $cardData], $this->_setParams($transactionId, $return, $amount));

		// Store the customers details on the transaction as we'll need them when they return.
		$this->_data = $this->_api->purchase(
			$this->_modifyParams($params, $transactionId, $return, $model, $amount, $items, $extra)
		);
	}

	/**
	 * getPayment
	 *
	 * Everything should be setup, actually take the payment
	 *
	 * @param int transactions id
	 * @return mixed dependent on the gateway, value is return straight from the transaction component to user anyway
	 */
	public function getPayment($transactionId) {
		try {
			$result = $this->_sendRequest($this->_data);
		} catch (Exception $e) {
			return [
				'result' => false,
				'message' => $e->getMessage(),
				'transaction_id' => $transactionId,
			];
		}

		return [
			'result' => $result['result'],
			'message' => ! empty($result['message']) ? $result['message'] : null,
			'transaction_id' => $transactionId,
		];

	}

	/**
	 * return
	 *
	 * deal with a return from the gateway and check for success / fail
	 *
	 * @return array - with three elements,
	 *				 - 'result' = true/false value
	 *  			 - 'message' = text message about transaction (i.e. reason for failing)
	 * 				 - 'transaction_id' = int of the transaction row
	 */
	public function processReturn() {
		return $this->process3dSecure();
	}

/**
 * Handles the standard EvTransaction return. For the case of SagePay we
 * just hand this response off to the process3dSecure method to keep the
 * logic simple and tidy.
 *
 * @return array Returns the standard EvTransactions response array
 *               consisting of a 'result' (bool), message (string) and the
 *               transaction_id.
 */

	public function process3dSecure() {
		$params = $this->_getParams();

		$vpsTxId = null;
		if (! empty($params['VPSTxId'])) {
			$vpsTxId = $params['VPSTxId'];
		}

		$cres = null;
		if (! empty($this->_controller->request->data['cres'])) {
			$cres = $this->_controller->request->data['cres'];
		}

		$request = $this->_api->completePurchase([
			'threeDSSessionData' => $vpsTxId,
			'CRes' => $cres
		]);

		if (isset($params['VPSTxId'])) {
			$request->setVPSTxId($params['VPSTxId']);
		}

		$transactionId = null;
		if (! empty($params['transactionId'])) {
			$transactionId = $params['transactionId'];
		}

		if (strpos($transactionId, '-') !== false) {
			$transactionId = substr($params['transactionId'], 0, strpos($params['transactionId'], '-'));
		}

		// If the vpsTxId value is empty then it means 3D secure isn't being
		// used, so we can skip to the getPayment step, which is what runs at
		// the end of the transaction payment process.
		if (empty($vpsTxId)) {
			return $this->getPayment($transactionId);
		}

		try {
			$result = $request->send();
			$responseData = $result->getData();
		} catch (\Exception $e) {
			return [
				'result' => false,
				'message' => $e->getMessage(),
				'transaction_id' => $transactionId,
			];
		}

		$message = '';
		if (! empty($responseData['StatusDetail'])) {
			$message = $responseData['StatusDetail'];
		}

		return [
			'result' => $responseData['Status'] == 'OK',
			'message' => $message,
			'transaction_id' => $transactionId,
		];

	}

/**
 * Provides a helper method to store an array of transaction data into the
 * database. Because we can't fully trust that session data is always
 * available, we store the values, so we can use them later.
 *
 * @param int $transactionId ID of the transaction to store the data for
 * @param array $params Array of data to store in key => value format
 * @return void
 */
	public function storeTransactionParams($transactionId, $params) {
		$this->_controller->loadModel('EvTransactions.TransactionData');

		$transactionData = [];

		foreach ($params as $name => $data) {
			$transactionData[] = [
				'transaction_id' => $transactionId,
				'name' => $name,
				'data' => $data,
			];
		}

		$this->_controller->TransactionData->saveMany($transactionData);
	}

/**
 * Retrieves the stored transaction data for a given transaction ID.
 *
 * @param int $transactionId ID of the transaction to retrieve the data for
 * @return array Array of data stored for the transaction in key => value format
 */
	public function readStoredTransactionData($transactionId) {
		if (empty($transactionId)) {
			return [];
		}

		$this->_controller->loadModel('EvTransactions.TransactionData');

		$transactionData = $this->_controller->TransactionData->find('all', [
			'conditions' => [
				'transaction_id' => $transactionId,
			],
		]);

		$data = [];

		foreach ($transactionData as $row) {
			$data[$row['TransactionData']['name']] = $row['TransactionData']['data'];
		}

		return $data;
	}

	protected function _sendRequest($request) {
		$params = $this->_getParams();

		$this->Transaction = EvClassRegistry::init('EvTransactions.Transaction');

		// The transaction id can sometimes contain the date depending on what
		// version of the EvTransactions plugin is being used. We need to strip
		// this out if it's there.
		if (strpos($params['transactionId'], '-')) {
			$this->Transaction->id = substr($params['transactionId'], 0, strpos($params['transactionId'], '-'));
		} else {
			$this->Transaction->id = $params['transactionId'];
		}

		$transactionResult = $this->Transaction->findById($this->Transaction->id);

		//If 3D secure is enabled then the transaction will need to be updated. Otherwise return that it has already been successful.
		if ($transactionResult['Transaction']['status'] == 'success') {
			return ['result' => true];
		}

		// If the request was empty it means we've either been redirected back
		// without a response, or we were unable to communicate with SagePay.
		// Either way the transaction has failed, and we can't continue.
		if (empty($request)) {
			$this->Transaction->save([
				'Transaction' => [
					'status' => 'failed',
					'message' => 'Unable to communicate with SagePay.',
				],
			]);

			return [
				'result' => false,
				'message' => 'No request object found',
			];
		}

		try {
			$response = $request->send();
			$responseData = $response->getData();

			//Customer is successfully paid
			if ($response->isSuccessful()) {
				// load the transactions model and update with the token
				$this->Transaction->save(array(
					'Transaction' => array(
						'payment_token' => $response->getTransactionReference(),
						'status' => 'success'
					)
				));
				$result = array(
					'message' => $response->getMessage(),
					'status' => 'success',
					'result' => true
				);

			} elseif ($response->isRedirect()) {

				// If SagePay tells us the transaction required 3D Secure checks
				// (as of 2022 almost every transaction will) then we need to
				// redirect the user to our internal 3D Secure page, where we'll
				// be posting back to SagePay, who will then redirect the user
				// to their bank's 3D Secure page
				if ($responseData['Status'] == '3DAUTH') {
					$this->Transaction->save(array(
						'Transaction' => array(
							'payment_token' => $response->getTransactionReference(),
							'status' => 'initiated'
						)
					));

					// Write a bunch of data to the session, so we can retrieve
					// it throughout the 3D Secure process
					$responseData['vendorTxCode'] = $this->Transaction->id;

					$creq = null;
					if (! empty($responseData['CReq'])) {
						$creq = $responseData['CReq'];
					} elseif (! empty($responseData['PaReq'])) {
						// This is a backwards compatible value for the CReq value. 
						$creq = $responseData['PaReq'];
					}

					$vpstxid = null;
					if (! empty($responseData['VPSTxId'])) {
						$vpstxid = $responseData['VPSTxId'];
					} elseif (! empty($responseData['MD'])) {
						// This is a backwards compatible value for the VPSTxId value. 
						$vpstxid = $responseData['MD'];
					}

					if (empty($creq) || empty($vpstxid)) {
						throw new Exception('Missing creq or vpstxid values');
					}

					$this->_controller->Session->write('SagePay.3dSecure.data', $responseData);
					$this->_controller->Session->write('SagePay.creq', $creq);
					$this->_controller->Session->write('SagePay.vpstxid', $vpstxid);

					// Store the transaction data in the db as a session can be lost
					$this->storeTransactionParams($this->Transaction->id, [
						'vpstxid' => $vpstxid,
						'creq' => $creq,
					]);

					return $this->_controller->redirect([
						'plugin' => 'ev_sage_pay',
						'controller' => 'sage_pay',
						'action' => 'secure3dRedirect',
					]);
				}

				// If the redirect isn't for the initial 3D Secure process then
				// we run the SagePay redirect logic. This will likely be for
				// the final 3D Secure return on most transactions, but supports
				// whatever SagePay or the bank in question wants to do here.
				$response->redirect();


			} else {

				$this->Transaction->save(array(
					'Transaction' => array(
						'status' => 'failed',
						'message' => $response->getMessage()
					)
				));

				$result = array(
					'message' => $response->getMessage(),
					'status' => 'failed',
					'result' => false
				);
			}

		} catch(Exception $e) {
			$result = array(
				'status' => 'failed',
				'result' => false
			);

			$message = $e->getMessage();

			// Only interested in real responses
			if ($message != 'Invalid response from payment gateway') {
				$this->Transaction->save(array(
					'Transaction' => array(
						'status' => 'failed',
						'message' => $message,
					)
				));

				$result['message'] = $message;
			}

		}

		return $result ? $result : [];
	}

/**
 * Set the parameters of the sage pay transaction and save it to the session.
 * @param int    $transactionId The id of the current transaction.
 * @param array  $return        The return urls for when the transaction has been completed.
 * @param array  $amount        The amount of the transaction.
 * @return array An array containing the transaction data
 */
	protected function _setParams($transactionId, $return, $amount) {
		if (is_array($amount)) {
			$currencyCode = $amount['currency'];
		} else {
			$currencyCode = Configure::read('EvSagePay.currency');
		}

		if (empty($this->_config['transactionReference'])) {
			$this->_config['transactionReference'] = $transactionId . "-" . time();
		}

		// We set the (new for 2022) threeDsNotificationUrl to a page within the
		// EvSagePay plugin. This is a basic form that simply auto-submits the
		// required details to the SagePay endpoint. In theory, you can override
		// this with your own url but this is not necessary.
		$threeDsNotificationUrl = Router::url([
			'plugin' => 'ev_sage_pay',
			'controller' => 'sage_pay',
			'action' => 'secure3dCallback',
			'admin' => false,
		], true);

		$this->_controller->Session->write('SagePay.returnUrl', $return['return']);

		$params = [
			'transactionId' => $transactionId . "-" . time(),
			'transactionReference' => $this->_config['transactionReference'],
			'returnUrl' => $return['return'],
			'ThreeDSNotificationURL' => $threeDsNotificationUrl . '?transactionId=' . $transactionId,
			'currency' => $currencyCode,
			'amount' => $amount['amount'],
			'description' => 'Payment to ' . Configure::read('SiteSetting.site_title'),
			'clientIp' => $this->_controller->request->clientIp(),

			// The Apply3DSecure flag has a few options, typically we never want
			// to be making a decision on if we want to use 3D Secure or not,
			// that should be left up to SagePay, hence we leave this set to 0.
			// 0 = default, 1 = force, 2 = disable, 3 = force when enabled (
			// regardless of card compatability)
			'Apply3DSecure' => 0,
		];

		// If the transaction is being taken by the client over the telephone we
		// have to tell SagePay to set the accountType to MOTO. This mainly a
		// legacy SagePay Server feature but does apply to some of our sites.
		// More details on this can be found here: https://www.opayo.co.uk/support/account/merchant-numbers/general-setup/account-type-m-everything-you-need-to-know
		if ($this->_controller->Session->read('EvSagePay.SagepayDirect.accountType') == 'M') {
			$params['accountType'] = 'M';
		}

		// We write all the params to the session so that we can access them
		// throughout the payment process, primarily for the 3D Secure callbacks
		$this->_controller->Session->write('EvSagePay.params', $params);

		$this->storeTransactionParams($transactionId, $params);

		return $params;
	}

/**
 * Modify the parameters sent with a purchase to sagepay. Allows extended functionality to be added.
 *
 * @param array       $params The current parameters to be sent.
 * @param int         $transactionId - transaction id we have created
 * @param array       $return - array with 2 keys of 'return', 'cancel'. Containing either a link or router array
 *                            for redirecting user
 * @param array       $model  - array with two keys of 'model' and 'model_id', used to link transactions
 *                            polymorphically to other model items
 * @param float|array $amount - amount of monies to take (I GOT YOUR MONIESSSSSS), or array of 'amount' and
 *                            'currency' to change currencies (will take default from config)
 * @param array       $items  - multidimenisional array break down of the transaction items, required elements are
 *                            'description' and 'amount'
 * @param mixed       $extra  - variable allowing you to pass ay data needed to the gateway, could be things like
 *                            addresses that are not tracked by the transactions model
 * @return array The modified parameters.
 */
	protected function _modifyParams($params, $transactionId, $return, $model, $amount, $items, $extra) {
		return $params;
	}

/**
 * Read the sage pay transaction parameters from the session. If the session is empty then the parameters
 * will be attempted to be reconstructed from the transaction in the database by reading the transaction
 * id from the current request query parameters.
 * @return array The sage pay parameter array.
 */
	protected function _getParams() {
		$params = [];
		if (! empty($this->_controller->Session->read('EvSagePay.params'))) {
			$params = $this->_controller->Session->read('EvSagePay.params');
		}
		
		$postData = $this->_controller->request->data;

		if (! empty($postData)) {
			$params = array_merge($params, $postData);
		}

		//If the session params are empty then try to construct it from the available data in the transaction row.
		if ((empty($params) || isset($params['cres'])) && !empty($this->_controller->request->query['transaction'])) {
			$transactionId = $this->_controller->request->query['transaction'];

			// Add the transaction id to the sagepay params
			$params['transactionId'] = $transactionId;

			//Attempt to find the transaction in the database
			$this->_controller->loadModel('EvTransactions.Transaction');
			$transaction = $this->_controller->Transaction->find(
				'first',
				[
					'conditions' => [
						'Transaction.id' => $transactionId
					],
					'contain' => [
						'Currency',
					]
				]
			);

			if (!empty($transaction)) {
				if (!empty($this->_config['transactionReference'])) {
					$params['transactionReference'] = $this->_config['transactionReference'];
				}

				if (!empty($transaction['Transaction']['transaction_amount'])) {
					$params['amount'] = $transaction['Transaction']['transaction_amount'];
				}

				if (!empty($transaction['Currency']['name'])) {
					$params['currency'] = $transaction['Currency']['name'];
				}

				$description = 'Website Payment';

				$siteTitle = Configure::read('SiteSetting.site_title');

				if (! empty($siteTitle)) {
					$description .= ' to ' . $siteTitle;
				}

				$params['description'] = $description;
				$params['clientIp'] = $this->_controller->request->clientIp();

				if (! empty($transaction['Transaction']['payment_token'])) {
					$paymentToken = json_decode($transaction['Transaction']['payment_token'], true);

					if (is_array($paymentToken)) {
						$params = array_merge($params, $paymentToken);
					}
				}
			}
		}

		return $params;
	}
}
