<?php

App::uses('Component', 'Controller');
use Omnipay\Omnipay as Omnipay;

class SagePayComponent extends Component {

	protected $_controller = null;

	protected $_api = null;

	protected $_data = null;

	protected $_errorUrl = null;

	/**
	 * Sets a reference to the calling controller from within the component.
	 *
	 * @see Component::initialize()
	 */
	public function initialize(Controller $controller) {
		parent::initialize($controller);

		$this->_controller = $controller;

		$this->_api = Omnipay::create('SagePay_Direct');
	}

	/**
	 * init
	 *
	 * Use the function setup and connection / settings that need setting up in the gateway
	 */
	public function setup() {
		if (strtolower(Configure::read('app.environment')) === 'production') {

			$this->_config = Configure::read('EvSagePay.SagePayDirect.live');

		} else {

			$this->_config = Configure::read('EvSagePay.SagePayDirect.dev');

			$this->_api->setTestMode(true);

		}
		$this->_api->setVendor($this->_config['vendor']);
	}

	/**
	 * setupPayment
	 *
	 * Use this function setup the actual payment, i.e. setup the basket, the amount to take etc...
	 *
	 * @param int - transaction id we have created
	 * @param array $return - array with 2 keys of 'return', 'cancel'. Containing either a link or router array for redirecting user
	 * @param array $model - array with two keys of 'model' and 'model_id', used to link transactions polymorphically to other model items
	 * @param float|array $amount - amount of monies to take (I GOT YOUR MONIESSSSSS), or array of 'amount' and 'currency' to change currencies (will take default from config)
	 * @param array $items - multidimenisional array break down of the transaction items, required elements are 'description' and 'amount'
	 * @param mixed $extra - variable allowing you to pass ay data needed to the gateway, could be things like addresses that are not tracked by the transactions model
	 */
	public function setupPayment($transactionId, $return, $model, $amount, $items, $extra = array()) {
		$this->Transaction = ClassRegistry::init('Transactions.Transaction');
		$this->Transaction->id = $transactionId;

		$return['return'] = $return['return'] . '?transaction=' . $transactionId;

		if (strpos($extra['CreditCard']['card_holder_name'], ' ') !== false) {
			$name = explode(' ', $extra['CreditCard']['card_holder_name'], 2);
			$firstName = $name[0];
			$lastName = $name[1];
		} else {
			$firstName = $extra['CreditCard']['card_holder_name'];
			$lastName = ' . ';
		}

		$cardData = array(
			'firstName' => $firstName,
			'lastName' => $lastName,
			'number' => $extra['CreditCard']['card_number'],
			'startMonth' => (!empty($extra['CreditCard']['start_month'])) ? $extra['CreditCard']['start_month'] : null,
			'startYear' => (!empty($extra['CreditCard']['start_year'])) ? $extra['CreditCard']['start_year'] : null,
			'expiryMonth' => $extra['CreditCard']['expiry_month'],
			'expiryYear' => $extra['CreditCard']['expiry_year'],
			'cvv' => $extra['CreditCard']['security_number'],
			'address1' => $extra['delivery']['Address']['address1'],
			'address2' => $extra['delivery']['Address']['address2'],
			'city' => $extra['delivery']['Address']['city'],
			'postcode' => $extra['delivery']['Address']['post_code'],
			'state' => '',
			'country' => 'GB',
			'email' => $extra['user']['User']['email'],
			'phone' => ''
		);

		//Set currency
		if (is_array($amount)) {
			$currencyCode = $amount['currency'];
		} else {
			$currencyCode = Configure::read('EvSagePay.currency');
		}

		$params = Hash::merge(['card' => $cardData], $this->_setParams($transactionId, $return, $amount));

		// Store the customers details on the transaction as we'll need them when they return.
		$this->_data = $this->_api->purchase(
			$params
		);
	}

	/**
	 * getPayment
	 *
	 * Everything should be setup, actually take the payment
	 *
	 * @param int transactions id
	 * @return mixed dependent on the gateway, value is return straight from the transaction component to user anyway
	 */
	public function getPayment($transactionId) {
		$result = $this->_sendRequest($this->_data);

		return array(
			'result' => $result,
			'transaction_id' => $transactionId
		);
	}

	/**
	 * return
	 *
	 * deal with a return from the gateway and check for success / fail
	 *
	 * @return array - with three elements,
	 *				 - 'result' = true/false value
	 *  			 - 'message' = text message about transaction (i.e. reason for failing)
	 * 				 - 'transaction_id' = int of the transaction row
	 */
	public function processReturn() {
		return $this->process3dSecure();
	}

	public function process3dSecure() {
		$params = $this->_getParams();

		$request = $this->_api->completePurchase($params);

		$result = $this->_sendRequest($request);

		return array(
			'result' => $result['result'],
			'message' => isset($result['message']) ? $result['message'] : '',
			// Remove everything after the - when returning the transaction id, -[TIMESTAMP] is added to the ID to make
			// it unique when dealing with the payment gateway
			'transaction_id' => substr($params['transactionId'], 0, strpos($params['transactionId'], '-'))
		);
	}

	protected function _sendRequest($request) {
		$params = $this->_getParams();

		$this->Transaction = ClassRegistry::init('EvTransactions.Transaction');
		$this->Transaction->id = substr($params['transactionId'], 0, strpos($params['transactionId'], '-'));

		$transactionResult = $this->Transaction->findById($this->Transaction->id);

		//If 3D secure is enabled then the transaction will need to be updated. Otherwise return that it has already been successful.
		if ($transactionResult['Transaction']['status'] == 'success') {
			return ['result' => true];
		}

		try {

			$response = $request->send();

			//Customer is successfully paid
			if ($response->isSuccessful()) {
				// load the transactions model and update with the token
				$this->Transaction->save(array(
					'Transaction' => array(
						'payment_token' => $response->getTransactionReference(),
						'status' => 'success'
					)
				));
				$result = array(
					'message' => $response->getMessage(),
					'status' => 'success',
					'result' => true
				);

			} elseif ($response->isRedirect()) {

				$response->redirect(); // this will automatically forward the customer. For 3D secure etc.

			} else {

				$this->Transaction->save(array(
					'Transaction' => array(
						'status' => 'failed',
						'message' => $response->getMessage()
					)
				));
				$result = array(
					'message' => $response->getMessage(),
					'status' => 'failed',
					'result' => false
				);
			}

		} catch(Exception $e) {
			$result = array(
				'status' => 'failed',
				'result' => false
			);

			$message = $e->getMessage();

			// Only interested in real responses
			if ($message != 'Invalid response from payment gateway') {
				$this->Transaction->save(array(
					'Transaction' => array(
						'status' => 'failed',
						'message' => $message,
					)
				));

				$result['message'] = $message;
			}

		}

		return $result;
	}

/**
 * Set the parameters of the sage pay transaction and save it to the session.
 * @param int    $transactionId The id of the current transaction.
 * @param array  $return        The return urls for when the transaction has been completed.
 * @param array  $amount        The amount of the transaction.
 * @return array An array containing the transaction data
 */
	protected function _setParams($transactionId, $return, $amount) {
		if (is_array($amount)) {
			$currencyCode = $amount['currency'];
		} else {
			$currencyCode = Configure::read('EvSagePay.currency');
		}

		$params = array(
			'transactionId' => $transactionId . "-" . time(),
			'transactionReference' => $this->_config['transactionReference'],
			'returnUrl' => $return['return'],
			'currency' => $currencyCode,
			'amount' => $amount['amount'],
			'description' => 'Payment to ' . Configure::read('SiteSetting.site_title'),
			'clientIp' => $this->_controller->request->clientIp(),
		);

		if ($this->_controller->Session->read('EvSagePay.SagepayDirect.accountType') == 'M') {
			$params['accountType'] = 'M';
		}

		$this->_controller->Session->write('EvSagePay.params', $params);

		return $params;
	}

/**
 * Read the sage pay transaction parameters from the session. If the session is empty then the parameters
 * will be attempted to be reconstructed from the transaction in the database by reading the transaction
 * id from the current request query parameters.
 * @return array The sage pay parameter array.
 */
	protected function _getParams() {
		$params = $this->_controller->Session->read('EvSagePay.params');

		//If the session params are empty then try to construct it from the available data in the transaction row.
		if (empty($params) && !empty($this->_controller->request->query['transaction'])) {
			$transactionId = $this->_controller->request->query['transaction'];

			//Add the transaction id to the sagepay params
			$params['transactionId'] = $transactionId . "-" . time();

			//Attempt to find the transaction in the database
			$this->_controller->loadModel('EvTransactions.Transaction');
			$transaction = $this->_controller->Transaction->find(
				'first',
				[
					'conditions' => [
						'Transaction.id' => $transactionId
					],
					'contain' => [
						'Currency',
					]
				]
			);

			if (!empty($transaction)) {
				if (!empty($this->_config['transactionReference'])) {
					$params['transactionReference'] = $this->_config['transactionReference'];
				}

				if (!empty($transaction['Transaction']['transaction_amount'])) {
					$params['amount'] = $transaction['Transaction']['transaction_amount'];
				}

				if (!empty($transaction['Currency']['name'])) {
					$params['currency'] = $transaction['Currency']['name'];
				}

				$params['description'] = 'Payment to ' . Configure::read('SiteSetting.site_title');
				$params['clientIp'] = $this->_controller->request->clientIp();
			}
		}

		return $params;
	}
}
