<?php

App::uses('Component', 'Controller');

use Omnipay\Omnipay as Omnipay;

class SagePayOffsiteComponent extends Component {

	protected $_controller = null;

	protected $_api = null;

	protected $_data = null;

	protected $_errorUrl = null;

/**
 * Sets a reference to the calling controller from within the component.
 *
 * @param Controller $controller The controller
 * @return void
 * @see Component::initialize()
 */
	public function initialize(Controller $controller) {
		parent::initialize($controller);

		$this->_controller = $controller;

		$this->_api = Omnipay::create('SagePay_Server');
	}

/**
 * init
 *
 * Use the function setup and connection / settings that need setting up in the gateway
 *
 * @return void
 */
	public function setup() {
		if (strtolower(Configure::read('app.environment')) === 'production') {
			$this->_config = Configure::read('EvSagePay.SagePayServer.live');
		} else {
			$this->_config = Configure::read('EvSagePay.SagePayServer.dev');

			$this->_api->setTestMode(true);
		}
		$this->_api->setVendor($this->_config['vendor']);
	}

/**
 * Split the name
 *
 * @param string $string The full name
 * @return array The split name
 */
	protected function _getName($string) {
		if (strpos($string, ' ') !== false) {
			$name = explode(' ', $string, 2);
			$firstName = $name[0];
			$lastName = $name[1];
		} else {
			$firstName = $string;
			$lastName = ' . ';
		}

		return ['firstName' => $firstName, 'lastName' => $lastName];
	}

/**
 * setupPayment
 *
 * Use this function setup the actual payment, i.e. setup the basket, the amount to take etc...
 *
 * @param int		  $transactionId - transaction id we have created
 * @param array       $return - array with 2 keys of 'return', 'cancel'. Containing either a link or router array
 *                            for redirecting user
 * @param array       $model  - array with two keys of 'model' and 'model_id', used to link transactions
 *                            polymorphically to other model items
 * @param float|array $amount - amount of monies to take (I GOT YOUR MONIESSSSSS), or array of 'amount' and
 *                            'currency' to change currencies (will take default from config)
 * @param array       $items  - multidimenisional array break down of the transaction items, required elements are
 *                            'description' and 'amount'
 * @param mixed       $extra  - variable allowing you to pass ay data needed to the gateway, could be things like
 *                            addresses that are not tracked by the transactions model
 * @return void
 */
	public function setupPayment($transactionId, $return, $model, $amount, $items, $extra = []) {
		$this->Transaction = EvClassRegistry::init('Transactions.Transaction');
		$this->Transaction->id = $transactionId;

		$return['return'] = $return['return'] . '?transaction=' . $transactionId;

		$name = $this->_getName(!empty($extra['name']) ? $extra['name'] : $extra['user']['User']['name']);

		$cardData = [
			'firstName' => $name['firstName'],
			'lastName' => $name['lastName'],
			'email' => $extra['user']['User']['email'],
			'clientIp' => CakeRequest::clientIp(),
			'billingAddress1' => $extra['billing']['Address']['address1'],
			'billingAddress2' => $extra['billing']['Address']['address2'],
			'billingCity' => $extra['billing']['Address']['city'],
			'billingState' => '',
			'billingPostcode' => $extra['billing']['Address']['post_code'],
			'billingCountry' => 'GB',
			'shippingAddress1' => $extra['delivery']['Address']['address1'],
			'shippingAddress2' => $extra['delivery']['Address']['address2'],
			'shippingCity' => $extra['delivery']['Address']['city'],
			'shippingState' => '',
			'shippingPostcode' => $extra['delivery']['Address']['post_code'],
			'shippingCountry' => 'GB',
			'shippingPhone' => '',
			'iframe' => false,
		];

		$params = $this->_setParams(
			$transactionId,
			$return,
			$amount,
			!empty($extra['transactionDescription']) ? $extra['transactionDescription'] : null
		);

		$params['card'] = $cardData;

		// If embedded as an Iframe on a site LOW profile should be used
		if (isset($this->_config['profile'])) {
			$params['profile'] = $this->_config['profile'];
		}

		// Store the customers details on the transaction as we'll need them when they return.
		$this->_data = $this->_api->purchase(
			$this->_modifyParams($params, $transactionId, $return, $model, $amount, $items, $extra)
		);
	}

/**
 * getPayment
 *
 * Everything should be setup, actually take the payment
 *
 * @param int $transactionId transactions id
 *
 * @return mixed dependent on the gateway, value is return straight from the transaction component to user anyway
 */
	public function getPayment($transactionId) {
		$result = $this->_sendRequest($this->_data);

		return [
			'result' => $result,
			'transaction_id' => $transactionId,
		];
	}

/**
 * return
 *
 * deal with a return from the gateway and check for success / fail
 *
 * @return array - with three elements,
 *                 - 'result' = true/false value
 *             - 'message' = text message about transaction (i.e. reason for failing)
 *                 - 'transaction_id' = int of the transaction row
 */
	public function processReturn() {
		$this->Transaction = EvClassRegistry::init('EvTransactions.Transaction');

		if (!empty($this->_controller->request->query['transaction'])) {
			$this->Transaction->id = $this->_controller->request->query['transaction'];
		} else {
			// If we don't have a query parameter then check the session
			$params = $this->_getParams();
			$this->Transaction->id = substr($params['transactionId'], 0, strpos($params['transactionId'], '-'));
		}

		$transactionResult = $this->Transaction->findById($this->Transaction->id);

		if (!empty($transactionResult)) {
			return [
				'result' => $transactionResult['Transaction']['status'] === 'success',
				'message' => $transactionResult['Transaction']['message'],
				'transaction_id' => $transactionResult['Transaction']['id']
			];
		} else {
			return ['result' => false];
		}
	}

/**
 * Process the notification (This is called from the server so has no access to the user's session)
 *
 * @return void
 * @throws Exception If no return URL can be found
 */
	public function processNotification() {
		// Find the transaction id in the url
		$transactionId = $this->_controller->request->query['transaction'];

		if (!empty($transactionId)) {

			$Transaction = EvClassRegistry::init('EvTransactions.Transaction');
			$transaction = $Transaction->findById($transactionId);

			if (!empty($transaction)) {
				// Get the current request
				$request = $this->_api->acceptNotification();

				$request->setTransactionReference($transaction['Transaction']['payment_token']);

				// Prepare a response
				$response = $request->send();

				// Where to send the user after validation
				if (!empty($this->_controller->OrderManager)) {
					$nextUrl = $this->_controller->OrderManager->getCompleteUrl();
				} elseif (!empty(Configure::read('EvSagePay.SagePayServer.completeUrl'))) {
					$nextUrl = Router::url(Configure::read('EvSagePay.SagePayServer.completeUrl'), true);
				} else {
					throw new Exception('No return URL specified');
				}
				$nextUrl .= '?transaction=' . $transactionId;

				if (!$request->isValid()) {
					CakeLog::write('error', 'Request is not valid: ' . $request->getMessage());
					$response->invalid($nextUrl, 'Request is not valid');
				}

				$status = null;

				switch ($request->getTransactionStatus()) {
					case $request::STATUS_COMPLETED:
						$status = 'success';
						break;
					case $request::STATUS_PENDING:
						$status = 'pending';
						break;
					case $request::STATUS_FAILED:
						$status = 'failed';
						$this->_dispatchFailedEvent($transactionId);
						break;
				}

				if (!empty($status)) {
					$result = $Transaction->save([
						'Transaction' => [
							'id' => $transactionId,
							'status' => $status,
							'message' => $request->getMessage(),
							'payment_token' => $response->getTransactionReference(),
						]
					]);

					if (!empty($result)) {
						$response->confirm($nextUrl);
					} else {
						CakeLog::write('error', 'Transaction ' . $transactionId . ' could not be saved: ' . $request->getMessage());
						$response->error($nextUrl, 'Transaction status could not be saved');
					}
				} else {
					CakeLog::write('error', 'Unrecognised transaction status for transaction  ' . $transactionId . ': ' . $request->getMessage());
					$response->error($nextUrl, 'Unrecognised transaction status');
				}
			} else {
				CakeLog::write('error', 'Transaction ' . $transactionId . ' could not be found: ' . $request->getMessage());
				$response->error($nextUrl, 'This transaction does not exist on the system');
			}
		}

		CakeLog::write('error', 'Transaction ' . $transactionId . ' could not be found: ' . $request->getMessage());
		$response->error($nextUrl, 'Invalid transaction ID');
	}

/**
 * Send request
 *
 * @param object $request Request
 * @return array
 */
	protected function _sendRequest($request) {
		$params = $this->_getParams();

		$this->Transaction = EvClassRegistry::init('EvTransactions.Transaction');
		$this->Transaction->id = substr($params['transactionId'], 0, strpos($params['transactionId'], '-'));

		$transactionResult = $this->Transaction->findById($this->Transaction->id);

		//If 3D secure is enabled then the transaction will need to be updated. Otherwise return that it has already been successful.
		if ($transactionResult['Transaction']['status'] == 'success') {
			return ['result' => true];
		}

		try {
			$response = $request->send();

			if ($response->isRedirect()) {
				$this->Transaction->save([
					'Transaction' => [
						'payment_token' => $response->getTransactionReference(),
						'message' => $response->getMessage(),
					],
				]);

				$response->redirect(); // Send the customer to the gateway
			} else {
				$this->Transaction->save([
					'Transaction' => [
						'status' => 'failed',
						'message' => $response->getMessage(),
					],
				]);
				$result = [
					'message' => $response->getMessage(),
					'status' => 'failed',
					'result' => false,
				];

				$this->_dispatchFailedEvent($this->Transaction->id);
			}
		} catch (Exception $e) {
			$result = [
				'status' => 'failed',
				'result' => false,
			];

			$message = $e->getMessage();

			// Only interested in real responses
			if ($message != 'Invalid response from payment gateway') {
				$this->Transaction->save([
					'Transaction' => [
						'status' => 'failed',
						'message' => $message,
					],
				]);

				$this->_dispatchFailedEvent($this->Transaction->id);

				$result['message'] = $message;
			}
		}

		return $result;
	}

/**
 * Set the parameters of the sage pay transaction and save it to the session.
 *
 * @param int   $transactionId The id of the current transaction.
 * @param array $return        The return urls for when the transaction has been completed.
 * @param array $amount        The amount of the transaction.
 * @param string $description  An optional transaction description
 *
 * @return array An array containing the transaction data
 */
	protected function _setParams($transactionId, $return, $amount, $description = null) {
		if (is_array($amount)) {
			$currencyCode = $amount['currency'];
			$amount = $amount['amount'];
		} else {
			$currencyCode = Configure::read('EvSagePay.currency');
		}

		$params = [
			'transactionId' => $transactionId . "-" . time(),
			'transactionReference' => $this->_config['transactionReference'],
			'notifyUrl' => Router::url('/ev_sage_pay/sage_pay_offsite/notification?transaction=' . $transactionId, true),
			'currency' => $currencyCode,
			'amount' => $amount,
			'description' => $description ?: ('Payment to ' . Configure::read('SiteSetting.site_title')),
			'clientIp' => $this->_controller->request->clientIp(),
		];

		$this->_controller->Session->write('EvSagePay.params', $params);

		return $params;
	}

/**
 * Modify the parameters sent with a purchase to sagepay. Allows extended functionality to be added.
 *
 * @param array       $params The current parameters to be sent.
 * @param int         $transactionId - transaction id we have created
 * @param array       $return - array with 2 keys of 'return', 'cancel'. Containing either a link or router array
 *                            for redirecting user
 * @param array       $model  - array with two keys of 'model' and 'model_id', used to link transactions
 *                            polymorphically to other model items
 * @param float|array $amount - amount of monies to take (I GOT YOUR MONIESSSSSS), or array of 'amount' and
 *                            'currency' to change currencies (will take default from config)
 * @param array       $items  - multidimenisional array break down of the transaction items, required elements are
 *                            'description' and 'amount'
 * @param mixed       $extra  - variable allowing you to pass ay data needed to the gateway, could be things like
 *                            addresses that are not tracked by the transactions model
 * @return array The modified parameters.
 */
	protected function _modifyParams($params, $transactionId, $return, $model, $amount, $items, $extra) {
		return $params;
	}

/**
 * Read the sage pay transaction parameters from the session. If the session is empty then the parameters
 * will be attempted to be reconstructed from the transaction in the database by reading the transaction
 * id from the current request query parameters.
 *
 * @return array The sage pay parameter array.
 */
	protected function _getParams() {
		$params = $this->_controller->Session->read('EvSagePay.params');

		//If the session params are empty then try to construct it from the available data in the transaction row.
		if (empty($params) && !empty($this->_controller->request->query['transaction'])) {
			$transactionId = $this->_controller->request->query['transaction'];

			//Add the transaction id to the sagepay params
			$params['transactionId'] = $transactionId . "-" . time();

			//Attempt to find the transaction in the database
			$this->_controller->loadModel('EvTransactions.Transaction');
			$transaction = $this->_controller->Transaction->find(
				'first',
				[
					'conditions' => [
						'Transaction.id' => $transactionId,
					],
					'contain' => [
						'Currency',
					],
				]
			);

			if (!empty($transaction)) {
				if (!empty($this->_config['transactionReference'])) {
					$params['transactionReference'] = $this->_config['transactionReference'];
				}

				if (!empty($transaction['Transaction']['transaction_amount'])) {
					$params['amount'] = $transaction['Transaction']['transaction_amount'];
				}

				if (!empty($transaction['Currency']['name'])) {
					$params['currency'] = $transaction['Currency']['name'];
				}

				$params['description'] = 'Payment to ' . Configure::read('SiteSetting.site_title');
				$params['clientIp'] = $this->_controller->request->clientIp();
			}
		}

		return $params;
	}

/**
 * Dispatch an event so an order can be marked as failed
 *
 * @param int $transactionId The transaction Id
 * @return CakeEvent event
 */
	protected function _dispatchFailedEvent($transactionId) {
		return $this->_controller->getEventManager()->dispatch(new CakeEvent('EvSagePay.Component.SagePayOffsite.transactionFailed', $this, [
			'transactionId' => $transactionId,
		]));
	}
}
