<?php
App::uses('RoutableAppModel', 'Routable.Model');
App::uses('CakeTime', 'Utility');

/**
 * Dynamic Routes model
 */
class Route extends RoutableAppModel {

/**
 * Default display field
 */
	public $displayField = "alias";

/**
 * Default validation rules
 */
	public $validate = array(
		'alias' => array(
			'unique' => array(
				'rule' => 'isUnique',
				'message' => 'must be unique',
				'allowEmpty' => true
			)
		)
	);

/**
 * Update the dynamic route cache
 *
 * @param bool $created True if this save created a new record
 * @param array $options Options passed from Model::save()
 * @return void
 */
	public function afterSave($created, $options = array()) {
		// Add Rebuild job to the queue
		$QueuedTask = EvClassRegistry::init('Queue.QueuedTask');

		$jobExists = $QueuedTask->find(
			'first',
			[
				'conditions' => [
					'QueuedTask.jobtype' => 'Route',
					'QueuedTask.fetched' => null
				]
			]
		);

		if (empty($jobExists)) {
			$QueuedTask->createJob(
				'Route'
			);
		}

		// Clear any duplicates so it doesn't run rebuild twice
		// $QueuedTask->clearDoublettes();

		// Generate the temporary cache of routes whilst we wait for the queue to run.
		$this->updateDynamicRoutesCache(true);

		return parent::afterSave($created, $options);
	}

/**
 * Recreate the dynamic routes cache file. The routes cache will be generated by the queue to avoid
 * slow build times affecting the user, so we can also generate a temporary cache containing just
 * the routes yet to be built until the queue runs the relevant task. The temporary cache needs
 * creating after save.
 *
 * @param bool $temporary Pass true to generate a temporary cache.
 * @return bool.
 */
	public function updateDynamicRoutesCache($temporary = false) {
		//Fire before event
		$beforeEvent = new CakeEvent(
			'Routable.Model.Route.updateDynamicRoutesCache.before',
			$this,
			[
				'temporary' => $temporary
			]
		);
		$this->getEventManager()->dispatch($beforeEvent);

		//Something has failed the before event so fail here.
		if (
			$beforeEvent->isStopped()
			|| (
				isset($beforeEvent->result['result'])
				&& $beforeEvent->result['result'] === false
			)
		) {
			return false;
		}

		$routes = $this->findForCache($temporary);

		$cache = "<?php\n";

		if ($temporary) {
			$routeFile = TMP . 'temp_dynamic_routes.php';
		} else {
			$routeFile = TMP . 'pre_dynamic_routes.php';
		}

		$fh = fopen($routeFile, 'w');
		fwrite($fh, $cache);
		fclose($fh);

		ini_set('memory_limit', '256M');

		foreach ($routes as $route) {
			//Reset the cache
			$cache = '';

			$alias = $route['Route']['alias'];

			// Check that the tokens have been removed from the alias before proceeding
			// (this shouldn't happen, but we want to catch it if it does).
			if (strrpos($alias, ':') === false) {
				$actual = Router::parse($route['Route']['actual']);
				$cache .= "Router::connect('/" . $alias . "', array(";

				foreach ($actual as $key => $value) {
					if (!empty($value)) {
						if (is_array($value)) {
							foreach ($value as $key1 => $param) {
								//handles pass variable
								if (is_numeric($key1)) {
									$cache .= "'" . $param . "',";
								} else { // Handles Named params
									$cache .= "'" . $key1 . "'=>'" . $param . "',";
								}
							}
						} else {
							$cache .= "'" . $key . "'=>'" . $value . "',";
						}
					}
				}
				$cache = substr($cache, 0, -1) . "));\n";
			}

			$fh = fopen($routeFile, 'a');
			fwrite($fh, $cache);
			fclose($fh);

		}

		if ($temporary === false) {
			// Rather than renaming we will write into the existing file
			// so that symlinks remain preserved
			$dymanicRoutesString = file_get_contents(TMP . 'pre_dynamic_routes.php');
			if ($dymanicRoutesString !== false) {
				// Update contents of dynamic routes file
				$fh = fopen(TMP . 'dynamic_routes.php', 'w');
				fwrite($fh, $dymanicRoutesString);
				fclose($fh);

				// Going to be a big string so release for garbage collection
				unset($dymanicRoutesString);

				// Remove the temporary file
				unlink(TMP . 'pre_dynamic_routes.php');
			}

			// Temporary routes are now in the main routes file so we can clear it out
			$this->clearTemporaryCache();
		}

		//Fire after event
		$afterEvent = new CakeEvent(
			'Routable.Model.Route.updateDynamicRoutesCache.after',
			$this,
			[
				'temporary' => $temporary
			]
		);
		$afterEvent->passParams = true;
		$this->getEventManager()->dispatch($afterEvent);

		//Something has failed the after event so fail here.
		if (
			$afterEvent->isStopped()
			|| (
				isset($afterEvent->result['result'])
				&& $afterEvent->result['result'] === false
			)
		) {
			return false;
		}

		return true;
	}

/**
 * Clear the temporary cache, this wants to be called whenever we update the primary cache file.
 *
 * @return void
 */
	public function clearTemporaryCache() {
		$fh = fopen(TMP . 'temp_dynamic_routes.php', 'w');
		fwrite($fh, "<?php\n");
		fclose($fh);

		//Fire clear cache event
		$clearCacheEvent = new CakeEvent(
			'Routable.Model.Route.clearTemporaryCache',
			$this
		);
		$this->getEventManager()->dispatch($clearCacheEvent);
	}

/**
 * Find routes for a cache file.
 *
 * @param bool $temporary Pass true to only fetch routes for the temporary cache
 * @return array Routes
 */
	public function findForCache($temporary) {
		$conditions = [];
		if ($temporary === true) {
			// If we're generating a temporary cache we need to determine when we last created the
			// dynamic routes file and fetch all modified routes since that time.
			$queuedTask = EvClassRegistry::init('Queue.QueuedTask')->find('first', [
				'conditions' => [
					'QueuedTask.jobtype' => 'Route',
					'QueuedTask.completed <>' => null
				],
				'order' => 'QueuedTask.created DESC'
			]);
			if (!empty($queuedTask)) {
				$conditions['Route.modified >'] = $queuedTask['QueuedTask']['created'];
			} else {
				$conditions['Route.modified >'] = CakeTime::format("-24 hours", "%G-%m-%d %H:%M:%S");
			}
		}

		return $this->find('all', compact('conditions'));
	}
}
