<?php
/**
 * Dynamic Routing Behavior
 */
App::uses('AppBehavior', 'Model/Behavior');
App::uses('Route', 'Routable.Model');

class RoutableBehavior extends AppBehavior {

/**
 * Initialise the behavior - store the settings against the model
 *
 * @see ModelBehavior::setup()
 */
	public function setup(Model $Model, $settings = array()) {
		$defaults = array(
			'actual' => ':controller/:action/:primaryKey',
			'alias' => ':controller/:displayField',
			'action' => 'view',
			'protectPrefix' => true
		);

		if (isset($settings['config']) && ! empty($settings['config'])) {

			$route = Configure::read($settings['config'] . '.route');

			// check to see if multiple routes based on models
			if (isset($route[$Model->alias])) {

				$routeSettings = $route[$Model->alias];
			} elseif (
				isset($route['controller']) &&
				isset($route['action']) &&
				isset($route['plugin'])
			) {
				// or standard for just one model
				$routeSettings = $route;
			}

			// check to make sure we found what we want
			if (isset($routeSettings)) {

				// plugin split the controller and replace the plugin / controller fields
				list($plugin, $controller) = pluginSplit($routeSettings['controller']);
				$routeSettings['plugin'] = $plugin;
				$routeSettings['controller'] = $controller;

				// TODO: review, this seems like a dirty hack
				$routeSettings['admin'] = false;

				$defaults['actual'] = Router::url($routeSettings) . '/:primaryKey';
			}
		}

		$this->settings[$Model->alias] = array_merge(
			$defaults,
			$settings
		);
	}

/**
 * Used by _adminFormFields when getting routable route-alias
 * Calculate the routing prefix and return if protected, turn protection off to let
 * User edit whole string
 *
 * @return 	string|null 	String containing the prefix or null if not protected
 */
	public function getUrlPrefix(Model $Model) {
		if (
			isset($this->settings[$Model->alias]['protectPrefix']) &&
			$this->settings[$Model->alias]['protectPrefix']
		) {

			$urlPrefix = str_replace(':displayField', '', $this->settings[$Model->alias]['alias']);
			$urlPrefix = trim($urlPrefix, '/*');
			if ($urlPrefix === ':controller') {

				$urlPrefix = InflectorExt::underscore(InflectorExt::pluralize($Model->alias));
			}

			$urlPrefix = preg_replace("/:([0-9A-Za-z_-]+)/", '', $urlPrefix);

			return trim($urlPrefix, '/*');
		}

		return null;
	}

/**
 * Check uniquness of alias. Generate one from template if not set
 *
 * @param Model $Model Primary model
 * @param array $options Options passed from Model::save()
 * @return bool True if the operation should continue, false if it should abort
 * @see Model::save()
 */
	public function beforeSave(Model $Model, $options = array()) {
		$Route = EvClassRegistry::init('Routable.Route');

		// If we're saving deep associated data we need to restore the route
		// in $Model->data into the expected format as the CakePHP core validation
		// methods will have re-arranged the data.
		if (isset($Model->data[$Model->alias]['Route'])) {

			$Model->data['Route'] = $Model->data[$Model->alias]['Route'];
			unset($Model->data[$Model->alias]['Route']);

		}

		if (empty($Model->data['Route']['alias']) && empty($Model->data['Route']['id'])) {

			// create a unique alias
			$baseAlias = $Model->alias();

			$suffix = 0;

			do {
				$suffix++;

				$alias = $baseAlias;
				if ($suffix > 1) {
					$alias = $Model->alias($suffix);
				}

				$found = $Route->findByAlias($alias);

			} while ($found);

			$Model->data['Route']['alias'] = $alias;

		}

		// validate the route
		$Route->set($Model->data);
		return $Route->validates();
	}

/**
 * Primary Model has saved, store the route info
 *
 * @param Model $Model Primary model
 * @param bool $created True if this save created a new record
 * @param array $options Options passed from Model::save()
 * @return void
 * @see ModelBehavior::afterSave()
 */
	public function afterSave(Model $Model, $created, $options = array()) {
		$data = $Model->data;

		$data['Route']['actual'] = $this->actual($Model, array($Model->alias => $Model->data[$Model->alias]));

		$Route = EvClassRegistry::init('Routable.Route');

		// If we are creating a new route then we want to check if we need to
		// setup a redirect for the original route alias.
		if (!empty($data['Route']['id'])) {

			$existingRoute = $Route->findById($Model->data['Route']['id']);

			if ($existingRoute['Route']['alias'] != $data['Route']['alias']) {
				$Redirect = ClassRegistry::init('Routable.Redirect');
				$redirectData = array(
					'source' => '/' . $existingRoute['Route']['alias'],
					'model' => $Model->alias,
					'model_id' => $Model->id,
					'action' => $this->settings[$Model->alias]['action']
				);
				$Redirect->create();
				$Redirect->save($redirectData);

			}
		}

		// Create/Update the route.

		// @TODO: There's a potential race condition here, someone else may
		// have just inserted the same route (unlikely). In this scenariom the
		// $Model will have saved, but the route won't.
		$Route->create();
		$Route->save($data);
	}

/**
 * get route for a given item. Used on readForEdit only
 * to prevent tons of queries happen
 *
 * @param Object 	Model Object behaviour was called from
 * @param array 	array of data we want to get the route for
 * @return array 	Either array of route data or blank array
 */
	public function getRoute(Model $Model, $data) {
		$Route = EvClassRegistry::init('Routable.Route');

		$route = $Route->findByActual($this->actual($Model, $data));

		if ($route) {

			return $route['Route'];
		}

		return array();
	}

/**
 * Delete the route associated with Primary Model
 *
 * @see ModelBehavior::beforeDelete()
 */
	public function beforeDelete(Model $Model, $cascade = true) {
		$data = $Model->findById($Model->id);

		if (! $Model->hasField('is_protected') || ! $data[$Model->alias]['is_protected']) {
			$Route = EvClassRegistry::init('Routable.Route');
			$Redirect = EvClassRegistry::init('Routable.Redirect');

			$routeData = $Route->findByActual($this->actual($Model, $data));

			if (isset($routeData['Route']['id']) && ! empty($routeData['Route']['id'])) {

				$Route->delete($routeData['Route']['id']);
				$Redirect->deleteAll(
					array(
						'model' => $Model->alias,
						'action' => $this->settings[$Model->alias]['action'],
						'model_id' => $Model->id
					)
				);
			}

			return true;

		} else {

			return false;

		}
	}

/**
 * Replaces tokens and fields in a path template
 *
 * @param Model $Model
 * @param unknown_type $path
 * @param unknown_type $data
 */
	protected function _replacePathTokens(Model $Model, $path, $data = null, $suffix = null) {
		if ($data == null) {
			$data = $Model->data;
		}

		// replace tokens
		// plugin path, check for settings, if not try and work it out
		if (isset($this->settings[$Model->alias]['plugin'])) {

			$pluginPath = $this->settings[$Model->alias]['plugin'];
		} else {

			$pluginPath = Inflector::underscore($Model->plugin);
		}

		// controller, check for settings, if not try and work it out
		if (isset($this->settings[$Model->alias]['controller'])) {

			$controllerPath = $this->settings[$Model->alias]['controller'];
		} else {

			$controllerPath = Inflector::underscore(Inflector::pluralize($Model->alias));
		}

		$path = str_replace(':controller', $controllerPath, $path);
		$path = str_replace(':plugin', $pluginPath, $path);
		$path = str_replace(':action', $this->settings[$Model->alias]['action'], $path);
		$path = str_replace(':primaryKey', ":" . $Model->primaryKey, $path);
		$path = str_replace(':displayField', ":" . $Model->displayField, $path);

		$path = $this->_replaceCustomTokens($path, $data[$Model->alias], $suffix);
		$path = $this->_replaceCustomTokens($path, $data[$Model->alias], $suffix, '-');
		return $path;
	}

/**
 * Constructs the alias from settings template
 *
 * @param Object $Model
 */
	public function alias(Model $Model, $suffix = null) {
		return $this->_replacePathTokens($Model, $this->settings[$Model->alias]['alias'], null, $suffix);
	}

/**
 * Constructs the actual path from settings
 *
 * @param Object $Model
 * @param unknown_type $item
 */
	public function actual(Model $Model, $item) {
		return $this->_replacePathTokens($Model, $this->settings[$Model->alias]['actual'], $item);
	}

/**
 * Replaces tokens from a given $path
 * @param  string $path       A path to be used at the haystack
 * @param  Model  $data       A $Model->data[$Model->alias]
 * @param  string $suffix     Any string after the $path
 * @param  string $delimiter  Character used to denote the end of a token
 * @return array              $path with repalced tokens
 */
	protected function _replaceCustomTokens($path, $data, $suffix, $delimiter = '/') {
		// replace field values if needed
		$pathParts = preg_split('/\\' . $delimiter . '/', $path);

		foreach ($pathParts as $id => $part) {
			$matches = array();

			if (preg_match("/:([0-9A-Za-z_]+)/", $part, $matches)) {
				//If column exists then switch out

				if (isset($data[$matches[1]])) {
					$partData = strtolower(Inflector::slug($data[$matches[1]], '-'));
					if ($suffix !== null) {
						$partData .= '-' . $suffix;
					}

					$pathParts[$id] = str_replace($matches[0], $partData, $pathParts[$id]);
				}
			}
		}

		return join($delimiter, $pathParts);
	}
}
