<?php

/**
 * Dynamic Routing Behavior
 */

App::uses('Route', 'Routable.Model');

class RoutableBehavior extends ModelBehavior {

/**
 * Initialise the behavior - store the settings against the model
 *
 * @see ModelBehavior::setup()
 */
	public function setup(Model $Model, $settings = array()) {

		$defaults = array(
			'actual' => ':controller/:action/:primaryKey',
			'alias' => ':controller/:displayField',
			'action' => 'view'
		);

		if (isset($settings['config']) && ! empty($settings['config'])) {

			$route = Configure::read($settings['config'] . '.route');

			// check to see if multiple routes based on models
			if (isset($route[$Model->alias])) {

				$routeSettings = $route[$Model->alias];
			} elseif (
				isset($route['controller']) &&
				isset($route['action']) &&
				isset($route['plugin'])
			) {
				// or standard for just one model
				$routeSettings = $route;
			}

			// check to make sure we found what we want
			if (isset($routeSettings)) {

				// plugin split the controller and replace the plugin / controller fields
				list($plugin, $controller) = pluginSplit($routeSettings['controller']);
				$routeSettings['plugin'] = $plugin;
				$routeSettings['controller'] = $controller;

				// TODO: review, this seems like a dirty hack
				$routeSettings['admin'] = false;

				$defaults['actual'] = Router::url($routeSettings) . '/:primaryKey';
			}
		}

		$this->settings[$Model->alias] = array_merge(
			$defaults,
			$settings
		);
	}


/**
 * Check uniquness of alias. Generate one from template if not set
 *
 * @see ModelBehavior::beforeSave()
 */
	public function beforeSave(Model $Model, $options=array()) {

		$Route = new Route();

		// If we're saving deep associated data we need to restore the route
		// in $Model->data into the expected format as the CakePHP core validation
		// methods will have re-arranged the data.
		if (isset($Model->data[$Model->alias]['Route'])) {

			$Model->data['Route'] = $Model->data[$Model->alias]['Route'];
			unset($Model->data[$Model->alias]['Route']);

		}

		if (empty($Model->data['Route']['alias']) && empty($Model->data['Route']['id'])) {

			// create a unique alias
			$baseAlias = $this->_alias($Model);

			$suffix = 0;

			do {

				$suffix++;

				$alias = ($suffix == 1) ? $baseAlias : $baseAlias . "_" . $suffix;

				$found = $Route->findByAlias($alias);

			} while($found);

			$Model->data['Route']['alias'] = $alias;

		}

		// validate the route
		$Route->set($Model->data);
		return $Route->validates();
	}


/**
 * Primary Model has saved, store the route info
 *
 * @see ModelBehavior::afterSave()
 */
	public function afterSave(Model $Model, $created, $options=array()) {

		$data = $Model->data;

		$data['Route']['actual'] = $this->_actual($Model, array($Model->alias => $Model->data[$Model->alias]));

		$Route = new Route();

		// If we are creating a new route then we want to check if we need to
		// setup a redirect for the original route alias.
		if (!empty($data['Route']['id'])) {

			$existingRoute = $Route->findById($Model->data['Route']['id']);

			if ($existingRoute['Route']['alias']!=$data['Route']['alias']) {

				$Redirect = ClassRegistry::init('Routable.Redirect');
				$redirectData = array(
					'source' => '/' . $existingRoute['Route']['alias'],
					'model' => $Model->alias,
					'model_id' => $Model->id,
					'action' => $this->settings[$Model->alias]['action']
				);
				$Redirect->create();
				$Redirect->save($redirectData);

			}
		}

		// Create/Update the route.

		// @TODO: There's a potential race condition here, someone else may
		// have just inserted the same route (unlikely). In this scenariom the
		// $Model will have saved, but the route won't.
		$Route->create();
		$Route->save($data);

		return;
	}


	/**
	 * get route for a given item. Used on readForEdit only
	 * to prevent tons of queries happen
	 *
	 * @param Object 	Model Object behaviour was called from
	 * @param array 	array of data we want to get the route for
	 * @return array 	Either array of route data or blank array
	 */
	public function getRoute(Model $Model, $data)
	{
		$Route = ClassRegistry::init('Routable.Route');

		$route = $Route->findByActual($this->_actual($Model, $data));

		if ($route) {

			return $route['Route'];
		}

		return array();
	}

/**
 * Delete the route associated with Primary Model
 *
 * @see ModelBehavior::beforeDelete()
 */
	public function beforeDelete(Model $Model, $cascade = true) {
		$data = $Model->findById($Model->id);

		if (! $Model->hasField('is_protected') || ! $data[$Model->alias]['is_protected']) {
			$Route = ClassRegistry::init('Routable.Route');
			$Redirect = ClassRegistry::init('Routable.Redirect');
			$routeData = $Route->findByActual($this->_actual($Model, $data));

			if (isset($routeData['Route']['id']) && ! empty($routeData['Route']['id'])) {

				$Route->delete($routeData['Route']['id']);
				$Redirect->deleteAll(
					array(
						'model' => $Model->alias,
						'action' => $this->settings[$Model->alias]['action'],
						'model_id' => $Model->id
					)
				);
			}

			return true;

		} else {

			return false;

		}
	}

/**
 * Replaces tokens and fields in a path template
 *
 * @param Model $Model
 * @param unknown_type $path
 * @param unknown_type $data
 */
	protected function _replacePathTokens(Model $Model, $path, $data = null) {

		if ($data == null) {

			$data = $Model->data;
		}

		// replace tokens
		// plugin path, check for settings, if not try and work it out
		if (isset($this->settings[$Model->alias]['plugin'])) {

			$pluginPath = $this->settings[$Model->alias]['plugin'];
		} else {

			$pluginPath = Inflector::underscore($Model->plugin);
		}

		// controller, check for settings, if not try and work it out
		if (isset($this->settings[$Model->alias]['controller'])) {

			$controllerPath = $this->settings[$Model->alias]['controller'];
		} else {

			$controllerPath = Inflector::underscore(Inflector::pluralize($Model->alias));
		}

		$path = str_replace(':controller', $controllerPath, $path);
		$path = str_replace(':plugin', $pluginPath, $path);
		$path = str_replace(':action', $this->settings[$Model->alias]['action'], $path);
		$path = str_replace(':primaryKey', ":" . $Model->primaryKey, $path);
		$path = str_replace(':displayField', ":" . $Model->displayField, $path);

		// replace field values if needed

		$pathParts = preg_split('/\//', $path);

		foreach ($pathParts as $id=>$part) {

			$matches = array();
			if(preg_match("/:([0-9A-Za-z_-]+)/", $part, $matches)) {

				//If column exists then switch out
				if(isset($data[$Model->alias][$matches[1]])) {

					$pathParts[$id] = str_replace($matches[0], strtolower(Inflector::slug($data[$Model->alias][$matches[1]], '-')), $pathParts[$id]);

				}
			}
		}

		$path = join('/', $pathParts);

		return $path;
	}


/**
 * Constructs the alias from settings template
 *
 * @param Object $Model
 */
	protected function _alias(Model $Model) {

		return $this->_replacePathTokens($Model, $this->settings[$Model->alias]['alias']);
	}


/**
 * Constructs the actual path from settings
 *
 * @param Object $Model
 * @param unknown_type $item
 */
	protected function _actual(Model $Model, $item) {

		return $this->_replacePathTokens($Model, $this->settings[$Model->alias]['actual'], $item);
	}
}
