<?php

/**
 * Dynamic Routing Behavior
 */
App::uses('AppBehavior', 'Model/Behavior');
App::uses('Route', 'Routable.Model');
App::uses('CakeSession', 'Model/Datasource');

class RoutableBehavior extends AppBehavior {

/**
 * Initialise the behavior - store the settings against the model
 *
 * @see ModelBehavior::setup()
 */
	public function setup(Model $Model, $settings = array()) {
		$defaults = array(
			'actual' => ':controller/:action/:primaryKey',
			'alias' => ':controller/:displayField',
			'action' => 'view',
			'protectPrefix' => true,
			'sharedAlias' => false // Custom for the ArbSites system, allows a route to be shared across all linked sites
		);

		if (isset($settings['config']) && ! empty($settings['config'])) {

			$route = Configure::read($settings['config'] . '.route');

			// check to see if multiple routes based on models
			if (isset($route[$Model->alias])) {

				$routeSettings = $route[$Model->alias];
			} elseif (
				isset($route['controller']) &&
				isset($route['action']) &&
				isset($route['plugin'])
			) {
				// or standard for just one model
				$routeSettings = $route;
			}

			// check to make sure we found what we want
			if (isset($routeSettings)) {

				// plugin split the controller and replace the plugin / controller fields
				list($plugin, $controller) = pluginSplit($routeSettings['controller']);
				$routeSettings['plugin'] = $plugin;
				$routeSettings['controller'] = $controller;

				// TODO: review, this seems like a dirty hack
				$routeSettings['admin'] = false;

				$defaults['actual'] = Router::url($routeSettings) . '/:primaryKey';
			}
		}

		$this->settings[$Model->alias] = array_merge(
			$defaults,
			$settings
		);
	}

	/**
	 * Used by _adminFormFields when getting routable route-alias
	 * Calculate the routing prefix and return if protected, turn protection off to let
	 * User edit whole string
	 *
	 * @return 	string|null 	String containing the prefix or null if not protected
	 */
	public function getUrlPrefix(Model $Model) {
		if (
			isset($this->settings[$Model->alias]['protectPrefix']) &&
			$this->settings[$Model->alias]['protectPrefix']
		) {

			$urlPrefix = str_replace(':displayField', '', $this->settings[$Model->alias]['alias']);
			$urlPrefix = trim($urlPrefix, '/*');
			if ($urlPrefix === ':controller') {

				$urlPrefix = InflectorExt::underscore(InflectorExt::pluralize($Model->alias));
			}

			return $urlPrefix;
		}

		return null;
	}

	/**
	 * Check uniquness of alias. Generate one from template if not set
	 *
	 * @see ModelBehavior::beforeSave()
	 */
	public function beforeSave(Model $Model, $options = array()) {
		$Route = new Route();

		// by defaulting the site to null, models without a site_id field will have their routes generated in the shared routes file
		$siteId = null;

		// If we're saving deep associated data we need to restore the route
		// in $Model->data into the expected format as the CakePHP core validation
		// methods will have re-arranged the data.

		if (isset($Model->data[$Model->alias]['Route'])) {
			$Model->data['Route'] = $Model->data[$Model->alias]['Route'];
			unset($Model->data[$Model->alias]['Route']);
		}

		if (isset($Model->data[$Model->alias]['site_id'])) {
			$siteId = $Model->data[$Model->alias]['site_id'];
			$Model->data['Route']['site_id'] = $siteId;
		}

		if (empty($Model->data['Route']['alias']) && empty($Model->data['Route']['id'])) {

			// create a unique alias
			$baseAlias = $this->_alias($Model);

			$suffix = 0;

			do {
				$suffix++;

				$alias = $baseAlias;
				if ($suffix > 1) {
					$alias = $this->_alias($Model, $suffix);
				}

				$found = $Route->find('first', array(
					'conditions' => array(
						'alias' => $alias,
						'site_id' => $siteId
					)
				));

			} while ($found);

			$Model->data['Route']['alias'] = $alias;
			$Model->data['Route']['site_id'] = $siteId;

			if ($this->settings[$Model->alias]['sharedAlias']) {
				$Model->data['Route']['site_id'] = 0;
			}

		}

		// validate the route
		$Route->set($Model->data);
		return $Route->validates();
	}

	/**
	 * Primary Model has saved, store the route info
	 *
	 * @see ModelBehavior::afterSave()
	 */
	public function afterSave(Model $Model, $created, $options = array()) {
		$data = $Model->data;

		// by defaulting the site to null, models without a site_id field will have their routes generated in the shared routes file
		$siteId = null;

		if (isset($Model->data[$Model->alias]['site_id'])) {
			$siteId = $Model->data[$Model->alias]['site_id'];
		}

		$data['Route']['actual'] = $this->_actual($Model, array($Model->alias => $Model->data[$Model->alias]));

		$Route = new Route();

		// If we are creating a new route then we want to check if we need to
		// setup a redirect for the original route alias.
		if (!empty($data['Route']['id'])) {

			$existingRoute = $Route->findById($Model->data['Route']['id']);

			if ($existingRoute['Route']['alias'] != $data['Route']['alias']) {
				$Redirect = ClassRegistry::init('Routable.Redirect');
				$redirectData = array(
					'source' => '/' . $existingRoute['Route']['alias'],
					'model' => $Model->alias,
					'model_id' => $Model->id,
					'action' => $this->settings[$Model->alias]['action'],
					'site_id' => $siteId
				);
				$Redirect->create();
				$Redirect->save($redirectData);

			}
		}

		// Create/Update the route.

		// @TODO: There's a potential race condition here, someone else may
		// have just inserted the same route (unlikely). In this scenariom the
		// $Model will have saved, but the route won't.
		$Route->create();
		$Route->save($data);

		return;
	}

	/**
	 * get route for a given item. Used on readForEdit only
	 * to prevent tons of queries happen
	 *
	 * @param Object 	Model Object behaviour was called from
	 * @param array 	array of data we want to get the route for
	 * @return array 	Either array of route data or blank array
	 */
	public function getRoute(Model $Model, $data) {
		$Route = EvClassRegistry::init('Routable.Route');

		$actual = $this->_actual($Model, $data);
		$siteId = CakeSession::read('siteId');

		if (isset($data[$Model->alias]['site_id'])) {
			$siteId = $data[$Model->alias]['site_id'];
		}

		$route = $Route->getMultisiteRoute($actual, $siteId);

		if ($route) {

			return $route['Route'];
		}

		return array();
	}

	/**
	 * Delete the route associated with Primary Model
	 *
	 * @see ModelBehavior::beforeDelete()
	 */
	public function beforeDelete(Model $Model, $cascade = true) {
		$data = $Model->findById($Model->id);

		if (! $Model->hasField('is_protected') || ! $data[$Model->alias]['is_protected']) {
			$Route = EvClassRegistry::init('Routable.Route');
			$Redirect = EvClassRegistry::init('Routable.Redirect');

			$actual = $this->_actual($Model, $data);
			$siteId = CakeSession::read('siteId');

			$routeData = $Route->getMultisiteRoute($actual, $siteId);

			if (isset($routeData['Route']['id']) && ! empty($routeData['Route']['id'])) {

				$Route->delete($routeData['Route']['id']);
				$Redirect->deleteAll(
					array(
						'model' => $Model->alias,
						'action' => $this->settings[$Model->alias]['action'],
						'model_id' => $Model->id,
						'OR' => array(
							'site_id' => $siteId,
							'site_id IS NULL'
						)
					)
				);
			}

			return true;

		} else {

			return false;

		}
	}

	/**
	 * Replaces tokens and fields in a path template
	 *
	 * @param Model $Model
	 * @param unknown_type $path
	 * @param unknown_type $data
	 */
	protected function _replacePathTokens(Model $Model, $path, $data = null, $suffix = null) {
		if ($data == null) {

			$data = $Model->data;
		}

		// replace tokens
		// plugin path, check for settings, if not try and work it out
		if (isset($this->settings[$Model->alias]['plugin'])) {

			$pluginPath = $this->settings[$Model->alias]['plugin'];
		} else {

			$pluginPath = Inflector::underscore($Model->plugin);
		}

		// controller, check for settings, if not try and work it out
		if (isset($this->settings[$Model->alias]['controller'])) {

			$controllerPath = $this->settings[$Model->alias]['controller'];
		} else {

			$controllerPath = Inflector::underscore(Inflector::pluralize($Model->alias));
		}

		$path = str_replace(':controller', $controllerPath, $path);
		$path = str_replace(':plugin', $pluginPath, $path);
		$path = str_replace(':action', $this->settings[$Model->alias]['action'], $path);
		$path = str_replace(':primaryKey', ":" . $Model->primaryKey, $path);
		$path = str_replace(':displayField', ":" . $Model->displayField, $path);

		// replace field values if needed

		$pathParts = preg_split('/\//', $path);

		foreach ($pathParts as $id => $part) {
			$matches = array();

			if (preg_match("/:([0-9A-Za-z_-]+)/", $part, $matches)) {
				//If column exists then switch out

				if (isset($data[$Model->alias][$matches[1]])) {
					$partData = strtolower(Inflector::slug($data[$Model->alias][$matches[1]], '-'));
					if ($suffix !== null) {
						$partData .= '-' . $suffix;
					}

					$pathParts[$id] = str_replace($matches[0], $partData, $pathParts[$id]);
				}
			}
		}

		$path = join('/', $pathParts);

		return $path;
	}

	/**
	 * Constructs the alias from settings template
	 *
	 * @param Object $Model
	 */
	protected function _alias(Model $Model, $suffix = null) {
		return $this->_replacePathTokens($Model, $this->settings[$Model->alias]['alias'], null, $suffix);
	}

	/**
	 * Constructs the actual path from settings
	 *
	 * @param Object $Model
	 * @param unknown_type $item
	 */
	protected function _actual(Model $Model, $item) {
		return $this->_replacePathTokens($Model, $this->settings[$Model->alias]['actual'], $item);
	}
}
