<?php

App::uses('ReviewListener','EvReview.Lib/Event');

class Review extends EvReviewAppModel {

	public $order = array(
		'Review.created' => 'DESC'
	);

	public $validate = array(
		'name' => array(
			'required' => array(
				'rule' => 'notEmpty',
				'message' => 'Required'
			),
			'maxLength' => array(
				'rule' => array('maxLength', 45),
				'message' => 'No more than 45 characters long'
			)
		),
		'email' => array(
			'required' => array(
				'rule' => 'notEmpty',
				'message' => 'Required'
			),
			'maxLength' => array(
				'rule' => array('maxLength', 254),
				'message' => 'No more than 254 characters long'
			),
			'email' => array(
				'rule' => 'email',
				'message' => 'Valid email addresses only'
			)
		),
		'location' => array(
			'required' => array(
				'rule' => 'notEmpty',
				'message' => 'Required'
			),
			'maxLength' => array(
				'rule' => array('maxLength', 45),
				'message' => 'No more than 45 characters long'
			)
		),
		'title' => array(
			'required' => array(
				'rule' => 'notEmpty',
				'message' => 'Required'
			),
			'maxLength' => array(
				'rule' => array('maxLength', 45),
				'message' => 'No more than 45 characters long'
			)
		),
		'review' => array(
			'required' => array(
				'rule' => 'notEmpty',
				'message' => 'Required'
			)
		)
	);


	public $findMethods = array('average' =>  true);

	
	public function __construct($id = false, $table = null, $ds = null) {

		parent::__construct($id, $table, $ds);

		// Dynamically build associations from the plugin's config file.
		$relations = Configure::read('EvReview.review_relation');
		if (!empty($relations)) {

			foreach ($relations as $model => $settings) {

				if (is_numeric($model)) {
					$model = $settings;
					$settings = [];
				}

				$settings['dependent'] = true;

				$this->belongsTo[$model] = $settings; 

			}

		}

		return;

	}


/**
 * Before save logic.
 * 
 * @param array $options
 * @return boolean
 */
	public function beforeValidate($options = []) {

		$response = parent::beforeValidate($options);

		// Check that the model/model_id have been set before saving and throw an error if not.
		if ((empty($this->data[$this->alias]['model']) || empty($this->data[$this->alias]['model_id'])) && empty($this->id)) {
			$this->log('EvReview unable to save review as model/model_id not set.');
			throw new InternalErrorException();
		}

		return $response;

	}


/**
 * Before save logic.
 * 
 * @param array $options
 * @return boolean
 */
	public function beforeSave($options = []) {

		$response = parent::beforeSave($options);

		// Save the reviewer's IP address.
		if (empty($this->data[$this->alias]['id']) && empty($this->id) && !empty($_SERVER['REMOTE_ADDR'])) {
			$this->data[$this->alias]['ip_address'] = $_SERVER['REMOTE_ADDR'];
		}

		if (!empty($this->data[$this->alias]['model']) && !empty($this->data[$this->alias]['model_id']) && empty($data[$this->alias]['reviewed_item_name'])) {

			$model = '';
			$model .= !empty($this->data[$this->alias]['plugin']) ? $this->data[$this->alias]['plugin'] . '.' : '';
			$model .= $this->data[$this->alias]['model'];

			$ReviewedItem = ClassRegistry::init($model);

			$ReviewedItem->id = $this->data[$this->alias]['model_id'];
			$this->data[$this->alias]['reviewed_item_name'] = $ReviewedItem->field($ReviewedItem->displayField);

		}

		return $response;

	}


/**
 * After save logic.
 * 
 * @param boolean $created
 * @param array $options
 * @return void
 */
	public function afterSave($created, $options = []) {

		if ($created===true) {

			$this->getEventManager()->attach(new ReviewListener());
			$event = new CakeEvent('Model.Review.submitted', $this, array(
					'id' => $this->id,
					'data' => $this->data
				)
			);
			$this->getEventManager()->dispatch($event);

		}

		$this->_updateAverageRating($this->id);

		return parent::afterSave($created, $options);

	}


/**
 * Extended toggleField to ensure that the average rating gets updated.
 * 
 * @param string $field
 * @param integer $id
 * @return boolean
 */
	public function toggleField($field, $id = null) {

		$response = parent::toggleField($field, $id);

		if ($response===true) {
			$id = empty($id) ? $this->id : $id;
			$this->_updateAverageRating($id);
		}

		return $response;

	}


/**
 * Update the average rating for a model item.
 * 
 * @param integer $id model item's ID
 * @return boolean
 */
	protected function _updateAverageRating($id) {

		if (empty($this->data['Review']['model']) || empty($this->data['Review']['model_id'])) {
			$review = $this->findById($id);
		} else {
			$review = array(
				'Review' => array(
					'model' => $this->data['Review']['model'],
					'model_id' => $this->data['Review']['model_id']
				)
			);
		}

		$RatingAverage = ClassRegistry::init('EvReview.RatingAverage');
		$ratingTally = $RatingAverage->find('first', array(
			'conditions' => array(
				'RatingAverage.model_id' => $review['Review']['model_id'],
				'RatingAverage.model' => $review['Review']['model']
			)
		));

		$data = array(
			'RatingAverage' => array(
				'model' => $review['Review']['model'],
				'model_id' => $review['Review']['model_id']				
			)
		);

		$data['RatingAverage']['id'] = !empty($ratingTally) ? $ratingTally['RatingAverage']['id'] : null;

		$data['RatingAverage']['rating'] = $this->find('average', array(
			'conditions' => array(
				'Review.model' => $review['Review']['model'],
				'Review.model_id' => $review['Review']['model_id'],
				'Review.is_active' => true
			)
		));

		$data['RatingAverage']['recommend'] = $this->find('average', array(
			'fields' => array(
				'Review.recommend'
			),
			'conditions' => array(
				'Review.model' => $review['Review']['model'],
				'Review.model_id' => $review['Review']['model_id'],
				'Review.is_active' => true
			)
		));

		$data['RatingAverage']['total_reviews'] = $this->find('count', array(
			'conditions' => array(
				'Review.model' => $review['Review']['model'],
				'Review.model_id' => $review['Review']['model_id'],
				'Review.is_active' => true
			)
		));

		return $RatingAverage->save($data);

	}


/**
 * Returns the average rating of reviews.
 * 
 * @param string $state
 * @param array $query
 * @param array $results
 * @return float
 */
	public function _findAverage($state, $query, $results = []) {

		if ($state === 'before') {

			if (empty($query['fields'])) {
				$query['fields'] = 'Review.rating';
			} elseif (count($query['fields']) > 1) {
				$query['fields'] = array_values($query['fields'])[0];
			}

			return $query;

		}

		$average = 0;

		foreach ($results as $result) {

			$average += array_shift($result['Review']);

		}

		return empty($results) ? 0 : $average/count($results);

	}
	
}