<?php

App::uses('AppBehavior', 'Model/Behavior');
App::uses('ArrayUtil', 'EvCore.Lib');

class RelatableBehavior extends AppBehavior {

	/**
	 * bind the related model to the model
	 *
	 * @param Model $model Model using this behavior
	 */
	public function linkRelatedItem(&$Model) {
		$Model->bindModel(
			array(
				'hasMany' => array(
					'RelatedItem' => array(
						'className' => 'EvRelatedItems.RelatedItem',
						'foreignKey' => 'model_id',
						'conditions' => array(
							'RelatedItem.model' => EvClassRegistry::getNameFromModel($Model)
						)
					)
				)
			),
			false
		);
	}

	/**
	 * bind the relatable models so we can just contain related data
	 *
	 * @param Model $model Model using this behavior
	 */
	public function linkRelatableItems(&$Model) {
		$relatableOptions = Configure::read('EvRelatedItems.RelatableOptions');
		$RelatedItems = EvClassRegistry::init('EvRelatedItems.RelatedItem');

		foreach ($relatableOptions as $key => $modelName) {
			$Model->bindModel(
				array(
					'hasAndBelongsToMany' => array(
						'Related' . $key => array(
							'className' => $modelName,
							'joinTable' => 'ev_related_items_related_item',
							'with' => 'RelatedItem',
							'foreignKey' => 'model_id',
							'associationForeignKey' => 'related_model_id',
							'conditions' => array(
								'RelatedItem.model' => EvClassRegistry::getNameFromModel($Model),
								'RelatedItem.related_model' => $modelName
							)
						)
					)
				),
				false
			);
		}
	}

	/**
	 * process any options and validation into json for storage
	 *
	 * @param Model $model Model using this behavior
	 * @param array $options Options passed from Model::save().
	 * @return mixed False if the operation should abort. Any other result will continue.
	 * @see Model::save()
	 */
	public function beforeBeforeSave(&$Model, $data, $options = array()) {
		$this->linkRelatedItem($Model);

		if (! empty($data['RelatedItem'])) {
			$modelAlias = EvClassRegistry::getNameFromModel($Model);

			foreach ($data['RelatedItem'] as $key => $item) {
				$data['RelatedItem'][$key]['model'] = $modelAlias;
			}
		}

		return $data;
	}

	/**
	 * Save the discounts
	 *
	 * @param Model $model Model using this behavior
	 * @param bool $created True if this save created a new record
	 * @param array $options Options passed from Model::save().
	 * @return bool
	 * @see Model::save()
	 */
	public function afterSave(Model $Model, $created, $options = array()) {
		$return = parent::afterSave($Model, $created, $options);
		$result = true;

		$RelatedItem = EvClassRegistry::init('EvRelatedItems.RelatedItem');

		// process any custom field deletions
		if (! empty($Model->data[$Model->alias]['RelatedItems']['toDelete'])) {

			$ids = explode(',', $Model->data[$Model->alias]['RelatedItems']['toDelete']);
			$ids = ArrayUtil::clearArray($ids);

			$RelatedItem->deleteAll(
				array(
					'RelatedItem.id' => $ids
				),
				true,
				true
			);
		}

		return $return;
	}

	/**
	 * beforeFind can be used to cancel find operations, or modify the query that will be executed.
	 * By returning null/false you can abort a find. By returning an array you can modify/replace the query
	 * that is going to be run.
	 *
	 * @param Model $model Model using this behavior
	 * @param array $query Data used to execute this query, i.e. conditions, order, etc.
	 * @return bool|array False or null will abort the operation. You can return an array to replace the
	 *   $query that will be eventually run.
	 */
	public function beforeFind(Model $Model, $query) {
		if (isset($query['readForView']) && $query['readForView'] === true) {
			$this->linkRelatableItems($Model);
			return $query;
		}

		if (isset($query['readForEdit']) && $query['readForEdit'] === true) {
			$this->linkRelatedItem($Model);
			$query['contain'][] = 'RelatedItem';
		}

		return $query;
	}

	/**
	 * After find callback. Can be used to modify any results returned by find.
	 *
	 * @param Model $model Model using this behavior
	 * @param mixed $results The results of the find operation
	 * @param bool $primary Whether this model is being queried directly (vs. being queried as an association)
	 * @return mixed An array value will replace the value of $results - any other value will be ignored.
	 */
	public function afterFind(Model $Model, $results, $primary = false) {
		if (! Hash::check($results, '{n}.RelatedItem')) {
			return $results;
		}

		$toFind = array();

		foreach ($results as $key => $item) {
			foreach ($item['RelatedItem'] as $relatedKey => $related) {
				$toFind[$related['related_model']][$relatedKey] = $related['related_model_id'];
			}
		}

		if (! empty($toFind)) {
			foreach ($toFind as $modelName => $modelIds) {
				$RelatedModel = EvClassRegistry::init($modelName);

				$relatedData = $RelatedModel->find(
					'list',
					array(
						'conditions' => array(
							$RelatedModel->alias . '.id' => $modelIds
						),
						'callbacks' => false
					)
				);

				if (! empty($relatedData)) {
					foreach ($results as $key => $item) {
						foreach ($item['RelatedItem'] as $relatedKey => $related) {
							if ($related['related_model'] === $modelName && isset($relatedData[$related['related_model_id']])) {
								$results[$key]['RelatedItem'][$relatedKey]['options'] = array(
									$related['related_model_id'] => $relatedData[$related['related_model_id']]
								);
							}
						}
					}
				}
			}
		}

		return $results;
	}
}
