<?php

App::uses('EvPropertyAppModel', 'EvProperty.Model');

class ResidentialProperty extends EvPropertyAppModel {

	const DEPARTMENT_SALES = 'Sales',
		DEPARTMENT_LETTINGS = 'Lettings';

	public $displayField = 'service_id';

	public $actsAs = [
		'MetaData.Meta',
		'Routable.Routable' => array(
			'config' => 'EvProperty'
		),
	];

	public $belongsTo = [
		'Branch' => [
			'className' => 'EvProperty.Branch',
		],
		'PropertyType' => [
			'className' => 'EvProperty.PropertyType',
		],
		'PropertyStyle' => [
			'className' => 'EvProperty.PropertyStyle',
		],
		'PropertyAge' => [
			'className' => 'EvProperty.PropertyAge',
		],
		'PropertyAvailability' => [
			'className' => 'EvProperty.PropertyAvailability',
		],
		'PriceQualifier' => [
			'className' => 'EvProperty.PriceQualifier',
		],
		'PropertyTenure' => [
			'className' => 'EvProperty.PropertyTenure',
		],
		'SaleType' => [
			'className' => 'EvProperty.SaleType',
		],
		'RentFrequency' => [
			'className' => 'EvProperty.RentFrequency',
		],
		'Region' => [
			'className' => 'EvProperty.Region',
		],
	];

	public $hasMany = [
		'PropertyFeature' => [
			'className' => 'EvProperty.PropertyFeature',
			'dependent' => true,
		],
		'PropertyFlag' => [
			'className' => 'EvProperty.PropertyFlag',
			'dependent' => true,
		],
		'PropertyLink' => [
			'className' => 'EvProperty.PropertyLink',
			'dependent' => true,
		],
	];

	public $imageSlots = [
		'general' => [
			'slots' => -1,
			'fields' => false,
		],
		'floorplan' => [
			'slots' => -1,
			'fields' => false,
		],
		'epcGraph' => [
			'slots' => -1,
			'fields' => false,
		],
		'epcFrontPage' => [
			'slots' => -1,
			'fields' => false,
		],
	];

	public $documentSlots = [
		'Brochure' => [
			'slots' => -1,
			'fields' => false
		]
	];

/**
 * Get an array of default parameters for a sales listing
 * @return array Parameters to query with
 */
	public function readForSaleParams() {
		$forSaleParams = $this->_readForListingParams(self::DEPARTMENT_SALES);

		$forSaleParams['order'] = 'ResidentialProperty.price DESC';

		return $forSaleParams;
	}
/**
 * Get an array of default parameters for a lettings listing
 * @return array Parameters to query with
 */
	public function readToLetParams() {
		$toLetParams = $this->_readForListingParams(self::DEPARTMENT_LETTINGS);

		$toLetParams['contain'][] = 'RentFrequency';
		$toLetParams['order'] = 'ResidentialProperty.rent DESC';

		return $toLetParams;
	}

	public function readForView($id, $query = []) {
		$propertyReadForViewDefaultQuery = [
			'fields' => [
				'*'
			],
			'joins' => [
				[
					'table' => 'ev_address_book_addresses',
					'alias' => 'Address',
					'type' => 'left',
					'conditions' => [
						'Address.model' => $this->alias,
						'Address.model_id = ' . $this->alias . '.id'
					]
				]
			],
			'contain' => [
				'Branch',
				'PropertyType',
				'PropertyStyle',
				'PropertyAge',
				'PropertyAvailability',
				'PriceQualifier',
				'PropertyTenure',
				'SaleType',
				'RentFrequency',
				'Region',
				'PropertyFeature',
				'PropertyFlag',
				'PropertyLink'
			],
		];

		if (!empty($query['joins'])) {
			$queryJoins = $query['joins'];
			unset($query['joins']);
		}

		$query = Hash::merge($propertyReadForViewDefaultQuery, $query);

		if (!empty($queryJoins)) {
			foreach ($queryJoins as $join) {
				$query['joins'][] = $join;
			}
		}

		return parent::readForView($id, $query);
	}

/**
 * Get the filters for a listing page. Based on the department so sales and lettings or others can have custom filters.
 * @param  string $department    The department the filters are for
 * @param  string $filterConfig  The filter config to use for these filters.
 * @param  array  $customFilters An array of filters that are specifically for these filters. Merged after filters are created.
 * @return array                 The filters to use for the current listing
 */
	public function getFilters($department, $filterConfig = null, $customFilters = []) {
		$filters = [];

		$filters['location'] = [
			'label' => 'Location',
			'type' => 'string',
		];

		$filters['property_type_id'] = [
			'label' => 'Property Type',
			'type' => 'select',
			'options' => $this->PropertyType->getTypesForDropdown($department),
		];

		if ($department == self::DEPARTMENT_SALES) {
			$filters['price'] = [
				'label' => 'Price',
				'type' => 'min-max',
				'minAttributes' => [
					'label' => 'Price',
					'type' => 'select',
					'options' => $this->_readMinMaxValues('price', true, 'currency'),
				],
				'maxAttributes' => [
					'label' => false,
					'type' => 'select',
					'options' => $this->_readMinMaxValues('price', false, 'currency'),
				],
			];
		} else {
			$filters['rent'] = [
				'label' => 'Rent',
				'type' => 'min-max',
				'minAttributes' => [
					'label' => 'Rent',
					'type' => 'select',
					'options' => $this->_readMinMaxValues('rent', true, 'currency'),
				],
				'maxAttributes' => [
					'label' => false,
					'type' => 'select',
					'options' => $this->_readMinMaxValues('rent', false, 'currency'),
				],
			];
		}

		$filters['bedrooms'] = [
			'label' => 'Bedrooms',
			'type' => 'min-max',
			'minAttributes' => [
				'label' => 'Bedrooms',
				'type' => 'select',
				'options' => $this->_readMinMaxValues('bedroom'),
			],
			'maxAttributes' => [
				'label' => false,
				'type' => 'select',
				'options' => $this->_readMinMaxValues('bedroom', false),
			],
		];

		$filters = Hash::merge($filters, $customFilters);

		if (!empty($filterConfig)) {
			$filterConfig = Configure::read('EvProperty.filters.' . $filterConfig);

			if (!empty($filterConfig)) {
				$filters = Hash::merge($filters, $filterConfig);
			}
		}

		if ($department == self::DEPARTMENT_SALES && isset($filters['rent'])) {
			unset($filters['rent']);
		} elseif (isset($filters['price'])) {
			unset($filters['price']);
		}

		return $filters;
	}

/**
 * Parameters for generic listing queries. Contains the basic parameters needed for a listing.
 * @param  string $department Add the department parameter on.
 * @return array              An array of parameters for a listing.
 */
	protected function _readForListingParams($department = null) {
		$params = [
			'fields' => [
				'*'
			],
			'joins' => [
				[
					'table' => 'ev_address_book_addresses',
					'alias' => 'Address',
					'type' => 'left',
					'conditions' => [
						'Address.model' => $this->alias,
						'Address.model_id = ResidentialProperty.id'
					]
				]
			],
			'conditions' => [
				'ResidentialProperty.is_active' => true,
			],
			'contain' => [
				'PropertyType',
				'GeneralImage',
			],
			'group' => 'ResidentialProperty.id',
		];

		if (!empty($department)) {
			$params['conditions']['ResidentialProperty.department'] = $department;
		}

		return $params;
	}

/**
 * Read min max values from the site settings and format them if required. Used as options for dropdowns in filters.
 * @param  string  $field   The field to read min max values for
 * @param  boolean $minimum Whether the current min max values are for a minimum or maximum
 * @param  string  $format  How to format the min max values.
 * @return array            The array of values to be used in the filters.
 */
	protected function _readMinMaxValues($field, $minimum = true, $format = null) {
		$minMaxValues = Configure::read('SiteSetting.ev_property.search_' . $field . '_' . ($minimum ? 'minimums' : 'maximums'));

		$minMaxValues = explode(',', $minMaxValues);

		$values = [];
		foreach ($minMaxValues as $value) {
			switch ($format) {
				case 'currency':
					$values[$value] = CakeNumber::currency($value, 'GBP', ['places' => 0]);
					break;
				default:
					$values[$value] = $value;
					break;
			}
		}

		return $values;
	}

}
