<?php

App::uses('CakeTime', 'Utility');
App::Uses('ArrayUtil', 'EvCore.Lib');

/**
 * Handle clearing out product caches that are affected by a promotion.
 */
class ProductCacheLib {

/**
 * Flushes all product caches that are affected by one or promotions being
 * created, updated or expiring.
 *
 * @param  array $conditions array of conditions to apply to the promotion lookup
 * @return void
 */
	public function flush($conditions = []) {
		if (! CakePlugin::loaded('EvShop')) {
			return;
		}

		$Promotion = EvClassRegistry::init('EvPromotion.Promotion');
		$CodeRestriction = EvClassRegistry::init('EvDiscount.CodeRestriction');
		$CategoriesProduct = EvClassRegistry::init('EvShop.CategoriesProduct');
		$Product = EvClassRegistry::init('EvShop.Product');

		//Find promotions that have expired recently
		$promotions = $Promotion->find('all', [
			'conditions' => $conditions,
			'fields' => ['id']
		]);

		if (empty($promotions)) {
			// No promotions to run through, exit
			return;
		}

		// Found some promotions, extract the ids
		$promotionIds = $this->__extractArrayItems($promotions, 'id');

		// Pull all restriction records - note that we only want restrictions
		// that are for products from EvShop.
		$restrictions = $CodeRestriction->find('all', [
			'conditions' => [
				'discount_model' => 'Promotion',
				'discount_model_id' => $promotionIds,
				'model LIKE' => 'EvShop%'
			]
		]);

		/**
		 * If there were promotion ids that dont have any restrictions then
		 * it means we've got to clear all products. If thats the case then
		 * there's no point doing any more with the restrictions, so we just
		 * clear the entire product cache end exit.
		 *
		 * As the promotion plugin is currently restricted to EvShop, we can
		 * build up lists of the brand, category and product ids to search by.
		 */
		$promotionRestrictions = [];
		$brandIds = [];
		$categoryIds = [];
		$productIds = [];

		if (! empty($restrictions)) {
			foreach ($restrictions as $restriction) {
				$data = $restriction['CodeRestriction'];
				$promotionRestrictions[$data['discount_model_id']][$data['model']] = $data['model_id'];

				if ($data['model'] === 'EvShop.Brand') {
					$brandIds[] = $data['model_id'];
				} elseif ($data['model'] === 'EvShop.Category') {
					$categoryIds[] = $data['model_id'];
				} elseif ($data['model'] === 'EvShop.Product') {
					$productIds[] = $data['model_id'];
				}
			}
		}

		if (count($promotionIds) > count($promotionRestrictions)) {
			// Found more promotions than restrictions, meaning we should
			// just clear everything.
			return $Product->clearCache();
		}

		$conditions = [];

		if (! empty($categoryIds)) {
			$categoryProductIds = [];
			$categoryProducts = $CategoriesProduct->find('all', [
				'conditions' => [
					'category_id' => $categoryIds
				],
				'fields' => ['product_id']
			]);

			// Merge resulting product ids into the main product id array
			$productIds = array_unique(
				array_merge(
					$this->__extractArrayItems($categoryProducts, 'product_id'),
					$productIds
				)
			);
		}

		// If we've got brands we need to do a query on the product table to get
		// all the product ids
		if (! empty($brandIds)) {
			$products = $Product->find('all', [
				'conditions' => [
					'brand_id' => $brandIds
				],
				'fields' => 'id',
				'callbacks' => false
			]);

			if (! empty($products)) {
				$productIds = array_unique(
					array_merge(
						$this->__extractArrayItems($products, 'product_id'),
						$productIds
					)
				);
			}
		}

		// No product ids to clear, exit
		if (empty($productIds)) {
			return;
		}

		foreach ($productIds as $id) {
			$Product->clearCache($id);
		}
	}

	/**
	 * Helper function to extract out the values from sub arrays
	 *
	 * @example http://snippi.com/s/apvy3ct
	 *
	 * @param  array $data The main array of data to extract from
	 * @param  string $itemKey The name of the key to extract (e.g id)
	 * @return array
	 */
	private function __extractArrayItems(array $data, $itemKey) {
		// If we're using a version of EvCore that supports it - use the
		// ArrayUtil version of this method as it'll be the most up to date.
		// A copy is provided here for backward compatability.
		$arrayUtil = new ArrayUtil();

		if (method_exists($arrayUtil, 'extractArrayItems')) {
			return $arrayUtil::extractArrayItems($data, $itemKey);
		}

		$ids = [];

		array_walk_recursive($data, function ($value, $key) use (&$ids, $data, $itemKey) {
			if ($key === $itemKey) {
				$ids[] = $value;
			}
		});

		return $ids;
	}

}