<?php
/**
 * Calculates whether a postcode surcharge is applicable
 */

class PostcodeSurcharge extends AppModel {
	
	public $displayField = 'postcode_area';

	public $default_vat_rate_id = '1'; //Standard RATE 20%
	
	public $validate = array(
		'postcode_area' => array(
			'required' => array(
				'rule' => array('notEmpty'),
				'message' => 'Please enter a postcode area',
			),
		), 
		'surcharge' => array(
			'numeric' => array(
				'rule' => array('numeric'),
				'message' => 'Please enter surcharge (can be 0.00)',
			)
		)
	);

	/**
	 * Works out if there is an applicable surcharge for the given postcode
	 * @param string $postcode
	 * @return array (PostcodeSurchargeModel object)
	 *
	 */
	public function calculateSurcharge($postcode) {

		$parsedPostcode = $this->parseUkPostcode($postcode);

		if(is_null($parsedPostcode)) {
			throw new Exception("Postcode not valid", 1);
		}

		//Looks at all the possible matching surcharges and finds the one that is the most specific
		$surcharges = $this->find('first', array(
				'conditions' => array(
					'postcode_area' => $parsedPostcode['area'],
					'OR' => array(
						'postcode_district_start IS NULL',
						'AND' => array(
							'postcode_district_start <=' => $parsedPostcode['district'], 
							'postcode_district_end >=' => $parsedPostcode['district'], 
						)
					)
				),
				'order' => array(
					'postcode_district_start DESC',
					'postcode_district_start - '.$parsedPostcode['district'].' + '.
					$parsedPostcode['district']. ' - postcode_district_end DESC'
				)
			)
		);

		if(empty($surcharges)) {
			return false;
		}


		$surcharges[$this->alias]['vat'] = 0;

		if($surcharges && $this->default_vat_rate_id !== null) {

			$VatRate = ClassRegistry::init('EvCheckout.VatRate');

			$vatRate = $VatRate->findById($this->default_vat_rate_id);

			if(! $vatRate) {

				return false; 

			}

			//Separate off the VAT if the admin inc vat flag has been set
			$pricesAlreadyIncludeVAT = (bool) Configure::read('EvShop.prices_include_vat');

			if($pricesAlreadyIncludeVAT) {

				//Calculate cost minus VAT
				$surcharges[$this->alias]['surcharge'] = $surcharges[$this->alias]['surcharge'] / (1 + ($vatRate[$VatRate->alias]['rate'] / 100));

			}

			$surcharges[$this->alias]['vat'] = ($vatRate[$VatRate->alias]['rate']/100) * $surcharges[$this->alias]['surcharge'];

		}

		return $surcharges;
	}

	/**
	 * Postcode validator
	 * @param string Postcode
	 * @return boolean True if valid postcode
	 */
	public function isValidPostcode($postcode) {
		return !is_null($this->parseUkPostcode($postcode));
	}

	/**
	 * Parses a given postcode returning an array of parts
	 * @param 	string 		Postcode 
	 * @return 	array|null 	Returns an array of postcode parts on success / Null if failed to parsed 
	 */
	private function parseUkPostcode($postcode) {

		$postcode = strtoupper(ereg_replace("[^A-Za-z0-9]", "", trim($postcode)));

		$length = strlen($postcode);
		
		if($length < 5){
			
			$regex = '/^([A-Z]{1,2})([0-9][0-9A-Z]?)$/';
			
			if (preg_match($regex, $postcode, $part)) { 
				
				$code = array(
			                 'full' 	=> $part[1] . $part[2],
			                 'area'     => $part[1],
			                 'district' => $part[2],
			                 'outer'    => $part[1] . $part[2],
			                 'sector'   => NULL,
			                 'walk'     => NULL,
			                 'inner'    => NULL
			                 );		 
			
			}else{
				
				$code = NULL;
				
			}
			
		}else{
			
			$regex = '/^([A-Z]{1,2})([0-9][0-9A-Z]?)\s*([0-9])([A-Z]{2})$/';
			
			if (preg_match($regex, $postcode, $part)) { 
				
				$code = array(
			                 'full' => $part[1].$part[2].' '.$part[3].$part[4],
			                 'area'     => $part[1],
			                 'district' => $part[2],
			                 'outer'    => $part[1] . $part[2],
			                 'sector'   => $part[3],
			                 'walk'     => $part[4],
			                 'inner'    => $part[3] . $part[4]
			                 );		 
			
			}else{
				
				$code = NULL;
				
			}
		}
	   return $code;                
	}

}