<?php

use PayPalCheckoutSdk\Core\PayPalHttpClient;
use PayPalCheckoutSdk\Core\ProductionEnvironment;
use PayPalCheckoutSdk\Core\SandboxEnvironment;
use PayPalCheckoutSdk\Orders\OrdersCreateRequest;
use PayPalCheckoutSdk\Orders\OrdersGetRequest;

App::uses('PaypalLib', 'EvPaypal.Lib');
App::uses('IdentityGenerateTokenRequest', 'EvPaypal.Requests');
App::uses('GatewayBaseComponent', 'EvTransactions.Controller/Component');
App::uses('GatewayInterface', 'EvTransactions.Lib');

class PaypalCheckoutComponent extends GatewayBaseComponent implements GatewayInterface {

/**
 * @var array The config for the current environment
 */
	protected $_config;

/**
 * @var PayPalHttpClient The API instance
 */
	protected $_api;

/**
 * {@inheritDoc}
 */
	public function setup() {
		$this->_config = PaypalLib::getConfig();

		if ($this->_config['testMode']) {
			$this->_api = new PayPalHttpClient(
				new SandboxEnvironment($this->_config['clientId'], $this->_config['clientSecret'])
			);
		} else {
			$this->_api = new PayPalHttpClient(
				new ProductionEnvironment($this->_config['clientId'], $this->_config['clientSecret'])
			);
		}
	}

/**
 * Sets the client token to the view data
 *
 * @return void
 * @throws Exception
 */
	public function setClientToken() {
		// Not called in the usual gateway flow so need to call setup manually
		$this->setup();

		// Generate a new hash salt for this request if it is not posting a previous request.
		$hashSalt = PaypalLib::getHashSalt(empty($this->_controller->request->data['Paypal']));
		$request = new IdentityGenerateTokenRequest();
		$response = $this->_api->execute($request);

		if (!empty($response->result->client_token)) {
			$this->_controller->set('paypal', [
				'clientToken' => $response->result->client_token,
				'clientId' => $this->_config['clientId'],
				'brandName' => $this->_config['brandName'],
				'hashSalt' => $hashSalt,
			]);
			return;
		}

		throw new Exception('Payment token could not be generated');
	}

/**
 * Creates a paypal order using the REST API and returns the ID
 *
 * @param array $orderData The order data
 * @return mixed The order ID
 * @throws InternalErrorException
 */
	public function createOrder($orderData) {
		$request = new OrdersCreateRequest();
		$request->body = $orderData;
		$response = $this->_api->execute($request);

		if ($response->result->status === 'CREATED') {
			return $response->result->id;
		}

		throw new InternalErrorException('Order could not be created');
	}

/**
 * {@inheritDoc}
 */
	public function setupPayment($transactionId, $return, $model, $amount, $items, $extra = array()) {
		$this->_data = [
			'return' => $return,
			'orderId' => $extra['auth']['orderId'],
			'amount' => $amount,
			'hash' => $extra['auth']['hash']
		];
	}

/**
 * get the payment from paypal and return a successful response if the payment is valid
 *
 * @param int $transactionId The transaction ID
 * @param array $data The data to validate the payment
 * @return array response data
 * @throws Exception
 */
	public function getPayment($transactionId, $data = []) {
		$this->Transaction = EvClassRegistry::init('EvTransactions.Transaction');
		$orderId = $this->_data['orderId'];

		if (! empty($data['orderId'])) {
			$orderId = $data['orderId'];
		}

		try {
			$request = new OrdersGetRequest($orderId);
			$response = $this->_api->execute($request);

			if ($response->result->status === 'COMPLETED' || $response->result->status === 'CREATED') {
				// Check none of the payment methods failed
				foreach ($response->result->purchase_units as $purchaseUnit) {
					foreach ($purchaseUnit->payments->captures as $capture) {
						if ($capture->status !== 'COMPLETED' && $capture->status !== 'CREATED') {
							throw new Exception('Payment method was ' . $capture->status . ' for paypal order ' . $response->result->id);
						}
					}
				}

				$this->Transaction->save([
					'Transaction' => [
						'id' => $transactionId,
						'payment_token' => $response->result->id,
						'status' => 'success',
					],
				]);
				return [
					'status' => 'success',
					'result' => true,
				];
			}
		} catch (Exception $e) {
			$this->Transaction->save([
				'Transaction' => [
					'status' => 'failed',
					'message' => $e->getMessage(),
				],
			]);
			return [
				'status' => 'failed',
				'result' => false,
			];
		}

		$this->Transaction->save([
			'Transaction' => [
				'payment_token' => $response->result->id,
				'status' => 'failed',
				'message' => 'Unrecognised status in response: ' . $response->result->status,
			],
		]);
		return [
			'status' => 'failed',
			'result' => false,
		];
	}

/**
 * {@inheritDoc}
 */
	public function processReturn() {
		// Check the status of the transaction for the result
		$this->Transaction = EvClassRegistry::init('EvTransactions.Transaction');
		$transaction = $this->Transaction->findById($this->_controller->request->query['transaction']);

		return [
			'result' => $transaction['Transaction']['status'] === 'success'
		];
	}

/**
 * Checks that the transaction is valid by comparing the hash against a hash of the provided data
 *
 * @param mixed $total Optional total to check against
 * @param mixed $currency Optional currency to check against
 * @param mixed $hash Optional hash to check against
 * @return bool
 */
	protected function _checkPaymentHash($total = null, $currency = null, $hash = null) {
		if (empty($total) && empty($currency)) {
			// Check the hash matches and nobody is fiddling the amounts
			if (is_array($this->_data['amount'])) {
				$total = $this->_data['amount']['amount'];
				$currency = $this->_data['amount']['currency'];
			} else {
				$config = PaypalLib::getConfig();
				$total = $this->_data['amount'];
				$currency = $config['currency'];
				CakeLog::write('error', 'Loaded from config');
			}
		}

		if (empty($hash)) {
			$hash = $this->_data['hash'];
		}

		// If the hash doesn't match record a failed transaction
		if (! PaypalLib::checkHash(number_format($total, 2), $currency, $hash)) {
			$this->Transaction = EvClassRegistry::init('EvTransactions.Transaction');
			$this->Transaction->save([
				'Transaction' => [
					'payment_token' => $this->_data['orderId'],
					'status' => 'failed',
					'message' => 'Payment hash check failed'
				],
			]);

			return false;
		}

		return true;
	}
}
