<?php

App::uses('PaypalLib', 'EvPaypal.Lib');
App::uses('GatewayBaseComponent', 'EvTransactions.Controller/Component');
App::uses('GatewayInterface', 'EvTransactions.Lib');
App::uses('CakeText', 'Utility');

/**
 * flow and processes used here were taken from
 * @link http://paypal.github.io/PayPal-PHP-SDK/sample/doc/payments/CreatePaymentUsingPayPal.html
 * @link http://paypal.github.io/PayPal-PHP-SDK/sample/doc/payments/ExecutePayment.html
 */
class PaypalComponent extends GatewayBaseComponent implements GatewayInterface {

	/**
	 * loaded API context
	 */
	protected $_apiContext = null;

	/**
	 * object storage
	 */
	protected $_objects = array();

	/**
	 * apiContext - setup oauth connection when requested
	 *
	 * @return 	PayPal\Rest\ApiContext
	 */
	protected function _apiContext() {
		if ($this->_apiContext !== null) {
			return $this->_apiContext;
		}

		$this->_apiContext = PaypalLib::apiContext($this->_config);

		return $this->_apiContext;
	}

	/**
	 * init
	 *
	 * Use the function setup and connection / settings that need setting up in the gateway
	 */
	public function setup() {
		$this->_config = PaypalLib::getConfig();

		$this->_objects['payer'] = new \PayPal\Api\Payer;
		$this->_objects['payer']->setPaymentMethod('paypal');

		$this->_objects['payment'] = new \PayPal\Api\Payment;
		$this->_objects['payment']->setIntent('sale');

		// check for and setup web profile if not set
		$profileId = Configure::read('SiteSetting.ev_paypal.paypal_experience_id');
		if (empty($profileId)) {

			// set the flow config
			$ProfileFlowConfig = new \PayPal\Api\FlowConfig;
			$ProfileFlowConfig->setLandingPageType($this->_config['flowConfig']['landingPageType']);

			if ($this->_config['flowConfig']['bankTxnPendingUrl'] !== null) {
				$ProfileFlowConfig->setBankTxnPendingUrl($this->_config['flowConfig']['bankTxnPendingUrl']);
			}

			// setup the input fields
			$ProfileInputFields = new \PayPal\Api\InputFields;
			$ProfileInputFields->setAllowNote($this->_config['inputFields']['allowNote']);
			$ProfileInputFields->setNoShipping($this->_config['inputFields']['noShipping']);
			$ProfileInputFields->setAddressOverride($this->_config['inputFields']['addressOverride']);

			// setup the presentation
			$ProfilePresentation = new \PayPal\Api\Presentation;
			$ProfilePresentation->setBrandName($this->_config['brandName']);
			$ProfilePresentation->setLocaleCode($this->_config['presentation']['localeCode']);

			if (! empty($this->_config['imageUrl'])) {
				$ProfilePresentation->setLogoImage($this->_config['imageUrl']);
			}

			$PaypalProfile = new \PayPal\Api\WebProfile;

			// Note: the timestamp on the end of the web profile is to prevent clashing with old
			// profiles should you ever re-issue the profile being used due to any config changes.
			$PaypalProfile->setName($this->_config['brandName'] . ' Web Profile - ' . time());
			$PaypalProfile->setFlowConfig($ProfileFlowConfig);
			$PaypalProfile->setInputFields($ProfileInputFields);
			$PaypalProfile->setPresentation($ProfilePresentation);

			try {
				$createProfileResponse = $PaypalProfile->create(
					$this->_apiContext()
				);
			} catch (\Exception $e) {
				return;
			}

			// save the id
			$SiteSetting = EvClassRegistry::init('EvSiteSettings.SiteSetting');
			$SiteSetting->updateAll(
				array(
					'SiteSetting.value' => '"' . $createProfileResponse->getId() . '"'
				),
				array(
					'SiteSetting.array_slug' => 'paypal_experience_id'
				)
			);

			$profileId = $createProfileResponse->getId();
		}

		if (! empty($profileId)) {
			$this->_objects['payment']->setExperienceProfileId($profileId);
		}
	}

	/**
	 * setupPayment
	 *
	 * Use this function setup the actual payment, i.e. setup the basket, the amount to take etc...
	 *
	 * @param int - transaction id we have created
	 * @param array $return - array with 2 keys of 'return', 'cancel'. Containing either a link or router array for redirecting user
	 * @param array $model - array with two keys of 'model' and 'model_id', used to link transactions polymorphically to other model items
	 * @param float|array $amount - amount of monies to take (I GOT YOUR MONIESSSSSS), or array of 'amount' and 'currency' to change currencies (will take default from config)
	 * @param array $items - multidimenisional array break down of the transaction items, required elements are 'description' and 'amount'
	 * @param mixed $extra - variable allowing you to pass ay data needed to the gateway, could be things like addresses that are not tracked by the transactions model
	 */
	public function setupPayment($transactionId, $return, $model, $amount, $items, $extra = array()) {
		// process the maount var to get array of amount / currency

		$amount = $this->processAmount($amount);
		$PaypalAmount = new \PayPal\Api\Amount;
		$PaypalAmount->setCurrency($amount['currency'])
			->setTotal($amount['amount']);

		// Round discounts to 2 d.p.
		$extra['totals']['discount'] = number_format($extra['totals']['discount'], 2, '.', '');

		// check for extra totals
		if (! empty($extra['totals'])) {
			$PaypalDetails = new \PayPal\Api\Details;

			if (! empty($extra['totals']['delivery'])) {
				$PaypalDetails->setShipping($extra['totals']['delivery']);
			}

			if (! empty($extra['totals']['tax'])) {
				$PaypalDetails->setTax($extra['totals']['tax']);
			}

			if (! empty($extra['totals']['subtotal'])) {
				$subtotal = $extra['totals']['subtotal'];

				if (! empty($extra['totals']['discount'])) {
					$subtotal += $extra['totals']['discount'];
				}

				$PaypalDetails->setSubtotal($subtotal);
			}

			$PaypalAmount->setDetails($PaypalDetails);
		}

		// setup description
		$description = 'Payment to ' . $this->_config['brandName'];
		// Override generic description with custom one if exists
		if (isset($extra['description'])) {
			$description = $extra['description'];
		}

		// loop transaction items
		$PaypalItemList = new \PayPal\Api\ItemList;

		if (! empty($items)) {
			foreach ($items as $item) {
				// check it has all the required fields
				if (isset($item['quantity']) && isset($item['unitPrice'])) {
					$PaypalItem = new \PayPal\Api\Item;
					$PaypalItem->setName(
						CakeText::truncate($item['description'], 124)
					)
					->setQuantity($item['quantity'])
					->setPrice($item['unitPrice'])
					->setCurrency($amount['currency']);

					if (! empty($item['taxAmount'])) {
						$PaypalItem->setTax($item['taxAmount']);
					}

					$PaypalItemList->addItem($PaypalItem);
				}
			}
		}

		if (! empty($extra['totals']['discount'])) {
			$PaypalItem = new \PayPal\Api\Item;
			$PaypalItem->setName(
				CakeText::truncate(Configure::read('EvBasket.labels.discount'), 124)
			)
			->setQuantity(1)
			->setPrice($extra['totals']['discount'])
			->setCurrency($amount['currency']);

			$PaypalItemList->addItem($PaypalItem);
		}

		// setup delivery address
		if (! empty($extra['delivery'])) {
			if (! empty($extra['user'])) {
				$PaypalItemList->setShippingAddress(
					$this->_setupShippingAddress(
						$extra['delivery'],
						$extra['user']
					)
				);
			}
		}

		// Setup the transaction
		$PaypalTransaction = new \PayPal\Api\Transaction;
		$PaypalTransaction->setAmount($PaypalAmount)
			->setDescription($description)
			->setInvoiceNumber(CakeText::uuid())
			->setItemList($PaypalItemList);

		// setup redirect urls
		$PaypalUrls = new \PayPal\Api\RedirectUrls;

		if (! empty($return['return'])) {
			$PaypalUrls->setReturnUrl($return['return'] . '?transaction=' . $transactionId);
		}

		if (! empty($return['cancel'])) {
			$PaypalUrls->setCancelUrl($return['cancel'] . '?transaction=' . $transactionId);
		}

		// set the transaction and redirect urls to the payment object so it's ready to create
		$this->_objects['payment']->setRedirectUrls($PaypalUrls)
			->setTransactions(array($PaypalTransaction))
			->setPayer($this->_objects['payer']);
	}

	/**
	 * getPayment
	 *
	 * Everything should be setup, actually take the payment
	 *
	 * @param int transactions id
	 * @return mixed dependent on the gateway, value is return straight from the transaction component to user anyway
	 */
	public function getPayment($transactionId) {
		// try to create the payment, this will give us a approval url
		try {
			$this->_objects['payment']->create(
				$this->_apiContext()
			);

			// set the payment id as the payment token in the transaction row
			$this->setPaymentToken($transactionId, $this->_objects['payment']->getId());

			// redirect to the approval link
			return $this->_controller->redirect(
				$this->_objects['payment']->getApprovalLink()
			);

		} catch (\PayPal\Exception\PayPalConnectionException $pce) {
			$errors = $this->_parseApiError($pce->getData());

			$this->failTransaction($transactionId, 'Failed to get paypal approval link: ' . $errors);

		} catch (Exception $e) {
			// fail set error message and gegt cancel url and redirect
			$this->failTransaction($transactionId, 'Failed to get paypal approval link: ' . $e->getmessage());
		}

		$this->_controller->Flash->fail(
			array(
				'title' => 'Payment Failed',
				'description' => 'There was a problem trying to setup payment (Transaction #' . $transactionId . '), please try again or contact us to complete your order'
			)
		);

		return $this->_controller->redirect(
			$this->_objects['payment']->getRedirectUrls()->getCancelUrl()
		);
	}

	/**
	 * return
	 *
	 * deal with a return from the gateway and check for success / fail
	 *
	 * @return array - with three elements,
	 *               - 'result' = true/false value
	 *               - 'message' = text message about transaction (i.e. reason for failing)
	 *               - 'transaction_id' = int of the transaction row
	 */
	public function processReturn() {
		$hasTransaction = ! empty($this->_controller->request->query['transaction']);
		$hasPayment = ! empty($this->_controller->request->query['paymentId']);
		$hasPayer = ! empty($this->_controller->request->query['PayerID']);

		if ($hasTransaction && $hasPayment && $hasPayer) {
			// we have all the data needed to process - first check the transaction id and payment id match
			$Transaction = EvClassRegistry::init('EvTransactions.Transaction');

			$transactionId = $this->_controller->request->query['transaction'];
			$paymentId = $this->_controller->request->query['paymentId'];
			$payerId = $this->_controller->request->query['PayerID'];

			if ($Transaction->validTransaction($transactionId, $paymentId)) {
				// all match - so far so good.
				// check with paypal

				$PaypalPayment = \PayPal\Api\Payment::get($paymentId, $this->_apiContext());
				$PaypalExecution = new \PayPal\Api\PaymentExecution;
				$PaypalExecution->setPayerId($payerId);

				try {
					// execute with paypal
					$PaypalPayment->execute(
						$PaypalExecution,
						$this->_apiContext()
					);

					$return = array(
						'result' => true,
						'transaction_id' => $transactionId
					);
				} catch (\PayPal\Exception\PayPalConnectionException $pce) {
					$errors = $this->_parseApiError($pce->getData());

					$return = array(
						'result' => false,
						'message' => 'There was a problem taking payment: ' . $errors,
						'transaction_id' => $transactionId
					);
				} catch(\Exception $e) {
					$return = array(
						'result' => false,
						'message' => 'There was a problem taking payment: ' . $e->getMessage(),
						'transaction_id' => $transactionId
					);
				}
			} else {
				// transaction ID / payment token don't match. Something fishy going on
				$return = array(
					'result' => false,
					'message' => 'Data mis-match. The transaction ID and payment token supplied do not match.',
					'transaction_id' => $transactionId
				);
			}
		} else {
			// at least one bit of the data is missing we need to process the order
			$return = array(
				'result' => false,
				'message' => 'The transaction failed due to missing data'
			);

			if ($hasTransaction) {
				$return['transaction_id'] = $this->_controller->request->query['transaction'];
			}
		}

		return $return;
	}

	/**
	 * setup a new address
	 *
	 * @param 	array 	$address 	The address array
	 * @param 	array 	$user 		The User details
	 * @return 	\Paypal\Api\Address
	 */
	protected function _setupShippingAddress($address, $user) {
		$PaypalAddress = new \PayPal\Api\ShippingAddress;

		$PaypalAddress->setRecipientName($user['User']['name']);
		$PaypalAddress->setLine1($address['Address']['address1']);
		$PaypalAddress->setCity($address['Address']['city']);
		$PaypalAddress->setState($address['Address']['county']);
		$PaypalAddress->setPostalCode($address['Address']['post_code']);
		$PaypalAddress->setCountryCode($address['Country']['iso2']);

		if (! empty($address['Address']['address2'])) {
			$PaypalAddress->setLine2($address['Address']['address2']);
		}

		return $PaypalAddress;
	}

	/**
	 * process paypal api errors (taken from the sample app)
	 *
	 * @param string $errorJson
	 * @return string
	 */
	protected function _parseApiError($errorJson) {
		$msg = '';

		$data = json_decode($errorJson, true);
		if (isset($data['name']) && isset($data['message'])) {
			$msg .= $data['name'] . " : " . $data['message'] . " :: ";
		}

		if (isset($data['details'])) {
			foreach ($data['details'] as $detail) {
				$msg .= $detail['field'] . " : " . $detail['issue'] . " // ";
			}
		}

		if ($msg == '') {
			$msg = $errorJson;
		}
		return $msg;
	}
}
