<?php

App::uses('EvNewsletterSignupsAppModel', 'EvNewsletterSignups.Model');

class Signup extends EvNewsletterSignupsAppModel {

	public $displayField = 'email';

/**
 * validation rules
 */
	public $validate = array(
		'email' => array(
			'required' => array(
				'rule' => 'notBlank',
				'message' => 'Email cannot be left blank'
			),
			'email' => array(
				'rule' => 'email',
				'message' => 'You must enter a valid email address'
			),
			'unique' => array(
				'rule' => 'isUnique',
				'message' => 'A user with that email address has already signed up for the newsletter'
			)
		)
	);

/**
 * afterFind callback
 * decode the extras if set
 *
 */
	public function afterFind($results, $primary = false) {
		$results = parent::afterFind($results, $primary);

		foreach ($results as $key => $item) {
			if (! empty($item['Signup']['extras'])) {
				$results[$key]['Signup']['extras'] = $this->_decode($item['Signup']['extras']);
			}
		}

		return $results;
	}

	public function beforeSave($options = []) {
		$data = $this->data;
		// check for extras
		$extraFields = Configure::read('EvNewsletterSignups.extraFields');

		if (!empty($extraFields)) {
			//	Initially, the extra fields had 'extras.' prepended to them in the form however, to allow for validation
			//	on these custom fields, that key has been removed from the form so now that the validation has passed
			//	we'll construct the extras in preparation for being saved to the db.
			foreach ($extraFields as $field => $attrs) {
				if (array_key_exists($field, $data['Signup'])) {
					$data['Signup']['extras'][$field] = $data['Signup'][$field];
					unset($data['Signup'][$field]);
				}
			}

			if (!empty($data['Signup']['extras'])) {
				$data['Signup']['extras'] = is_array($data['Signup']['extras']) ? json_encode($data['Signup']['extras']) : $data['Signup']['extras'];
			}

			$this->data = $data;
		}

		return parent::beforeSave($options);
	}

	/**
	 * This function only exists because of an issue on CTA where the extras were being double encoded. Once in an extended
 * component and then again in this plugin. As of 5/10/18, the plugin now checks if the data needs encoding before
 * encoding it. This function is simply a safety net just in case the extras data has been encoded more times than it
 * needs to be.
 *
 * @param string|array $data The data that should be json decoded.
 *
 * @return string|array
 */
	protected function _decode($data) {
		if (is_array($data)) {
			return $data;
		}

		$data = json_decode($data, true);

		return $this->_decode($data);
	}

/**
 * afterSave callback
 * (optional) send subscriber details to Mailchimp
 * (optional) send subscriber details to CampaignMonitor
 * (optional) send subscriber details to Capsule CRM
 */
	public function afterSave($created, $options = array()) {
		$mailchimpConfig = Configure::read('EvNewsletterSignups.mailchimp');
		$campaignMonitorConfig = Configure::read('EvNewsletterSignups.campaignMonitor');
		$capsuleConfig = Configure::read('EvNewsletterSignups.capsule');

		// Subscribe to Mailchimp list
		if (is_array($mailchimpConfig)) {
			$this->_subscribeWithMailchimp($this->data, $mailchimpConfig);
		}

		// Subscribe to Campaign Monitor list
		if (is_array($campaignMonitorConfig)) {
			$this->_subscribeWithCampaignMonitor($this->data, $campaignMonitorConfig);
		}

		// Subscribe to Capsule CRM
		if (is_array($capsuleConfig)) {
			$this->_subscribeWithCapsule($this->data, $capsuleConfig);
		}
	}

/**
 * Adds a subscriber to a Mailchimp list
 *
 * @param array data - The POST-ed mail data
 * @param array config - The Mailchimp config
 * @return null
 */
	protected function _subscribeWithMailchimp($data, $config) {
		//Select config based on environment
		if (!empty($config['live']) && Configure::read('app.environment') === 'PRODUCTION') {
			$config = $config['live'];
		} elseif (!empty($config['dev'])) {
			$config = $config['dev'];
		}

		//Check we have the necessary config
		if (empty($config['pluginConfig']['apiKey']) || empty($config['pluginConfig']['defaultListId'])) {
			CakeLog::write(
				'error',
				'Attempted to subscribe newsletter signup #' .
				!empty($data['Signup']['id']) ? $data['Signup']['id'] : '' .
				' to mailchimp but apiKey or defaultListId is missing'
			);
			return;
		}

		// Set config options and init MailChimp
		Configure::write('Mailchimp', $config['pluginConfig']);
		$Mailchimp = EvClassRegistry::init('Mailchimp.Mailchimp');
		$MailchimpSubscriber = EvClassRegistry::init('Mailchimp.MailchimpSubscriber');

		$MailchimpSubscriber->settings = array_merge($MailchimpSubscriber->settings, $config['pluginConfig']);

		// Prep Subscriber Data array
		$subscriber = array(
			'email' => $data['Signup']['email']
		);

		// Prep Mailchimp Options array
		$options = array(
			'doubleOptin' => false,
			'updateExisting' => false
		);

		// Parse any extras and add to Subscriber Data array
		if (!empty($data['Signup']['extras'])) {
			$extras = json_decode($data['Signup']['extras'], true);

			$subscriber = array_merge($subscriber, $extras);
		}

		// if emailType is an extra field, pass it as an option rather than as part of the subscriber as this is a field built into mail chimp
		if (!empty($subscriber['emailType'])) {
			$options['emailType'] = $subscriber['emailType'];
			unset($subscriber['emailType']);
		}

		// Call Mailchimp > Subscribe function
		$MailchimpSubscriber->subscribe($subscriber, $options);
	}

/**
 * Adds a subscriber to a Campaign Monitor list
 *
 * @param array data - The POST-ed mail data
 * @param array config - The Campaign Monitor config
 * @return null
 */
	protected function _subscribeWithCampaignMonitor($data, $config) {
		$campaignMonitorDir = ROOT . DS . 'Vendor' . DS . 'campaignmonitor' . DS . 'createsend-php' . DS;
		require_once $campaignMonitorDir . 'csrest_general.php';

		$auth = array('api_key' => $config['pluginConfig']['apiKey']);
		$wrap = new CS_REST_Subscribers($config['pluginConfig']['defaultListId'], $auth);

		// Prep Subscriber Data array
		$subscriber = array(
			'EmailAddress' => $data['Signup']['email'],
			'Resubscribe' => true
		);

		// Parse any extras
		$extras = array();
		if (!empty($data['Signup']['extras'])) {
			$extras = json_decode($data['Signup']['extras'], true);

			foreach ($extras as $key => $value) {
				$subscriber['CustomFields'][] = array(
					'Key' => $key,
					'Value' => $value
				);
			}
		}

		// Include name?
		if (isset($extras[$config['defaultFields']['name']])) {
			$subscriber['Name'] = $extras[$config['defaultFields']['name']];
		}

		$result = $wrap->add($subscriber);
	}

/**
 * Adds a subscriber to Capsule CRM
 * Please note that Capsule refers to such an entry as a "Party"
 * https://developer.capsulecrm.com/v2/operations/Party#createParty
 *
 * @param array data - The POST-ed mail data
 * @param array config - The Capsule config
 * @return null
 */
	protected function _subscribeWithCapsule($data, $config) {
		// Parse any extras
		if (!empty($data['Signup']['extras'])) {
			$extras = json_decode($data['Signup']['extras'], true);
		}

		// Prep Subscriber Data array
		$subscriber = array(
			'party' => array(
				'type' => 'person',
				'emailAddresses' => array(
					array(
						'address' => $data['Signup']['email']
					)
				)
			)
		);

		// Add values from Extras array to payload
		foreach ($extras as $extraKey => $extraValue) {
			if ($extraKey == 'name') {
				// Organise name in to first/last
				$splitName = explode(' ', trim($extraValue));

				if (count($splitName) > 1) {
					$subscriber['party']['lastName'] = array_pop($splitName);
					$subscriber['party']['firstName'] = implode(' ', $splitName);
				} else {
					$subscriber['party']['lastName'] = '';
					$subscriber['party']['firstName'] = array_pop($splitName);
				}
			} elseif ($extraKey == 'organisation') {
				$subscriber['party']['organisation'] = array(
					'name' => $extraValue
				);

				$subscriber['party']['embed'][] = 'organisation';
			} else {
				$subscriber['party'][$extraKey] = $extraValue;
			}
		}

		// Create JSON
		$subscriberJSON = json_encode($subscriber);

		// Build cURL request
		$ch = curl_init();
		curl_setopt($ch, CURLOPT_HTTPHEADER, array(
			"Authorization: Bearer {$config['pluginConfig']['accessToken']}",
			"Content-Type: application/json",
			"Accept: application/json"
		));
		curl_setopt($ch, CURLOPT_POST, 1);
		curl_setopt($ch, CURLOPT_RETURNTRANSFER, 1);
		curl_setopt($ch, CURLOPT_URL, 'https://api.capsulecrm.com/api/v2/parties');
		curl_setopt($ch, CURLOPT_POSTFIELDS, $subscriberJSON);
		$result = curl_exec($ch);

		curl_close($ch);
	}
}


