<?php

App::uses('AppHelper', 'View/Helper');

App::uses('Menu', 'EvNavigation.Model');

/**
 * Navigation Helper
 *
 * To create a formatted menu pass the return from Menu::getMenu($id):-
 *
 * $this->Navigation->menu($menu, $attr = array());
 *
 * You can check the active state of an individual menu item:-
 *
 * $this->Navigation->checkActive($item);
 *
 * @author  Andy Carter
 * @package Navigation
 */
class NavigationHelper extends AppHelper {

/**
 * Helper dependencies
 *
 * @var array
 * @access public
 */
	public $helpers = array(
		'Html',
		'Image',
		'Session',
		'EvCore.Permissions'
	);

	public $settings = array(
		'tag' => 'li',
		'wrapTag' => 'ul',
		'childWrapTag' => 'div',
		'childrenClass' => 'menu-item--has-children',
		'activeClass' => 'menu-item--active',
		'activeTrailClass' => 'menu-item--active-trail',
		'itemClassPrefix' => 'menu-item--',
		'model' => 'Menu',
		'childrenImageClass' => 'menu-item--has-children-image',
		'depth' => null,
		'prepend' => null,
		'append' => null
	);

/**
 * Default Constructor
 *
 * @param View $View The View this helper is being attached to.
 * @param array $settings Configuration settings for the helper.
 * @return void.
 */
	public function __construct(View $View, $settings = array()) {
		parent::__construct($View, $settings);
		$this->settings = array_merge($this->settings, (array)$settings);
		$this->user_permission_level = intval($this->Session->read('Auth.User.UserGroup.level'));
		$this->user_group_id = intval($this->Session->read('Auth.User.UserGroup.id'));
	}

/**
 * Returns a formatted menu
 *
 * @param array $data array Menu data.
 * @param array $attr Menu attributes.
 * @return string Html string of the menu.
 */
	public function menu($data, $attr = array()) {
		// Set the depth of menu to return.
		$attr = array_merge($this->settings, (array)$attr);
		$depth = $attr['depth'];

		// Build up the menu with the active and active_trail flags set.
		$this->buildMenu($data, $depth, $attr);

		// Return the marked up menu.
		return $this->_formatMenu($data, $attr, true);
	}

/**
 * Build up the menu array with the correct active/active-trail flags set.
 *
 * @param array $data  menu data
 * @param int   $depth depth to recursively run the method for reduces by 1 each run (set to null to build
 *                     the entire menu)
 * @param array $attr  settings data merged with passed overwritten parameters
 * @param int   $activeParent	ID of parent of active item
 * @return array
 */
	public function buildMenu(&$data, $depth = null, $attr = array(), &$activeParent = false) {
		foreach ($data as &$item) {

			$hasChildren = isset($item['children']) && !empty($item['children']);

			if ($hasChildren && ($depth === null | $depth > 1)) {

				$this->buildMenu(
						$item['children'], ($depth === null ? null : $depth - 1), $attr, $activeParent
				);
			} else {

				// We've reached the maximum depth of the current menu so
				// get rid of the children.
				unset($item['children']);
			}

			$item[$attr['model']]['active_trail'] = false;
			// Check for active states

			if ($this->checkActive($item, $attr['model'])) {

				$item[$attr['model']]['active'] = true;
				$activeParent = $item[$attr['model']]['parent_id'];
			} elseif ($activeParent === $item[$attr['model']]['id']) {

				$item[$attr['model']]['active_trail'] = true;
				$activeParent = $item[$attr['model']]['parent_id'];
			}
		}

		return $data;
	}

/**
 * Build up the menu recursively.
 *
 * @param array   $data    array containing menu data
 * @param array   $attr
 * @param boolean $primary true on the first call, false thereafter to indicate child menu
 * @return string  formatted menu
 */
	protected function _formatMenu($data, $attr, $primary = false) {
		$return = $this->_buildChildWrapItem($primary, $attr);

		$return .= $this->_buildWrapItem($primary, $attr);

		$allHidden = true;
		foreach ($data as $item) {
			if (!$this->_checkMenuItemHasPermission($item, $attr)) {
				continue;
			}
			$allHidden = false;

			//Check to see if any attributes have been set for the level this menu item belongs to.
			$childAttr = $attr;
			if (isset($item[$attr['model']]['level']) && !empty($attr['levels'][$item[$attr['model']]['level']])) {
				$attr = Hash::merge($attr, $attr['levels'][$item[$attr['model']]['level']]);
			}

			$hasChildren = $this->_checkMenuItemHasChildren($item, $attr);

			$hasChildrenImage = $this->_checkMenuItemHasChildrenImage($item, $attr);

			$children = $hasChildren ? $this->_formatMenu($item['children'], $childAttr, false) : null;

			$classes = $this->_buildClasses($item, $attr, $hasChildren, $hasChildrenImage);

			$return .= $this->_buildMenuItemTag($attr, $classes);

			$return .= $this->_buildMenuItem($item, $attr, $hasChildren, $children);

			$return .= $this->_buildMenuItemTag($attr, $classes, false);

			//Reset attributes to their original values
			$attr = $childAttr;
		}

		$return .= $this->_buildWrapItem($primary, $attr, false);

		$return .= $this->_buildChildWrapItem($primary, $attr, false);

		if ($allHidden) {
			$return = '';
		}

		return $return;
	}

/**
 * Check for active state of a menu item. Used by _formatMenu() and can be
 * called from within a view when wanting to play with the unprocessed
 * menu array.
 *
 * @param array $item 	menu item
 * @param string $menu_model the menu model as defined by settings or as overwritten by user
 * @return boolean      true if active
 */
	public function checkActive($item, $menuModel) {
		if (isset($item[$menuModel]['active']) && $item[$menuModel]['active']) {
			return true;
		} else {

			return $this->_checkActiveByRoute($item, $menuModel);
		}
	}

/**
 * Check the menu item against the current route.
 *
 * @param array $item	menu item
 * @param string $menu_model the menu model as defined by settings or as overwritten by user
 * @return boolean		true if there is a match
 */
	protected function _checkActiveByRoute($item, $menuModel) {
		$isActive = false;

		if (($pos = strrpos($this->here, '/page:')) !== false) {
			$this->here = substr($this->here, 0, $pos);
		}
		// Check if the menu item's URL matches the current page.
		if (!$isActive && $item[$menuModel]['url']) {

			$isActive = Router::normalize($this->here) === Router::normalize($item[$menuModel]['url']);
		}

		// Check if the menu item's pattern matches the current page.
		if (!$isActive && $item[$menuModel]['pattern']) {

			$isActive = preg_match($item[$menuModel]['pattern'], Router::normalize($this->here));
		}

		// Check to see if a this menu item has had a redirect url set
		if (!$isActive && !empty($item['Page']['redirect_url'])) {
			$isActive = Router::normalize($this->here) === Router::normalize($item['Page']['redirect_url']);
		}

		return $isActive;
	}

/**
 * Build the wrapping element that wraps the child menu items.
 *
 * @param bool  $primary True if this is the first time constructing the wrapping element. False otherwise.
 * @param array $attr    The menu attributes.
 * @param bool  $open    True if we are opening the wrapping element. False otherwise.
 * @return string
 */
	protected function _buildChildWrapItem($primary, $attr, $open = true) {
		$return = '';

		if (!$primary && !empty($attr['childWrapTag'])) {
			if ($open) {
				$childWrapClass = '';

				if (! empty($attr['childWrapClass'])) {
					$childWrapClass = ' class="' . $attr['childWrapClass'] . '"';
				}

				$return = "<{$attr['childWrapTag']}{$childWrapClass}>";
			} else {
				$return = "</{$attr['childWrapTag']}>";
			}
		}

		return $return;
	}

/**
 * Build the wrapping element for a menu item. An empty string will be returned if 'wrapTag' has not been set
 * in the menu attributes.
 *
 * @param bool  $primary True if this is the first time constructing the wrapping element. False otherwise.
 * @param array $attr    The menu attributes.
 * @param bool  $open    True if we are opening the wrapping element. False otherwise.
 * @return string Html string for the wrapping element.
 */
	protected function _buildWrapItem($primary, $attr, $open = true) {
		$return = '';

		if (!empty($attr['wrapTag'])) {
			if ($open) {
				$id = '';
				$classes = [];

				if ($primary && isset($attr['class']) && !empty($attr['class'])) {

					$classes[] = $attr['class'];
				}

				if ($primary && isset($attr['id']) && !empty($attr['id'])) {

					$id = " id='{$attr['id']}'";
				}

				$class = implode(' ', $classes);

				if (! $primary && ! empty($attr['wrapClass'])) {
					$class .= $attr['wrapClass'];
				}

				$dataAttr = '';
				if ($primary && isset($attr['data']) && !empty($attr['data'])) {
					foreach ($attr['data'] as $k => $v) {
						$dataAttr .= ' data-' . $k . '="' . $v . '"';
					}
				}

				$return = "<{$attr['wrapTag']}$id class='$class'{$dataAttr}>";
			} else {
				$return = "</{$attr['wrapTag']}>";
			}
		}

		return $return;
	}

/**
 * Check if a menu item can be displayed based on the current user's permission level.
 *
 * @param array $item The menu item to check.
 * @param array $attr The menu attributes.
 * @return bool True if the user has permission to view menu item. False otherwise.
 */
	protected function _checkMenuItemHasPermission($item, $attr) {
		if (!empty(Configure::read('EvNavigation.use_acl_permissions'))) {
			if (empty($item[$attr['model']]['url'])) {
				return true;
			}

			if (empty($item[$attr['model']]['requires_login'])) {
				return true;
			}

			if (empty($this->user_group_id)) {
				return false;
			}

			$menuItemUrl = $item[$attr['model']]['url'];
			$menuUrlHash = md5(json_encode($menuItemUrl));

			return Cache::remember('menu_permissions_' . $menuUrlHash . '_' . $this->user_group_id, function () use ($menuItemUrl) {
				if (is_array($menuItemUrl)) {
					//If the url is an array, then convert it into a string so that the area of site is included.
					//This makes it so that the url is parsed correctly when urls are relative.
					//For example relative admin menus will start /admin.
					$menuItemUrl = Router::url($menuItemUrl);
				}

				$menuItemUrl = Router::parse($menuItemUrl);

				return $this->Permissions->check($menuItemUrl, $this->user_group_id);
			}, 'EvNavigation_Permissions');
		} else {
			//Check if a permission level exists for this menu item and if it does check if the user has the level to view it.
			if (empty($item[$attr['model']]['permission_level']) || $item[$attr['model']]['permission_level'] >= $this->user_permission_level) {
				return true;
			}
		}

		return false;
	}

/**
 * Check if a menu item has any children to display. If all the children are menu hidden then it acts as if
 * no children exist.
 *
 * @param array $item The menu item to check.
 * @param array $attr The menu attributes.
 * @return bool True if the menu item has children to display. False otherwise.
 */
	protected function _checkMenuItemHasChildren($item, $attr) {
		//If the menu item has children, check that they aren't all hidden. If they are then the menu item
		//shouldn't display as having children.
		if (!empty($item['children'])) {
			return count(Hash::extract($item['children'], '{n}.' . $attr['model'] . '[is_menu_hidden=false]')) > 0;
		}

		return false;
	}

/**
 * Check if a menu item has any children with images.
 *
 * @param array $item The menu item to check.
 * @return bool True if any children have menu images. False if not.
 */
	protected function _checkMenuItemHasChildrenImage($item) {
		//If the menu item has children, check if any of them have MenuImages. If they do then the menu item
		//should display as having child images.
		if (!empty($item['children'])) {
			return count(Hash::extract($item['children'], '{n}.MenuImage.{n}.id')) > 0;
		}

		return false;
	}

/**
 * Build an array of classes for a menu item.
 *
 * @param array $item             The menu item that is being built.
 * @param array $attr             The menu attributes.
 * @param bool  $hasChildren      True if the menu item has children. False if not.
 * @param bool  $hasChildrenImage True if any children have images. False if not.
 * @param array $classes          Any current classes. (Defaults to empty array).
 * @return array The built class array.
 */
	protected function _buildClasses($item, $attr, $hasChildren, $hasChildrenImage, $classes = []) {
		if (isset($item[$attr['model']]['id']) && !empty($attr['itemClassPrefix'])) {
			$classes[] = $attr['itemClassPrefix'] . $item[$attr['model']]['id'];
		}

		if (isset($item[$attr['model']]['class']) && !empty($item[$attr['model']]['class'])) {
			$classes[] = $item[$attr['model']]['class'];
		}

		if ($hasChildren && !empty($attr['childrenClass'])) {
			$classes[] = $attr['childrenClass'];
		}

		if ($hasChildrenImage && !empty($attr['childrenImageClass'])) {
			$classes[] = $attr['childrenImageClass'];
		}

		if ($item[$attr['model']]['active'] && !empty($attr['activeClass'])) {
			$classes[] = $attr['activeClass'];
		}

		if ($item[$attr['model']]['active_trail'] && !empty($attr['activeTrailClass'])) {
			$classes[] = $attr['activeTrailClass'];
		}

		return $classes;
	}

/**
 * Construct the element for the menu item. Default helper setting is an 'li' element.
 *
 * @param array $attr    The menu attributes.
 * @param array $classes The classes to add to the current menu item element.
 * @param bool  $open    True if opening the element. False if closing the element.
 * @return string Html string for the menu item element.
 */
	protected function _buildMenuItemTag($attr, $classes, $open = true) {
		if ($open) {
			$class = implode(' ', $classes);
			$return = "<{$attr['tag']} class='$class'>";
		} else {
			$return = "</{$attr['tag']}>";
		}

		return $return;
	}

/**
 * Build the html string for a menu item. If the menu item is to use a content element then attempt to use
 * that otherwise construct a link.
 *
 * @param array  $item        The menu item.
 * @param array  $attr        The menu attributes.
 * @param bool   $hasChildren True if children exist for this menu item. Otherwise false.
 * @param string $children    The html string for the children to include.
 * @return string A html string for the menu item.
 */
	protected function _buildMenuItem($item, $attr, $hasChildren, $children) {
		if (
			!empty($item[$attr['model']]['content_element']) &&
			$this->_View->elementExists($item[$attr['model']]['content_element'])
		) {
			$return = $this->_buildContent($item, $attr);
		} else {
			$return = $this->_buildLink($item, $attr, $hasChildren, $children);
		}

		return $return;
	}

	/**
	 * build a link menu item
	 *
	 * @param 	array 	menu item from database we are building link for
	 * @param  	array   $attr
	 * @param 	bool 	whether the item has children
	 * @param 	string 	the formatted child elements
	 * @return 	string
	 */
	protected function _buildLink($item, $attr, $hasChildren, $children) {
		$return = '';

		$addImage = '';
		if (isset($item['MenuImage'][0])) {
			$addImage = $this->Image->resize($item['MenuImage'][0], array('width' => 100, 'height' => 100, 'crop' => true));
		}

		$name = $item[$attr['model']]['name'];

		if (isset($attr['prepend']) && ! empty($attr['prepend'])) {
			$name = $attr['prepend'] . $name;
		}

		if (isset($attr['append']) && ! empty($attr['append'])) {
			$name = $name . $attr['append'];
		}

		$linkAttr = array(
			'escape' => false,
			'target' => (isset($item[$attr['model']]['new_window']) && $item[$attr['model']]['new_window'] ? '_blank' : null)
		);

		if (isset($hasChildren) && $hasChildren === true && ! empty($attr['childrenLinkAttr'])) {
			$linkAttr = array_merge($linkAttr, $attr['childrenLinkAttr']);
		}

		if (empty($item[$attr['model']]['url']) && Configure::read('EvNavigation.no_url_use_spans')) {
			if ($hasChildren) {
				$linkAttr['role'] = 'button';
			}
			$return .= $this->Html->tag('span', $addImage . $name, $linkAttr);
		} else {
			$return .= $this->Html->link($addImage . $name,	$item[$attr['model']]['url'], $linkAttr);
		}

		if ($hasChildren) {
			$return .= $children;
		}

		return $return;
	}

	/**
	 * build a content element li rather then a link
	 *
	 * @param 	array 	menu item from database we are building link for
	 * @param  array   $attr
	 * @return 	string
	 */
	protected function _buildContent($item, $attr) {
		return $this->_View->element(
			$item[$attr['model']]['content_element'],
			array(
				'item' => $item,
				'attr' => $attr
			)
		);
	}
}
